package com.treweren.save;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;

import org.knime.bio.types.PdbValue;
import org.knime.chem.types.SdfValue;
import org.knime.chem.types.SmilesValue;
import org.knime.core.data.DataCell;
import org.knime.core.data.DataColumnSpec;
import org.knime.core.data.DataColumnSpecCreator;
import org.knime.core.data.DataRow;
import org.knime.core.data.DataTableSpec;
import org.knime.core.data.RowKey;
import org.knime.core.data.def.DefaultRow;
import org.knime.core.data.def.DoubleCell;
import org.knime.core.data.def.IntCell;
import org.knime.core.data.def.StringCell;
import org.knime.core.node.BufferedDataContainer;
import org.knime.core.node.BufferedDataTable;
import org.knime.core.node.CanceledExecutionException;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelString;
import org.knime.core.node.ExecutionContext;
import org.knime.core.node.ExecutionMonitor;
import org.knime.core.node.InvalidSettingsException;
import org.knime.core.node.NodeLogger;
import org.knime.core.node.NodeModel;
import org.knime.core.node.NodeSettingsRO;
import org.knime.core.node.NodeSettingsWO;
import org.knime.core.util.MutableBoolean;




/**
 * This is the model implementation of Save.
 * Saves molecular data to disk file
 *
 * @author Treweren Consultants
 */
public class SaveNodeModel extends NodeModel {
    
    // the logger instance
    private static final NodeLogger logger = NodeLogger
            .getLogger(SaveNodeModel.class);
    private static int n_Column = 0;
    private static String m_sdfColumn = null;
    private static String m_pdbColumn = null;
    private static String m_smilesColumn = null;
    private static String m_stringColumn = null;
   private static final int CANCEL_CHECK_INTERVAL = 1000;
    
    /** the settings key which is used to retrieve and 
        store the settings (from the dialog or from a settings file)    
       (package visibility to be usable from the dialog). */
	static final String CFGKEY_SAVE = "Save";
	static final String CFGKEY_SMILES = "Smiles";
	static final String CFGKEY_HYDROGENS = "Hydrogens";
	static final String CFGKEY_WORKING = "Working";
	
	/** initial default count value. */
	static final String DEFAULT_SAVE = "";
	static final boolean DEFAULT_HYDROGENS = true; 
	static final Boolean DEFAULT_SMILES = false;
	static final String DEFAULT_WORKING = System.getenv("THINK_WORKING");

	// example value: the models count variable filled from the dialog 
	// and used in the models execution method. The default components of the
	// dialog work with "SettingsModels".
	
	private final SettingsModelString m_save =
		new SettingsModelString(SaveNodeModel.CFGKEY_SAVE,
                    SaveNodeModel.DEFAULT_SAVE );

	private final SettingsModelBoolean m_hydrogens =
		new SettingsModelBoolean(SaveNodeModel.CFGKEY_HYDROGENS,
                    SaveNodeModel.DEFAULT_HYDROGENS );
	
	private final SettingsModelBoolean m_smiles =
		new SettingsModelBoolean(SaveNodeModel.CFGKEY_SMILES,
                    SaveNodeModel.DEFAULT_SMILES );
	
	private final SettingsModelString m_working =
		new SettingsModelString(SaveNodeModel.CFGKEY_WORKING, SaveNodeModel.DEFAULT_WORKING );
    /**
     * Constructor for the node model.
     */
    protected SaveNodeModel() {
    
        // one incoming port and no outgoing port
        super(1, 0);
    }

    /**
     * @see org.knime.core.node.NodeModel #execute(BufferedDataTable[],
     *      ExecutionContext)
     */
    @Override
    protected BufferedDataTable[] execute(final BufferedDataTable[] inData,
            final ExecutionContext exec) throws Exception {


        notifyViews(null);

        final String cwdString = m_working.getStringValue();
        // clean up 
        File outputReport = new File(cwdString, "output.log"); 
        outputReport.delete();
         File progressFile = new File(cwdString, "progress0.dat"); 
        progressFile.delete();
        BufferedDataTable in = inData[0];
        
        if ( n_Column == 0 ) {
        	DataTableSpec InSpec = in.getDataTableSpec();
        	n_Column = columnSelect(InSpec);
        }
      
        // write out input file
        String fileString = null; 
        if ( m_pdbColumn != null ) {
            fileString = "save.pdb"; 
        }
        else if ( m_sdfColumn != null ) {
            fileString = "save.sdf"; 
        }
        else {
            fileString = "save.smiles"; 
        }
        File inFile = new File(cwdString, fileString); 
        BufferedWriter outWriter = new BufferedWriter(new FileWriter(inFile));
        int colIndex;
        if ( m_pdbColumn != null ) {
        	colIndex = in.getDataTableSpec().findColumnIndex(m_pdbColumn);
        }
        else if ( m_sdfColumn != null ) {
        	colIndex = in.getDataTableSpec().findColumnIndex(m_sdfColumn);
        }
        else if ( m_smilesColumn != null ) {
        	colIndex = in.getDataTableSpec().findColumnIndex(m_smilesColumn);        
        }
        else {
            colIndex = in.getDataTableSpec().findColumnIndex(m_stringColumn);        
        }
       	  
        final double count = in.getRowCount(); // floating point operations
        int i = 0; 
        int missingCount = 0;
        for (DataRow r : in) {
            exec.checkCanceled();
            DataCell c = r.getCell(colIndex);
            if (c.isMissing()) {
                missingCount++;
            } 
            else if ( m_pdbColumn != null ) {
                PdbValue v = (PdbValue)c;
                String toString = v.toString();
                outWriter.write(toString);
            }
            else if ( m_sdfColumn != null ) {
                SdfValue v = (SdfValue)c;
                String toString = v.toString();
                outWriter.write(toString);
                if (!toString.trim().endsWith("$$$$")) {
                    outWriter.newLine();
                    outWriter.append("$$$$");
                    outWriter.newLine();
                }
            }
            else if ( m_smilesColumn != null ) {
                SmilesValue v = (SmilesValue)c;
                String toString = v.toString() + " " + r.getKey();
                outWriter.write(toString);        	
                outWriter.newLine();
            }
            else {
            	String toString = c.toString() + " " + r.getKey();
            	outWriter.write(toString);        	
            	outWriter.newLine();
            }
            i++;
         }

        outWriter.close();
        if (missingCount > 0) {
            setWarningMessage("Skipped " + missingCount 
                    + " row(s) because of missing values");
        }

        // execute THINK
        File cwdFile = new File (cwdString);
        int exitVal;
        try {
            exec.setProgress("Starting THINK");
            Runtime rt = Runtime.getRuntime();
            String s_save = m_save.getStringValue();
            String s_hydrogens = "NOHYDROGENS";
            if ( m_hydrogens.getBooleanValue()) {
            	s_hydrogens = " ";
            }
            // prepare the command string
             String cmdString = System.getenv("THINK_EXEC") + "think THINK_EXEC:save.log output.log " + fileString + " " + " \"" + s_save + "\""  + " " + s_hydrogens;
            // Go!
            logger.info("THINK command line: '" + cmdString + "'");
            final Process proc = rt.exec(cmdString, null, cwdFile);

            final MutableBoolean procDone = new MutableBoolean(false);
            new Thread(new Runnable() {
                public void run() {
                    synchronized (procDone) {
                        while (!procDone.booleanValue()) {
                            try {
                                exec.checkCanceled();
                            } catch (CanceledExecutionException cee) {
                                // blow away the running external process
                                proc.destroy();
                                return;
                            }
                            try {
                               	procDone.wait(CANCEL_CHECK_INTERVAL);
                                // Read progress file and set progress value       
                               	double d= 0.;
                                File progressReport = new File(cwdString, "progress0.dat");
                                if ( progressReport.exists() && progressReport.isFile() ) {
                               		if ( progressReport.length() > 0 ) {
                               			BufferedReader in = new BufferedReader(new FileReader(progressReport));
                               			String line;
                               			if ( (line = in.readLine()) != null) {
                               			  d = Double.valueOf(line).doubleValue();
                               			}               
                              			if ( (line = in.readLine()) == null) {
                              				line = "Running THINK";
                              			}
                               			in.close();
                                        exec.setProgress( d, line.toString());
                               		}
                                }
                            } catch (InterruptedException e) {
                                // do nothing
                            } catch ( FileNotFoundException e) {
                                // do nothing
                            } catch ( IOException e) {
                                // do nothing
                            }
                        }
                    }
                }

            }).start();

            // wait until the external process finishes.
            exitVal = proc.waitFor();

            synchronized (procDone) {
                // this should terminate the check cancel thread
                procDone.setValue(true);
            }

            exec.checkCanceled();

            exec.setProgress("Wrapping up");
            logger.info("THINK terminated with exit code: " + exitVal);
        } catch (Throwable t) {
           	warningReport();
            logger.error("THINK failed (with exception)", t);
            throw new Exception(t);
        }

        if (exitVal != 0) {
        	warningReport();
            // before we return barfing, we save the output in the failing list
            throw new IllegalStateException("THINK failed (error code "
                    + exitVal + ")");
        }
  
        return  new BufferedDataTable[0] ;

    }

    private void warningReport( ) throws Exception {

    	StringBuilder warningMessage = new StringBuilder();
        String cwdString = m_working.getStringValue();
        File outputReport = new File(cwdString, "output.log"); 

        if ( outputReport.exists() && outputReport.isFile() ) {
       		if ( outputReport.length() > 0 ) {
       			BufferedReader in = new BufferedReader(new FileReader(outputReport));
       			String line;
       			while ((line = in.readLine()) != null) {
       				warningMessage.append( line + "\n");
       			}               
       			in.close();
       		}
       		else { 
                warningMessage.append( "Report from THINK is blank");      	     			
       		}
        }
       	else {
            warningMessage.append( "No report from THINK");      	
       	}	
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
   }
    

    /**
     * @see org.knime.core.node.NodeModel#reset()
     */
    @Override
    protected void reset() {
        // TODO Code executed on reset.
        // Models build during execute are cleared here.
        // Also data handled in load/saveInternals will be erased here.
    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #configure(org.knime.core.data.DataTableSpec[])
     */
    @Override
    protected DataTableSpec[] configure(final DataTableSpec[] inSpecs)
            throws InvalidSettingsException {
        
        // TODO: check if user settings are available, fit to the incoming
        // table structure, and the incoming types are feasible for the node
        // to execute. If the node can execute in its current state return
        // the spec of its output data table(s) (if you can, otherwise an array
        // with null elements), or throw an exception with a useful user message
 
    	if ( System.getenv("THINK_EXEC") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_EXEC is unset - It should be set to the folder containing the THINK software.");       	  		
    	}
        if ( System.getenv("THINK_WORKING") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_WORKING is not set to the folder in which working files will be created.");       	  		
    	}
    	if ( m_save.getStringValue().equals("") ) {
            throw new InvalidSettingsException(
            "File to be saved not specified");       	  		
    	}

    	File saveFile = new File ( m_save.getStringValue() ); 
       	if ((saveFile.exists() | saveFile.isFile())) {
     		throw new InvalidSettingsException("Selected file already exists ('"
                    + saveFile.getAbsolutePath() + "')");
        }

        DataTableSpec in = inSpecs[0];
        n_Column = columnSelect (in);
        return null;
    }
    protected int columnSelect ( final DataTableSpec in ) throws  InvalidSettingsException {

        StringBuilder warningMessage = new StringBuilder();
        int n_column=0;
        int pdbColCount = 0;
        int n_pdb=0;
        int sdfColCount = 0;
        int n_sdf=0;
        int smilesColCount = 0;
        int n_smiles=0;
        int stringColCount = 0;
        int n_string=0;
        for (int i = 0; i < in.getNumColumns(); i++) {
            DataColumnSpec s = in.getColumnSpec(i);
            if (s.getType().isCompatible(PdbValue.class)) {
                if (m_pdbColumn == null) {
                    m_pdbColumn = in.getColumnSpec(i).getName();
                    n_pdb = i;
                }
                pdbColCount++;
            }
            else if (s.getType().isCompatible(SdfValue.class)) {
                if (m_sdfColumn == null) {
                    m_sdfColumn = in.getColumnSpec(i).getName();
                    n_sdf = i;
                }
                sdfColCount++;
            }
            else if ( s.getType().isCompatible(SmilesValue.class)) {
            	if (m_smilesColumn == null ) {
             		m_smilesColumn = in.getColumnSpec(i).getName();
             		n_smiles=i;
              	}
               	smilesColCount++;
            }
            else if ( in.getColumnSpec(i).getName().equalsIgnoreCase("SMILES"))  {
            	if (m_stringColumn == null ) {
             		m_stringColumn = in.getColumnSpec(i).getName();
             		n_string=i;
              	}
               	stringColCount++;
            }
        }
        if ( pdbColCount == 0 & smilesColCount == 0 & sdfColCount == 0 & stringColCount == 0) {
            throw new InvalidSettingsException(
                    "No column with Smiles or SD or PDB file compatible type");       	
        }
        else if ( pdbColCount > 0 ) {
           	m_sdfColumn = null;
           	m_smilesColumn = null;
        	m_stringColumn = null;
        	n_column = n_pdb;
        	if (pdbColCount > 1) {
                warningMessage.append("More than one PDB compatible column in " 
                        + "input, using column \"" + m_pdbColumn + "\".");
        	}
        }
        else if ( ( m_smiles.getBooleanValue() | sdfColCount == 0 ) & smilesColCount > 0 ) {
           	m_sdfColumn = null;
           	m_pdbColumn = null;
        	m_stringColumn = null;
        	n_column = n_smiles;
        	if (smilesColCount > 1) {
                warningMessage.append("More than one Smiles compatible column in " 
                        + "input, using column \"" + m_smilesColumn + "\".");
        	}
        }
        else if ( sdfColCount > 0 ) {
          	m_pdbColumn = null;
           	m_smilesColumn = null;
        	m_stringColumn = null;
        	n_column = n_sdf;
        	if (sdfColCount > 1) {
                warningMessage.append("More than one SDF compatible column in " 
                        + "input, using column \"" + m_sdfColumn + "\".");
        	}
        }
        else if ( stringColCount > 0 ) {
          	m_pdbColumn = null;
        	m_smilesColumn = null;
        	m_sdfColumn = null;
        	n_column = n_string;
           	if (stringColCount > 1) {
                warningMessage.append("More than one Smiles string compatible column in " 
                        + "input, using column \"" + m_stringColumn + "\".");
           	}
        }
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
        return n_column;
    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #saveSettingsTo(org.knime.core.node.NodeSettings)
     */
    @Override
    protected void saveSettingsTo(final NodeSettingsWO settings) {

        // TODO save user settings to the config object.
		
		m_save.saveSettingsTo(settings);
		m_hydrogens.saveSettingsTo(settings);
		m_smiles.saveSettingsTo(settings);
		m_working.saveSettingsTo(settings);

    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #loadValidatedSettingsFrom(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void loadValidatedSettingsFrom(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO load (valid) settings from the config object.
        // It can be safely assumed that the settings are valided by the 
        // method below.
        
        m_save.loadSettingsFrom(settings);
        m_hydrogens.loadSettingsFrom(settings);
        m_smiles.loadSettingsFrom(settings);
        m_working.loadSettingsFrom(settings);

    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #validateSettings(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void validateSettings(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO check if the settings could be applied to our model
        // e.g. if the count is in a certain range (which is ensured by the
        // SettingsModel).
        // Do not actually set any values of any member variables.

        m_save.validateSettings(settings);
        m_hydrogens.validateSettings(settings);
        m_smiles.validateSettings(settings);
        m_working.validateSettings(settings);

    }
    
    /**
     * @see org.knime.core.node.NodeModel #loadInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void loadInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
        
		// TODO load internal data. 
		// Everything handed to output ports is loaded automatically (data
		// returned by the execute method, models loaded in loadModelContent,
		// and user settings set through loadSettingsFrom - is all taken care 
		// of). Load here only the other internals that need to be restored
		// (e.g. data used by the views).

    }
    
    /**
     * @see org.knime.core.node.NodeModel #saveInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void saveInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
       
       	// TODO save internal models. 
		// Everything written to output ports is saved automatically (data
		// returned by the execute method, models saved in the saveModelContent,
		// and user settings saved through saveSettingsTo - is all taken care 
		// of). Save here only the other internals that need to be preserved
		// (e.g. data used by the views).

    }

}
