package com.treweren.display;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;

import org.knime.chem.types.SmilesCell;
import org.knime.core.data.DataCell;
import org.knime.core.data.DataColumnSpec;
import org.knime.core.data.DataColumnSpecCreator;
import org.knime.core.data.DataRow;
import org.knime.core.data.DataTableSpec;
import org.knime.core.data.DataType;
import org.knime.core.data.RowKey;
import org.knime.core.data.def.DefaultRow;
import org.knime.core.data.def.DoubleCell;
import org.knime.core.data.def.IntCell;
import org.knime.core.data.def.StringCell;
import org.knime.core.node.BufferedDataContainer;
import org.knime.core.node.BufferedDataTable;
import org.knime.core.node.CanceledExecutionException;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelDoubleBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelString;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;
import org.knime.core.node.port.PortType;
import org.knime.core.node.ExecutionContext;
import org.knime.core.node.ExecutionMonitor;
import org.knime.core.node.InvalidSettingsException;
import org.knime.core.node.NodeLogger;
import org.knime.core.node.NodeModel;
import org.knime.core.node.NodeSettingsRO;
import org.knime.core.node.NodeSettingsWO;
import org.knime.core.util.MutableBoolean;

import org.knime.bio.types.PdbCell;
import org.knime.bio.types.PdbValue;
import org.knime.chem.types.SdfCell;
import org.knime.chem.types.SdfValue;
import org.knime.chem.types.SmilesValue;
import org.knime.core.data.DataColumnSpec;
import org.knime.core.data.DataRow;
import org.knime.core.data.DataCell;
import org.knime.core.data.DataTableSpec;

/**
 * This is the model implementation of Display.
 * Uses THINK to display molecules
 *
 * @author Treweren Consultants
 */
public class DisplayNodeModel extends NodeModel {
    
	   // the logger instance
    private static final NodeLogger logger = NodeLogger
            .getLogger(DisplayNodeModel.class);
    private static int n_Column =-1;
    private static int n_Query =-1;
    private static String m_sdfColumn = null;
    private static String m_smilesColumn = null;
    private static String m_pdbColumn = null;
    private static String m_stringColumn = null;
    private static final int CANCEL_CHECK_INTERVAL = 1000;
        
    /** the settings key which is used to retrieve and 
        store the settings (from the dialog or from a settings file)    
       (package visibility to be usable from the dialog). */
	static final String CFGKEY_MODE = "Mode";
	static final String CFGKEY_TILE = "Tile";
	static final String CFGKEY_SMILES = "Smiles";
	static final String CFGKEY_MAPSEL = "Maptype";
	static final String CFGKEY_LINESEL = "Linetype";
	static final String CFGKEY_CONTACTS = "Contacts";
	static final String CFGKEY_QUERY = "Query";
	static final String CFGKEY_WORKING = "Working";

	/** initial default SMILES usage. */
	static final Boolean DEFAULT_MODE = false;
	static final Integer DEFAULT_TILE = 150;
	static final String DEFAULT_QUERY = "None";
	static final String DEFAULT_MAPSEL = "None";
	static final String DEFAULT_LINESEL = "None";
	static final Double DEFAULT_CONTACTS = 0.6;
	static final Boolean DEFAULT_SMILES = false;
	static final String DEFAULT_WORKING =  System.getenv("THINK_WORKING");

	// example value: the models count variable filled from the dialog 
	// and used in the models execution method. The default components of the
	// dialog work with "SettingsModels".
	private final SettingsModelBoolean m_mode =
		new SettingsModelBoolean(DisplayNodeModel.CFGKEY_MODE,
                    DisplayNodeModel.DEFAULT_MODE );
	
	private final SettingsModelIntegerBounded m_tile =
		new SettingsModelIntegerBounded(DisplayNodeModel.CFGKEY_TILE,
                    DisplayNodeModel.DEFAULT_TILE,
                    Integer.MIN_VALUE, Integer.MAX_VALUE);

	private final SettingsModelBoolean m_smiles =
		new SettingsModelBoolean(DisplayNodeModel.CFGKEY_SMILES,
                    DisplayNodeModel.DEFAULT_SMILES );
	
	private final SettingsModelString m_query =
	new SettingsModelString(DisplayNodeModel.CFGKEY_QUERY, DisplayNodeModel.DEFAULT_QUERY );

	private final SettingsModelString m_maptype =
		new SettingsModelString(DisplayNodeModel.CFGKEY_MAPSEL, DisplayNodeModel.DEFAULT_MAPSEL );

	private final SettingsModelString m_linetype =
		new SettingsModelString(DisplayNodeModel.CFGKEY_LINESEL, DisplayNodeModel.DEFAULT_LINESEL );

	private final SettingsModelDoubleBounded m_contacts =
		new SettingsModelDoubleBounded(DisplayNodeModel.CFGKEY_CONTACTS,
                    DisplayNodeModel.DEFAULT_CONTACTS,
                    0., 5.0);

	private final SettingsModelString m_working =
		new SettingsModelString(DisplayNodeModel.CFGKEY_WORKING, DisplayNodeModel.DEFAULT_WORKING );
	
   /**
     * Constructor for the node model.
     */
	public static final PortType OPTIONAL_PORT_TYPE = new PortType(BufferedDataTable.class, true);

    protected DisplayNodeModel() {
    
        // One incoming optional port and one outgoing port
        super(createOPOs(2, 1, 2), createOPOs(1));
    }
   	private static PortType[] createOPOs(final int nrDataPorts, final int... optionalPortsIds)
   	{
   	    PortType[] portTypes = new PortType[nrDataPorts];
   	    Arrays.fill(portTypes, BufferedDataTable.TYPE);        
    	    if (optionalPortsIds.length > 0) {
   	        for (int portId : optionalPortsIds) {
   	            if ((portId - 1) < nrDataPorts) {
   	                portTypes[portId - 1] = OPTIONAL_PORT_TYPE;
   	            }
   	        }
   	    }
   	    return portTypes;
   	} 
    @Override
    protected DataTableSpec[] configure(final DataTableSpec[] inSpecs)
    throws InvalidSettingsException {
        if ( System.getenv("THINK_EXEC") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_EXEC is unset - It should be set to the folder containing the THINK software.");       	  		
    	}
        if ( System.getenv("THINK_WORKING") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_WORKING is not set to the folder in which working files will be created.");       	  		
    	}

        DataTableSpec in = inSpecs[0];
        if ( in != null )
        	n_Column = columnSelect (in);
        DataTableSpec inQuery = inSpecs[1];
        if ( m_query.getStringValue().equalsIgnoreCase("Input") ) {
          if ( inQuery != null )
        	  n_Query = columnSelect (inQuery);
          else
              throw new InvalidSettingsException(
              "Input query mode selected without query input table connected");       	  		
        }
        return new DataTableSpec[] {null };
    }
    protected int columnSelect ( final DataTableSpec in ) throws  InvalidSettingsException {

        StringBuilder warningMessage = new StringBuilder();
        int n_column=-1;
        int sdfColCount = 0;
        int n_sdf=0;
        int smilesColCount = 0;
        int n_smiles=0;
        int pdbColCount = 0;
        int n_pdb=0;
        int stringColCount = 0;
        int n_string=0;
      	if ( in.getNumColumns() >= 3 ){
    		if ( in.getColumnSpec(2).getName().equals("File") )
    		 return -999;
    	}
        
        for (int i = 0; i < in.getNumColumns(); i++) {
            DataColumnSpec s = in.getColumnSpec(i);
            if (s.getType().isCompatible(PdbValue.class)) {
                if (m_pdbColumn == null) {
                    m_pdbColumn = in.getColumnSpec(i).getName();
                    n_pdb = i;
                }
                pdbColCount++;
            }
            else if (s.getType().isCompatible(SdfValue.class)) {
                if (m_sdfColumn == null) {
                    m_sdfColumn = in.getColumnSpec(i).getName();
                    n_sdf = i;
                }
                sdfColCount++;
            }
            else if ( s.getType().isCompatible(SmilesValue.class)) {
            	if (m_smilesColumn == null ) {
             		m_smilesColumn = in.getColumnSpec(i).getName();
             		n_smiles=i;
              	}
               	smilesColCount++;
            }
            else if ( in.getColumnSpec(i).getName().equalsIgnoreCase("smiles"))  {
            	if (m_stringColumn == null ) {
             		m_stringColumn = in.getColumnSpec(i).getName();
             		n_string=i;
              	}
               	stringColCount++;
            }
        }
        if ( pdbColCount == 0 & smilesColCount == 0 & sdfColCount == 0 & stringColCount == 0) {
            throw new InvalidSettingsException(
                    "No column with Smiles or SD or PDB file compatible type in molecule input");       	
        }
        else if ( pdbColCount > 0 ) {
        	n_column = n_pdb;
        	if (pdbColCount > 1) {
                warningMessage.append("More than one PDB compatible column in " 
                        + "input, using column \"" + m_pdbColumn + "\".");
        	}
        }
        else if ( ( m_smiles.getBooleanValue() | sdfColCount == 0 ) & smilesColCount > 0 ) {
        	m_sdfColumn = null;
        	m_pdbColumn = null;
        	m_stringColumn = null;
        	n_column = n_smiles;
        	if (smilesColCount > 1) {
                warningMessage.append("More than one Smiles compatible column in " 
                        + "input, using column \"" + m_smilesColumn + "\".");
        	}
        }
        else if ( sdfColCount > 0 ) {
           	m_smilesColumn = null;
           	m_pdbColumn = null;
        	m_stringColumn = null;
        	n_column = n_sdf;
        	if (sdfColCount > 1) {
                warningMessage.append("More than one SDF compatible column in " 
                        + "input, using column \"" + m_sdfColumn + "\".");
        	}
        }
        else if ( stringColCount > 0 ) {
        	m_smilesColumn = null;
           	m_pdbColumn = null;
        	m_sdfColumn = null;
        	n_column = n_string;
           	if (stringColCount > 1) {
                warningMessage.append("More than one Smiles string compatible column in " 
                        + "input, using column \"" + m_stringColumn + "\".");
           	}
        }
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
        return n_column;
    }


    /**
     * @see org.knime.core.node.NodeModel #execute(BufferedDataTable[],
     *      ExecutionContext)
     */
    protected String writeMoleculeFile ( final BufferedDataTable in, String filename )throws Exception {
        final String cwdString = m_working.getStringValue();
        String fileString = "None"; 

        DataTableSpec inSpec = in.getDataTableSpec();
        n_Column = columnSelect(inSpec);
        if ( n_Column == -999 ) {
          final double count = in.getRowCount(); // floating point operations
          if ( count > 1 ) {
              setWarningMessage("Using only first row of protein sites");
          }

          for (DataRow r : in ) {
            DataCell c = r.getCell(2);
            if (!c.isMissing()){
              fileString = c.toString();
            }
            break;
          }

       	return fileString; 
        }

        int n_pdbColumn = -1;
    	int n_sdfColumn = -1;
    	int n_smilesColumn = -1;
        // write out input file
    	if ( inSpec.getColumnSpec(n_Column).getType().isCompatible(PdbValue.class) ) {
     	    n_pdbColumn = n_Column;
    		fileString = filename + ".pdb"; 
    	}
    	else if ( inSpec.getColumnSpec(n_Column).getType().isCompatible(SdfValue.class) ) {
     	    n_sdfColumn = n_Column;
    		fileString = filename + ".sdf"; 
    	}
       	else if ( inSpec.getColumnSpec(n_Column).getType().isCompatible(SmilesValue.class) ) {
     	    n_smilesColumn = n_Column;
    		fileString = filename + ".smiles"; 
    	}
    	else {
    		fileString = filename + ".smiles"; 
    	}
    	File inFile = new File(cwdString, fileString); 
    	BufferedWriter outWriter = new BufferedWriter(new FileWriter(inFile));
	  
        int colIndex;
    	if ( n_Column >= 0 )
    		colIndex = n_Column;
    	else {
            throw new InvalidSettingsException(
            "Molecular data column not found");       	  		
        }
        final double count = in.getRowCount(); // floating point operations
        int i = 0; 
        int missingCount = 0;
        for (DataRow r : in) {
            DataCell c = r.getCell(colIndex);
            if (c.isMissing()) {
                missingCount++;
            } 
            else if ( n_pdbColumn >= 0 ) {
                PdbValue v = (PdbValue)c;
                String toString = v.toString();
                outWriter.write(toString);
             }
            else if ( n_sdfColumn >= 0 ) {
                SdfValue v = (SdfValue)c;
                String toString = v.toString();
                outWriter.write(toString);
                if (!toString.trim().endsWith("$$$$")) {
                    outWriter.newLine();
                    outWriter.append("$$$$");
                    outWriter.newLine();
                }
            }
            else if ( n_smilesColumn >= 0 ) {
                SmilesValue v = (SmilesValue)c;
                String toString = v.toString() + " " + r.getKey();
                outWriter.write(toString);        	
                outWriter.newLine();
            }
            else {
            	String toString = c.toString() + " " + r.getKey();
            	outWriter.write(toString);        	
            	outWriter.newLine();
            }
            i++;
        }

        outWriter.close();

        if (missingCount > 0) {
	            setWarningMessage("Skipped " + missingCount 
	                    + " row(s) because of missing values");
	       }

        return fileString;

    }
    @Override
    protected BufferedDataTable[] execute(final BufferedDataTable[] inData,
            final ExecutionContext exec) throws Exception {

        notifyViews(null);

        final String cwdString = m_working.getStringValue();
        // clean up 
        File outputReport = new File(cwdString, "output99.log"); 
        outputReport.delete();
        File outputFile = new File(cwdString, "display.csv"); 
        outputFile.delete();
        outputFile = new File(cwdString, "display.pdb"); 
        outputFile.delete();
        outputFile = new File(cwdString, "display.sdf"); 
        outputFile.delete();
 
        String s_query = m_query.getStringValue();
        String queryString = "Undefined";
        BufferedDataTable inQuery = inData[1];
       	if ( s_query.equalsIgnoreCase("Input") ) {
            if ( inQuery != null ) {
                 queryString = writeMoleculeFile(inQuery,"displayQuery");	
             }
            else
                throw new InvalidSettingsException(
                s_query + " query mode selected without query input table connected");       	  		
        }
          
        BufferedDataTable in = inData[0];
        String fileString = "None";
        if ( in != null)
            fileString = writeMoleculeFile(in,"display");	

        File cwdFile = new File (cwdString);
         // execute THINK
        int exitVal;
        try {
            exec.setProgress(0., "Starting THINK");
            Runtime rt = Runtime.getRuntime();
            // prepare the command string
            String s_maptype = m_maptype.getStringValue();
            String s_mode = "3D";
            if ( m_mode.getBooleanValue() ) {
            	s_mode = "TILE";
            }
            int n_tile = m_tile.getIntValue();
            String s_linetype = m_linetype.getStringValue();
            double d_contacts = m_contacts.getDoubleValue();
            String cmdString = System.getenv("THINK_EXEC") + "think THINK_EXEC:display.log output99.log " + fileString + " " + s_query +  " " + s_maptype + " " + s_mode + " " + n_tile + " " + s_linetype + " " + d_contacts + " " + queryString ;
            // Go!
            logger.info("THINK command line: '" + cmdString + "'");
            final Process proc = rt.exec(cmdString, null, cwdFile);

            final MutableBoolean procDone = new MutableBoolean(false);
            new Thread(new Runnable() {
                public void run() {
                    synchronized (procDone) {
                        while (!procDone.booleanValue()) {
                            try {
                                exec.checkCanceled();
                            } catch (CanceledExecutionException cee) {
                                // blow away the running external process
                                proc.destroy();
                                return;
                            }
                            try {
                               	procDone.wait(CANCEL_CHECK_INTERVAL);
                                // Read progress file and set progress value       
                               	double d= 0.;
                                File progressReport = new File(cwdString, "progress0.dat");
                                if ( progressReport.exists() && progressReport.isFile() ) {
                               		if ( progressReport.length() > 0 ) {
                               			BufferedReader in = new BufferedReader(new FileReader(progressReport));
                               			String line;
                               			if ( (line = in.readLine()) != null) {
                               			  d = Double.valueOf(line).doubleValue();
                               			}               
                              			if ( (line = in.readLine()) == null) {
                              				line = "Running THINK";
                              			}
                               			in.close();
                                        exec.setProgress( d, line.toString());
                               		}
                                }
                            } catch (InterruptedException e) {
                                // do nothing
                            } catch ( FileNotFoundException e) {
                                // do nothing
                            } catch ( IOException e) {
                                // do nothing
                            }
                        }
                    }
                }

            }).start();
           

            // wait until the external process finishes.
            exitVal = proc.waitFor();

            synchronized (procDone) {
                // this should terminate the check cancel thread
                procDone.setValue(true);
            }

            exec.checkCanceled();

            exec.setProgress("Wrapping up");
            logger.info("THINK terminated with exit code: " + exitVal);
        } catch (Throwable t) {
        	warningReport();
        	logger.error("THINK failed (with exception)", t);
            throw new Exception(t);
        }

        if (exitVal != 0) {
            // before we return barfing, we save the output in the failing list
            warningReport();
        	throw new IllegalStateException("THINK failed (error code "
                    + exitVal + ")");
        }

        // Read the output and echo it to the console
        exec.setProgress("Reading data from output file");
        BufferedDataTable outTable = null;
        if ( m_mode.getBooleanValue() | fileString == "None" ) {
          	outTable = readCSVFile(exec);
        }
        else if ( fileString.endsWith( ".pdb") ) { 
        	outTable = readPDBFile(exec);
        }
        else {
        	outTable = readSDFFile(exec);
        }

       return new BufferedDataTable[]{outTable};

    }
    private BufferedDataTable readPDBFile(final ExecutionContext exec) throws Exception {

    	String cwdString = m_working.getStringValue();
        String line;
        File outFile = new File (cwdString, "display.pdb");

    	if (!(outFile.exists() && outFile.isFile())) {
    		warningReport();
    		throw new IllegalStateException("THINK didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
         
        // Create the table
        DataColumnSpec[] allColSpecs;
        allColSpecs = new DataColumnSpec[1];
        int count =0;
        allColSpecs[0] = 
        new DataColumnSpecCreator("Protein",PdbCell.TYPE).createSpec();
        DataTableSpec outputSpec = new DataTableSpec(allColSpecs);
        BufferedDataContainer cont = exec.createDataContainer(outputSpec);
                
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        
        HashSet<String> titles = new HashSet<String>();

        StringBuilder buf = new StringBuilder(4096000);
        boolean titleRead = false;
        String title = "";
    		
        DataCell[] cells = new DataCell[count+1];
        int molcount=0;
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();

            if (line.toString().startsWith("NAME")) {
            	if ( titleRead ){
                    cells[0] = new PdbCell(buf.toString());
                    RowKey key = new RowKey(title);
                    DefaultRow row = new DefaultRow(key, cells);
                    cont.addRowToTable(row);
                    buf.delete(0, buf.length());           		
            	}
                title = line.substring(4).trim();
                if (title.length() == 0) {
                    title = "Mol " + molcount;
                }
                if (titles.contains(title)) {
                    title = title + "-" + molcount;
                }
                titles.add(title);
                titleRead = true;
            }
            buf.append(line).append('\n');

        }
        if ( titleRead ) {
          cells[0] = new PdbCell(buf.toString());
          RowKey key = new RowKey(title);
          DefaultRow row = new DefaultRow(key, cells);
          cont.addRowToTable(row);
          buf.delete(0, buf.length());
        }

        cont.close();
        in.close();
 
        return cont.getTable();
 
    }

    private void warningReport( ) throws Exception {

    	StringBuilder warningMessage = new StringBuilder();
        String cwdString = m_working.getStringValue();
        File outputReport = new File(cwdString, "output2.log"); 

        if ( outputReport.exists() && outputReport.isFile() ) {
       		if ( outputReport.length() > 0 ) {
       			BufferedReader in = new BufferedReader(new FileReader(outputReport));
       			String line;
       			while ((line = in.readLine()) != null) {
       				warningMessage.append( line + "\n");
       			}               
       			in.close();
       		}
       		else { 
                warningMessage.append( "Report from THINK is blank");      	     			
       		}
        }
       	else {
            warningMessage.append( "No report from THINK");      	
       	}	
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
   }
    private BufferedDataTable readCSVFile(final ExecutionContext exec) throws Exception { 

        String cwdString =  m_working.getStringValue();
        File outFile = new File (cwdString, "display.csv");
       	if (!(outFile.exists() && outFile.isFile())) {
    		return null;
    	}
        HashSet<String> titles = new HashSet<String>();
        String title;
        String line;
        boolean headerRead = false;
        boolean columnsCreated = false;
        DataColumnSpec[] allColSpecs;
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        DataTableSpec outputSpec;
        BufferedDataContainer cont =null;
        RowKey key =null;
        int count =0;
        int colcount =0;
        int rowcount =0;
        char chr='0';
        int keys = 0;
        int ilast = 0;
        String fieldNames[] = new String[100];
        int fieldTypes[] = new int[100];
        while ((line = in.readLine()) != null) {
             exec.checkCanceled();
// Extract column headers
            if ( !headerRead ) {
                headerRead = true;
                colcount =0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) 
                        chr = line.charAt(i);
                    else
                        chr = ',';
                    if ( chr == ',' ) { 
                        fieldNames[colcount] = line.substring(ilast,i);
                        if ( line.substring(ilast,i).equalsIgnoreCase("SMILES"))
                        	fieldTypes[colcount]= 2;
                        else if ( line.substring(ilast,i).equalsIgnoreCase("NAME"))
                        	fieldTypes[colcount]= 1;
                        else if ( line.substring(ilast,i).equalsIgnoreCase("KEYS"))
                        	fieldTypes[colcount]= 3;
                        ilast = i+1;
                        colcount++;
                    }
                }
                colcount--;
            }
// Load data
            else {
                DataCell[] cells = new DataCell[colcount];
                rowcount++;
                count =0;                 
                double d_value;
                 for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) 
                        chr = line.charAt(i);
                    else
                        chr = ',';
                    if ( chr == ',' ) { 
                        if ( count == 0 ) {
                            title = line.substring(0,i);
                            if (title.length() == 0) 
                                title = "Mol " + rowcount;
                            if (titles.contains(title)) 
                                title = title + "-" + rowcount;
                            titles.add(title);
                            key = new RowKey(title);                           
                        }
                        else if  ( fieldTypes[count] == 2 ) 
                            cells[count-1] = new SmilesCell(line.substring(ilast,i));
                        else if  ( fieldTypes[count]== 3) 
                            cells[count-1] = new StringCell(line.substring(ilast,i));
                        else if ( count <= colcount ) {
                          try {
                            d_value = Double.valueOf(line.substring(ilast,i)).doubleValue();
                            cells[count-1] = new DoubleCell(d_value);
                            if ( fieldTypes[count] == 0 )
                            	fieldTypes[count] = 4;
                          } catch (Throwable t) {
                            if ( fieldTypes[count] == 0 ) {
                            	fieldTypes[count] = 3;
                                cells[count-1] = new StringCell(line.substring(ilast,i));
                            }
                            else
                                cells[count-1] = DataType.getMissingCell();
                          }
                        }
                        count ++;
                        ilast = i + 1;
                    }
                 }
  // Create columns?
                 if (!columnsCreated )
                 {
                     allColSpecs = new DataColumnSpec[colcount];
                     columnsCreated = true;
  // count= 0 will be molecule names
                     for (count=1; count<=colcount; count++)
                     {                   	 
                       if  ( fieldTypes[count] == 2 ) 
                         allColSpecs[count-1] = new DataColumnSpecCreator(fieldNames[count],SmilesCell.TYPE).createSpec();
                       else if  ( fieldTypes[count] == 4 ) 
                         allColSpecs[count-1] = new DataColumnSpecCreator(fieldNames[count], DoubleCell.TYPE).createSpec();
                       else if ( fieldTypes[count] == 3 )
                         allColSpecs[count-1] = new DataColumnSpecCreator(fieldNames[count],StringCell.TYPE).createSpec();
                       else
                    	 allColSpecs[count-1] = new DataColumnSpecCreator("Unknown",StringCell.TYPE).createSpec();
                     }
                     outputSpec = new DataTableSpec(allColSpecs);
                     cont = exec.createDataContainer(outputSpec);               	 
                 }
                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
            }
        }
        cont.close();
        in.close();

        return cont.getTable();
    }

    private BufferedDataTable readOutputFile(final ExecutionContext exec) throws Exception {

    	String cwdString = m_working.getStringValue();
        File outFile = new File (cwdString, "display.csv");

    	if (!(outFile.exists() && outFile.isFile())) {
    		return null;
    	}
    	
        HashSet<String> titles = new HashSet<String>();
        String title;
        String line;
        boolean headerRead = false;
        DataColumnSpec[] allColSpecs;
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        DataTableSpec outputSpec;
        BufferedDataContainer cont =null;
        RowKey key =null;
        int count =0;
        int rowcount =0;
        char chr='0';
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();
// Extract column headers
            if ( !headerRead ) {
                headerRead = true;
                count =0;
                for ( int i=0; i<line.length(); i++ ) {
                    if ( line.charAt(i) == ',' ) count ++;
                }
// Create column definitions
                allColSpecs = new DataColumnSpec[1];
                count =0;
                int ilast=0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else {
                        chr = ',';
                    }
                    if ( chr == ',' ) {
                        if  ( count == 1 ) {
                            allColSpecs[0] = 
                            new DataColumnSpecCreator(line.substring(ilast,i),SmilesCell.TYPE).createSpec();
                        }
                        count ++;
                        ilast = i + 1;
                    }
                }
                outputSpec = new DataTableSpec(allColSpecs);
                cont = exec.createDataContainer(outputSpec);
            }
// Load data
            else {
                DataCell[] cells = new DataCell[1];
                rowcount++;
                count =0;                  
                double d_value;
                int ilast=0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else
                        chr = ',';
                    if ( chr == ',' ) { 
                        if ( count == 0 ) {
                            title = line.substring(0,i);
                            if (title.length() == 0) {
                                title = "Mol " + rowcount;
                            }
                            if (titles.contains(title)) {
                                title = title + "-" + rowcount;
                            }
                            titles.add(title);
                            key = new RowKey(title);                           
                        }
                        else if  ( count == 1 ) {
                            cells[0] = new SmilesCell(line.substring(ilast,i));   	
                        }
                        count ++;
                        ilast = i + 1;
                    }
                }
                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
            }
        }
        cont.close();
        in.close();

        return cont.getTable();
    }

    private BufferedDataTable readSDFFile(final ExecutionContext exec) throws Exception {

    	String cwdString = m_working.getStringValue();
        String line;
        String fieldName;
        String doubleFields[] = new String[50];
        String stringFields[]= new String[50];
        int smileCount=0;
        int keyCount=0;
        int doubleCount=0;
        int stringCount=0;
         File outFile = new File (cwdString, "display.sdf");

    	if (!(outFile.exists() && outFile.isFile())) {
    		warningReport();
    		throw new IllegalStateException("THINK didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
    	// Scan file for fields and data types
    	
        BufferedReader looksee = new BufferedReader(new FileReader(outFile));
        while ((line = looksee.readLine()) != null) {
            if (line.startsWith("$$$$")) 
            	break;
            else if (line.startsWith("> <")) {
            	int l = line.length() - 1;
            	fieldName = line.substring(3,l);
            	line = looksee.readLine();
            	if ( fieldName.equals("SMILES")) {
            		smileCount = 1;
            	}
            	else if ( fieldName.equals("KEYS")) {
            		keyCount = 1;
            	}
            	else {
            		try {
            			double d_value = Double.valueOf(line).doubleValue();
            			doubleFields[doubleCount] = fieldName;
            			doubleCount ++;
            		}
            		catch (Throwable t){
            			stringFields[stringCount] = fieldName;
            			stringCount++;
            		}
            	}
            }	
        }	        
        looksee.close();
        
        // Create the table
        DataColumnSpec[] allColSpecs;
        allColSpecs = new DataColumnSpec[stringCount+doubleCount+smileCount+keyCount+1];
        int count =0;
        int keyStart=0;
        int doubleStart=0;
        int stringStart=0;
        allColSpecs[0] = 
            new DataColumnSpecCreator("Molecule",SdfCell.TYPE).createSpec();
        if ( smileCount == 1 ) {
        	count ++;
            allColSpecs[count] = 
            	new DataColumnSpecCreator("SMILES",SmilesCell.TYPE).createSpec();
        }
        if ( keyCount == 1 ) {
        	keyCount = count + 1;
        	count ++;
            allColSpecs[count] = 
            	new DataColumnSpecCreator("KEYS",StringCell.TYPE).createSpec();
        }
        if ( stringCount > 0 ) {
        	stringStart=count + 1;
        	for (int i=0; i<stringCount; i++ ) {
        		count ++;
                allColSpecs[count] = 
        		new DataColumnSpecCreator(stringFields[i],StringCell.TYPE).createSpec();
        	}
        }
        if ( doubleCount > 0 ) {
           	doubleStart=count + 1;      	
        	for (int i=0; i<doubleCount; i++ ) {
        		count ++;
                allColSpecs[count] = 
        		new DataColumnSpecCreator(doubleFields[i],DoubleCell.TYPE).createSpec();
        	}
        }
        DataTableSpec outputSpec = new DataTableSpec(allColSpecs);
        BufferedDataContainer cont = exec.createDataContainer(outputSpec);
                
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        
        HashSet<String> titles = new HashSet<String>();

        StringBuilder buf = new StringBuilder(4096);
        boolean titleRead = false;
        String title = "";
    		
        DataCell[] cells = new DataCell[count+1];
        int molcount=0;
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();

            if (!titleRead) {
                title = line;
                if (title.length() == 0) {
                    title = "Mol " + molcount;
                }
                if (titles.contains(title)) {
                    title = title + "-" + molcount;
                }
                titles.add(title);
                titleRead = true;
            }
            buf.append(line).append('\n');

            if (line.startsWith("$$$$")) {
            	molcount ++;
                cells[0] = new SdfCell(buf.toString());
                RowKey key = new RowKey(title);

                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
                buf.delete(0, buf.length());
                titleRead = false;
            }
            else if ( line.startsWith("> <")) {
            	int l = line.length() - 1;
            	fieldName = line.substring(3,l);
            	line = in.readLine();
            	if ( smileCount == 1 & fieldName.equals("SMILES")) {
            		cells[1] = new SmilesCell(line);
            	}
            	else if ( keyCount == 1 & fieldName.equals("KEYS")) {
                    cells[keyStart]= new StringCell(line);
            	}
            	else {
            		for (int i=0; i<doubleCount; i++ ) {
            			if ( doubleFields[i].equals(fieldName)) {
            				double d_value = Double.valueOf(line).doubleValue();
            				cells[doubleStart+i]= new DoubleCell(d_value);
            				break;
            			}
            		}
            		for (int i=0; i<stringCount; i++ ) {
            			if ( stringFields[i].equals(fieldName)) {
            				cells[stringStart+i]= new StringCell(line);
            				break;
            			}
            		}
            	}
            }
        }
        cont.close();
        in.close();
 
        return cont.getTable();
 
  
    }
 /**
     * @see org.knime.core.node.NodeModel#reset()
     */
    @Override
    protected void reset() {
        // TODO Code executed on reset.
        // Models build during execute are cleared here.
        // Also data handled in load/saveInternals will be erased here.
    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #configure(org.knime.core.data.DataTableSpec[])
     */
    /*
    @Override
    protected DataTableSpec[] configure(final DataTableSpec[] inSpecs)
            throws InvalidSettingsException {
        
        // TODO: check if user settings are available, fit to the incoming
        // table structure, and the incoming types are feasible for the node
        // to execute. If the node can execute in its current state return
        // the spec of its output data table(s) (if you can, otherwise an array
        // with null elements), or throw an exception with a useful user message

        return new DataTableSpec[]{null};
    }
    */
    /**
     * @see org.knime.core.node.NodeModel
     *      #saveSettingsTo(org.knime.core.node.NodeSettings)
     */
    @Override
    protected void saveSettingsTo(final NodeSettingsWO settings) {

        // TODO save user settings to the config object.
		
		m_mode.saveSettingsTo(settings);
		m_tile.saveSettingsTo(settings);
		m_smiles.saveSettingsTo(settings);
		m_query.saveSettingsTo(settings);
		m_maptype.saveSettingsTo(settings);
		m_linetype.saveSettingsTo(settings);
		m_contacts.saveSettingsTo(settings);
		m_working.saveSettingsTo(settings);

    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #loadValidatedSettingsFrom(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void loadValidatedSettingsFrom(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO load (valid) settings from the config object.
        // It can be safely assumed that the settings are valided by the 
        // method below.
        
        m_mode.loadSettingsFrom(settings);
        m_tile.loadSettingsFrom(settings);
        m_smiles.loadSettingsFrom(settings);
        m_maptype.loadSettingsFrom(settings);
        m_linetype.loadSettingsFrom(settings);
        m_contacts.loadSettingsFrom(settings);
        m_query.loadSettingsFrom(settings);
        m_working.loadSettingsFrom(settings);

    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #validateSettings(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void validateSettings(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO check if the settings could be applied to our model
        // e.g. if the count is in a certain range (which is ensured by the
        // SettingsModel).
        // Do not actually set any values of any member variables.

        m_mode.validateSettings(settings);
        m_tile.validateSettings(settings);
        m_smiles.validateSettings(settings);
        m_maptype.validateSettings(settings);
        m_linetype.validateSettings(settings);
        m_contacts.validateSettings(settings);
        m_query.validateSettings(settings);
        m_working.validateSettings(settings);

    }
    
    /**
     * @see org.knime.core.node.NodeModel #loadInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void loadInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
        
		// TODO load internal data. 
		// Everything handed to output ports is loaded automatically (data
		// returned by the execute method, models loaded in loadModelContent,
		// and user settings set through loadSettingsFrom - is all taken care 
		// of). Load here only the other internals that need to be restored
		// (e.g. data used by the views).

    }
    
    /**
     * @see org.knime.core.node.NodeModel #saveInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void saveInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
       
       	// TODO save internal models. 
		// Everything written to output ports is saved automatically (data
		// returned by the execute method, models saved in the saveModelContent,
		// and user settings saved through saveSettingsTo - is all taken care 
		// of). Save here only the other internals that need to be preserved
		// (e.g. data used by the views).

    }

}
