package com.treweren.display;


import java.awt.FlowLayout;

import org.knime.core.node.defaultnodesettings.DefaultNodeSettingsPane;
import org.knime.core.node.defaultnodesettings.DialogComponentBoolean;
import org.knime.core.node.defaultnodesettings.DialogComponentNumber;
import org.knime.core.node.defaultnodesettings.DialogComponentString;
import org.knime.core.node.defaultnodesettings.DialogComponentStringSelection;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelDoubleBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelString;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;

/**
 * <code>NodeDialog</code> for the "Display" Node.
 * Uses THINK to display molecules
 *
 * This node dialog derives from {@link DefaultNodeSettingsPane} which allows
 * creation of a simple dialog with standard components. If you need a more 
 * complex dialog please derive directly from 
 * {@link org.knime.core.node.NodeDialogPane}.
 * 
 * @author Treweren Consultants
 */
public class DisplayNodeDialog extends DefaultNodeSettingsPane {

    /**
     * New pane for configuring Display node dialog.
     * This is just a suggestion to demonstrate possible default dialog
     * components.
     */
    protected DisplayNodeDialog() {
        super();
        createNewGroup("2D options");
   
        DialogComponentBoolean c_mode = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        DisplayNodeModel.CFGKEY_MODE,
                        DisplayNodeModel.DEFAULT_MODE),
                        "Display in 2D style" );
        addDialogComponent(c_mode);
        c_mode.getComponentPanel().setLayout( new FlowLayout(0));
        c_mode.setToolTipText("2D style is normally used for informatics and 3D style for modelling");
        
        DialogComponentNumber c_tile = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                    DisplayNodeModel.CFGKEY_TILE,
                    DisplayNodeModel.DEFAULT_TILE,
                    Integer.MIN_VALUE, Integer.MAX_VALUE),
                    "2D Tile size", /*step*/ 25, /*componentwidth*/ 4);
        addDialogComponent(c_tile);
        c_tile.getComponentPanel().setLayout( new FlowLayout(2));
        c_tile.setToolTipText("Increasing the size of tile (pane) reduces the number of molecules per window");
        
        closeCurrentGroup();
        
        createNewGroup("Query options");
        
        DialogComponentStringSelection c_query = new DialogComponentStringSelection (
        		new SettingsModelString(DisplayNodeModel.CFGKEY_QUERY, null),
        		"Include query from", "None", "Input","Search", "Similarity","Docking");
        addDialogComponent (c_query); 
        c_query.getComponentPanel().setLayout( new FlowLayout(2));
        c_query.setToolTipText("Visualising how the hits relate to a query can be important");

        DialogComponentStringSelection c_mapsel = new DialogComponentStringSelection (
        		new SettingsModelString(DisplayNodeModel.CFGKEY_MAPSEL, null),
        		"Map type", "None", "Pharmacophore", "VDW", "Potential", "Ligand");
        addDialogComponent (c_mapsel); 
        c_mapsel.getComponentPanel().setLayout( new FlowLayout(2));
        c_mapsel.setToolTipText("A map may be constructed for the query (if present)");

        DialogComponentStringSelection c_linesel = new DialogComponentStringSelection (
        		new SettingsModelString(DisplayNodeModel.CFGKEY_LINESEL, null),
        		"Line type", "None", "Mapping", "Interactions", "Contacts");
        addDialogComponent (c_linesel); 
        c_linesel.getComponentPanel().setLayout( new FlowLayout(2));
        c_linesel.setToolTipText("It can be helpful to highlight atom relationships between the hits and the query (if present)");

        DialogComponentNumber c_contacts = new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                        DisplayNodeModel.CFGKEY_CONTACTS,
                        DisplayNodeModel.DEFAULT_CONTACTS, 0.1, 2. ),
                        "VdW contacts multiplier", 0.1, 3 );
        addDialogComponent(c_contacts );
        c_contacts.getComponentPanel().setLayout( new FlowLayout(2));
        c_contacts.setToolTipText("It can be useful to use an extended radius for contacts as a result of missing hydrogens and errors");
        
        closeCurrentGroup();

        
        createNewTab("THINK setup");         

        DialogComponentBoolean c_column = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        DisplayNodeModel.CFGKEY_SMILES,
                        DisplayNodeModel.DEFAULT_SMILES),
                        "Use SMILES in preference to SD column" );
        addDialogComponent(c_column);
        c_column.getComponentPanel().setLayout( new FlowLayout(0));
        c_column.setToolTipText("The molecule data is transferred to THINK using SD or SMILES format depending on this setting");
  
        DialogComponentString c_working = new DialogComponentString(
                new SettingsModelString (
                        DisplayNodeModel.CFGKEY_WORKING,
                        DisplayNodeModel.DEFAULT_WORKING),
                        "Working folder" );
        addDialogComponent(c_working);
        c_working.getComponentPanel().setLayout( new FlowLayout(2));
        c_working.setToolTipText("The temporary files used to transfer the data to and from THINK are placed in this folder");

     }
}
