package com.treweren.findsites;

import java.awt.FlowLayout;

import javax.swing.JFileChooser;

import org.knime.core.node.defaultnodesettings.DefaultNodeSettingsPane;
import org.knime.core.node.defaultnodesettings.DialogComponentBoolean;
import org.knime.core.node.defaultnodesettings.DialogComponentFileChooser;
import org.knime.core.node.defaultnodesettings.DialogComponentNumber;
import org.knime.core.node.defaultnodesettings.DialogComponentString;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelDoubleBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelString;


/**
 * <code>NodeDialog</code> for the "FindSites" Node.
 * Find docking sites for a protein
 *
 * This node dialog derives from {@link DefaultNodeSettingsPane} which allows
 * creation of a simple dialog with standard components. If you need a more 
 * complex dialog please derive directly from 
 * {@link org.knime.core.node.NodeDialogPane}.
 * 
 * @author Treweren Consultants
 */
public class FindSitesNodeDialog extends DefaultNodeSettingsPane {

    /**
     * New pane for configuring FindSites node dialog.
     * This is just a suggestion to demonstrate possible default dialog
     * components.
     */
    protected FindSitesNodeDialog() {
        super();
        
        DialogComponentFileChooser c_search = new DialogComponentFileChooser(
                new SettingsModelString (
                        FindSitesNodeModel.CFGKEY_SEARCH,
                        FindSitesNodeModel.DEFAULT_SEARCH ),
                        "Protein file", JFileChooser.OPEN_DIALOG, "pdb" );
        addDialogComponent(c_search );
        c_search.setToolTipText("The sites will be created from the protein in this file");

        DialogComponentBoolean c_ligands = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        FindSitesNodeModel.CFGKEY_LIGANDS,
                        FindSitesNodeModel.DEFAULT_LIGANDS),
                        "Use bound ligand sites" );
        addDialogComponent(c_ligands);
        c_ligands.getComponentPanel().setLayout( new FlowLayout(0));
        c_ligands.setToolTipText("Other molecules are likely to bind in the same site as an existing ligand");

        DialogComponentBoolean c_pdb = new DialogComponentBoolean(
                new SettingsModelBoolean (
                    FindSitesNodeModel.CFGKEY_PDB,
                    FindSitesNodeModel.DEFAULT_PDB),
                    "Use PDB site records" );
        addDialogComponent(c_pdb);
        c_pdb.getComponentPanel().setLayout( new FlowLayout(0));
        c_pdb.setToolTipText("PDB files can contain site definitions reported by the crystallographer");

        DialogComponentBoolean c_lipophile = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        FindSitesNodeModel.CFGKEY_LIPOPHILE,
                        FindSitesNodeModel.DEFAULT_LIPOPHILE),
                        "Include lipophiles" );
        addDialogComponent(c_lipophile);
        c_lipophile.getComponentPanel().setLayout( new FlowLayout(0));
        c_lipophile.setToolTipText("Create lipophilic interactions");
 
        DialogComponentBoolean c_sites = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        FindSitesNodeModel.CFGKEY_SITES,
                        FindSitesNodeModel.DEFAULT_SITES),
                        "Search for all sites" );
        addDialogComponent(c_sites);
        c_sites.getComponentPanel().setLayout( new FlowLayout(0));
        c_sites.setToolTipText("THINK can identify cavities/clefts which are sufficiently large for a molecule to bind");
   
        DialogComponentNumber c_tolerance = new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                        FindSitesNodeModel.CFGKEY_TOLERANCE,
                        FindSitesNodeModel.DEFAULT_TOLERANCE, 0., 2. ),
                        "Distance tolerance", .05, 5 );
        
        addDialogComponent(c_tolerance);
        c_tolerance.setToolTipText("Set the maximum separation of the interaction centre from the ligand atom");
        c_tolerance.getComponentPanel().setLayout( new FlowLayout(2));

        DialogComponentNumber c_radius = new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                    FindSitesNodeModel.CFGKEY_RADIUS,
                    FindSitesNodeModel.DEFAULT_RADIUS, 0.6, 2.5 ),
                    "Extended radius to VdW radius ratio", .05, 5 );
        addDialogComponent(c_radius );
        c_radius.getComponentPanel().setLayout( new FlowLayout(2));
        c_radius.setToolTipText("To accomodate errors and omitted hydrogens it is necessary to use a contact radius larger than the VdW radii identifying residues interacting with a ligand");

        createNewTab("Advanced");
        
        DialogComponentBoolean c_require = new DialogComponentBoolean(
                new SettingsModelBoolean (
                    FindSitesNodeModel.CFGKEY_REQUIRE,
                    FindSitesNodeModel.DEFAULT_REQUIRE),
                    "Preserve ligand interactions" );
        addDialogComponent(c_require);
        c_require.getComponentPanel().setLayout( new FlowLayout(0));
        c_require.setToolTipText("Require interactions with the same atoms as a docked ligand");
        
        DialogComponentString c_centre1 = new DialogComponentString(
                new SettingsModelString (
                    FindSitesNodeModel.CFGKEY_CENTRE1,
                    FindSitesNodeModel.DEFAULT_CENTRE1),
                    "Residues for interaction 1" );
        addDialogComponent(c_centre1);
        c_centre1.getComponentPanel().setLayout( new FlowLayout(0));
        c_centre1.setToolTipText("Comma separated list of alternative residues in protein for first interaction with syntax _residue(sequence)chain eg _(230),_PHE(207A),_HOH(218)C");

        DialogComponentString c_centre2 = new DialogComponentString(
                new SettingsModelString (
                    FindSitesNodeModel.CFGKEY_CENTRE2,
                    FindSitesNodeModel.DEFAULT_CENTRE2),
                    "Residues for interaction 2" );
        addDialogComponent(c_centre2);
        c_centre2.getComponentPanel().setLayout( new FlowLayout(0));
        c_centre2.setToolTipText("Comma separated list of alternative residues in protein for second interaction with syntax _residue(sequence)chain eg _(230),_PHE(207A),_HOH(218)C");

        DialogComponentString c_centre3 = new DialogComponentString(
                new SettingsModelString (
                    FindSitesNodeModel.CFGKEY_CENTRE3,
                    FindSitesNodeModel.DEFAULT_CENTRE3),
                    "Residues for interaction 3" );
        addDialogComponent(c_centre3);
        c_centre3.getComponentPanel().setLayout( new FlowLayout(0));
        c_centre3.setToolTipText("Comma separated list of alternative residues in protein for third interaction with syntax _residue(sequence)chain eg _(230),_PHE(207A),_HOH(218)C");

        DialogComponentString c_centre4 = new DialogComponentString(
                new SettingsModelString (
                    FindSitesNodeModel.CFGKEY_CENTRE4,
                    FindSitesNodeModel.DEFAULT_CENTRE4),
                    "Residues for interaction 4" );
        addDialogComponent(c_centre4);
        c_centre4.getComponentPanel().setLayout( new FlowLayout(0));
        c_centre4.setToolTipText("Comma separated list of alternative residues in protein for fourth interaction with syntax _residue(sequence)chain eg _(230),_PHE(207A),_HOH(218)C");


        createNewTab("THINK setup");         
    
        DialogComponentString c_working = new DialogComponentString(
                new SettingsModelString (
                        FindSitesNodeModel.CFGKEY_WORKING,
                        FindSitesNodeModel.DEFAULT_WORKING),
                        "Working folder" );
        addDialogComponent(c_working);
        c_working.getComponentPanel().setLayout( new FlowLayout(2));
        c_working.setToolTipText("The temporary files used to transfer the data to and from THINK are placed in this folder");
                  
    }
}
