package com.treweren.findsites;

import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.HashSet;

import org.knime.base.node.io.filereader.FileAnalyzer;
import org.knime.base.node.io.filereader.FileReaderNodeSettings;
import org.knime.base.node.io.filereader.FileReaderSettings;
import org.knime.base.node.io.filereader.FileTable;
import org.knime.base.node.io.filereader.SettingsStatus;
import org.knime.chem.types.SdfCell;
import org.knime.chem.types.SdfValue;
import org.knime.chem.types.SmilesCell;
import org.knime.chem.types.SmilesValue;
import org.knime.core.data.DataCell;
import org.knime.core.data.DataColumnSpec;
import org.knime.core.data.DataColumnSpecCreator;
import org.knime.core.data.DataRow;
import org.knime.core.data.DataTableSpec;
import org.knime.core.data.DataType;
import org.knime.core.data.RowKey;
import org.knime.core.data.def.DefaultRow;
import org.knime.core.data.def.DoubleCell;
import org.knime.core.data.def.IntCell;
import org.knime.core.data.def.StringCell;
import org.knime.core.node.BufferedDataContainer;
import org.knime.core.node.BufferedDataTable;
import org.knime.core.node.CanceledExecutionException;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelDoubleBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelString;
import org.knime.core.node.ExecutionContext;
import org.knime.core.node.ExecutionMonitor;
import org.knime.core.node.InvalidSettingsException;
import org.knime.core.node.NodeLogger;
import org.knime.core.node.NodeModel;
import org.knime.core.node.NodeSettingsRO;
import org.knime.core.node.NodeSettingsWO;
import org.knime.core.util.MutableBoolean;


 
/**
 * This is the model implementation of FindSites.
 * Find docking sites for a protein
 *
 * @author Treweren Consultants
 */
public class FindSitesNodeModel extends NodeModel {
    
    // the logger instance
    private static final NodeLogger logger = NodeLogger
            .getLogger(FindSitesNodeModel.class);
        
    /** the settings key which is used to retrieve and 
        store the settings (from the dialog or from a settings file)    
       (package visibility to be usable from the dialog). */
	static final String CFGKEY_SEARCH = "Search";
	static final String CFGKEY_LIGANDS = "Ligands";
	static final String CFGKEY_PDB = "PDB";
	static final String CFGKEY_RADIUS = "Radius";
	static final String CFGKEY_TOLERANCE = "Tolerance";
	static final String CFGKEY_SITES = "Sites";
	static final String CFGKEY_LIPOPHILE = "Lipophile";
	static final String CFGKEY_REQUIRE = "Require";
	static final String CFGKEY_CENTRE1 = "Centre1";
	static final String CFGKEY_CENTRE2 = "Centre2";
	static final String CFGKEY_CENTRE3 = "Centre3";
	static final String CFGKEY_CENTRE4 = "Centre4";
	static final String CFGKEY_WORKING = "Working";
    private static final int CANCEL_CHECK_INTERVAL = 1000;

	/** initial default count value. */
	static final String DEFAULT_SEARCH = "";
	static final boolean DEFAULT_LIGANDS = true;
	static final boolean DEFAULT_PDB = true;
	static final boolean DEFAULT_SITES = false;
	static final boolean DEFAULT_LIPOPHILE = false;
	static final boolean DEFAULT_REQUIRE = false;
	static final String DEFAULT_CENTRE1 = "";
	static final String DEFAULT_CENTRE2 = "";
	static final String DEFAULT_CENTRE3 = "";
	static final String DEFAULT_CENTRE4 = "";
	static final double DEFAULT_RADIUS = 1.25;	
	static final double DEFAULT_TOLERANCE = 0.5;
	static final String DEFAULT_WORKING = System.getenv("THINK_WORKING");

	// example value: the models count variable filled from the dialog 
	// and used in the models execution method. The default components of the
	// dialog work with "SettingsModels".
	private final SettingsModelString m_search =
		new SettingsModelString(FindSitesNodeModel.CFGKEY_SEARCH,
                    FindSitesNodeModel.DEFAULT_SEARCH);
	private final SettingsModelBoolean m_ligands =
		new SettingsModelBoolean(FindSitesNodeModel.CFGKEY_LIGANDS,
                    FindSitesNodeModel.DEFAULT_LIGANDS );
	private final SettingsModelBoolean m_pdb =
		new SettingsModelBoolean(FindSitesNodeModel.CFGKEY_PDB,
                    FindSitesNodeModel.DEFAULT_PDB );
	private final SettingsModelBoolean m_findsites =
		new SettingsModelBoolean(FindSitesNodeModel.CFGKEY_SITES,
                    FindSitesNodeModel.DEFAULT_SITES );

	private final SettingsModelBoolean m_lipophile =
			new SettingsModelBoolean(FindSitesNodeModel.CFGKEY_LIPOPHILE,
	                    FindSitesNodeModel.DEFAULT_LIPOPHILE );
	
	private final SettingsModelDoubleBounded m_radius =
		new SettingsModelDoubleBounded(FindSitesNodeModel.CFGKEY_RADIUS,
                    FindSitesNodeModel.DEFAULT_RADIUS,
                    0.6, 2.5);
	private final SettingsModelDoubleBounded m_tolerance =
		new SettingsModelDoubleBounded(FindSitesNodeModel.CFGKEY_TOLERANCE,
                    FindSitesNodeModel.DEFAULT_TOLERANCE,
                    0., 5.);
	private final SettingsModelBoolean m_require =
		new SettingsModelBoolean(FindSitesNodeModel.CFGKEY_REQUIRE,
                    FindSitesNodeModel.DEFAULT_REQUIRE );
	private final SettingsModelString m_centre1 =
		new SettingsModelString(FindSitesNodeModel.CFGKEY_CENTRE1,
                    FindSitesNodeModel.DEFAULT_CENTRE1);
	private final SettingsModelString m_centre2 =
		new SettingsModelString(FindSitesNodeModel.CFGKEY_CENTRE2,
                    FindSitesNodeModel.DEFAULT_CENTRE2);
	private final SettingsModelString m_centre3 =
		new SettingsModelString(FindSitesNodeModel.CFGKEY_CENTRE3,
                    FindSitesNodeModel.DEFAULT_CENTRE3);
	private final SettingsModelString m_centre4 =
		new SettingsModelString(FindSitesNodeModel.CFGKEY_CENTRE4,
                    FindSitesNodeModel.DEFAULT_CENTRE4);
	private final SettingsModelString m_working =
		new SettingsModelString(FindSitesNodeModel.CFGKEY_WORKING,
                    FindSitesNodeModel.DEFAULT_WORKING);
	

    /**
     * Constructor for the node model.
     */
    protected FindSitesNodeModel() {
    
        // One incoming port and two outgoing ports
        super(0, 3);
    }

    /**
     * @see org.knime.core.node.NodeModel #execute(BufferedDataTable[],
     *      ExecutionContext)
     */
    @Override
    protected BufferedDataTable[] execute(final BufferedDataTable[] inData,
            final ExecutionContext exec) throws Exception {

        notifyViews(null);

        final String cwdString = m_working.getStringValue();
        // clean up 
        File outputReport = new File(cwdString, "output.log"); 
        outputReport.delete();
        File outputFile = new File(cwdString, "sites.csv"); 
        outputFile.delete();
        File outputInteract = new File(cwdString, "interact.csv"); 
        outputInteract.delete();
 
        // execute THINK
        File cwdFile = new File (cwdString);
        int exitVal;
        try {
            exec.setProgress("Starting THINK");
            Runtime rt = Runtime.getRuntime();
            // prepare the command string
            String inFile = m_search.getStringValue();
            String s_ligands = "NO";
            if ( m_ligands.getBooleanValue() ) {
            	s_ligands ="LIGANDS";
            }
            String s_pdb = "NO";
            if ( m_pdb.getBooleanValue() ) {
            	s_pdb ="PDB";
            }
            String s_findsites = "NO";
            if ( m_findsites.getBooleanValue() ) {
            	s_findsites ="FIND";
            }
            double d_radius = m_radius.getDoubleValue();
            double d_tolerance = m_tolerance.getDoubleValue();
            String s_require = "NO";
            if ( m_require.getBooleanValue() ) {
            	s_require ="REQUIRE";
            }
            String s_lipophile = "NOLIP";
            if ( m_lipophile.getBooleanValue())
            	s_lipophile = "LIP";
            String s_centre1 = m_centre1.getStringValue();
            String s_centre2 = m_centre2.getStringValue();
            String s_centre3 = m_centre3.getStringValue();
            String s_centre4 = m_centre4.getStringValue();
            
            String cmdString = System.getenv("THINK_EXEC") + "think THINK_EXEC:sites.log output.log \"" + inFile + "\" " + s_ligands + " " + s_pdb + " " + s_findsites + " " + d_radius + " " + d_tolerance + " "+ s_require  + " " + s_lipophile + " " + s_centre1+ " " + s_centre2+ " " + s_centre3+ " " + s_centre4;
            // Go!
            logger.info("THINK command line: '" + cmdString + "'");
            final Process proc = rt.exec(cmdString, null, cwdFile);

            final MutableBoolean procDone = new MutableBoolean(false);
            new Thread(new Runnable() {
                public void run() {
                    synchronized (procDone) {
                        while (!procDone.booleanValue()) {
                            try {
                                exec.checkCanceled();
                            } catch (CanceledExecutionException cee) {
                                // blow away the running external process
                                proc.destroy();
                                return;
                            }
                            try {
                               	procDone.wait(CANCEL_CHECK_INTERVAL);
                                // Read progress file and set progress value       
                               	double d= 0.;
                                File progressReport = new File(cwdString, "progress0.dat");
                                if ( progressReport.exists() && progressReport.isFile() ) {
                               		if ( progressReport.length() > 0 ) {
                               			BufferedReader in = new BufferedReader(new FileReader(progressReport));
                               			String line;
                               			if ( (line = in.readLine()) != null) {
                               			  d = Double.valueOf(line).doubleValue();
                               			}               
                              			if ( (line = in.readLine()) == null) {
                              				line = "Running THINK";
                              			}
                               			in.close();
                                        exec.setProgress( d, line.toString());
                               		}
                                }
                            } catch (InterruptedException e) {
                                // do nothing
                            } catch ( FileNotFoundException e) {
                                // do nothing
                            } catch ( IOException e) {
                                // do nothing
                            }
                        }
                    }
                }

            }).start();
           

            // wait until the external process finishes.
            exitVal = proc.waitFor();

            synchronized (procDone) {
                // this should terminate the check cancel thread
                procDone.setValue(true);
            }

            exec.checkCanceled();

            exec.setProgress("Wrapping up");
            logger.info("THINK terminated with exit code: " + exitVal);
        } catch (Throwable t) {
        	warningReport();
        	logger.error("THINK failed (with exception)", t);
            throw new Exception(t);
        }

        if (exitVal != 0) {
            // before we return barfing, we save the output in the failing list
            warningReport();
        	throw new IllegalStateException("THINK failed (error code "
                    + exitVal + ")");
        }

        exec.setProgress("Creating KNIME tables");
        BufferedDataTable outTable = readSitesFile(exec);
        BufferedDataTable sumTable = readInteractFile(exec);
        BufferedDataTable ligandTable = readSDFile(exec);
        
        return new BufferedDataTable[]{outTable, sumTable, ligandTable};

    }

    private void warningReport( ) throws Exception {

    	StringBuilder warningMessage = new StringBuilder();
        String cwdString = m_working.getStringValue();
        File outputReport = new File(cwdString, "output.log"); 

        if ( outputReport.exists() && outputReport.isFile() ) {
       		if ( outputReport.length() > 0 ) {
       			BufferedReader in = new BufferedReader(new FileReader(outputReport));
       			String line;
       			while ((line = in.readLine()) != null) {
       				warningMessage.append( line + "\n");
       			}               
       			in.close();
       		}
       		else { 
                warningMessage.append( "Report from THINK is blank");      	     			
       		}
        }
       	else {
            warningMessage.append( "No report from THINK");      	
       	}	
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
   }

    private BufferedDataTable readInteractFile(final ExecutionContext exec) throws Exception {

    	String cwdString = m_working.getStringValue();
        File outFile = new File (cwdString, "interact.csv");

    	if (!(outFile.exists() && outFile.isFile())) {
    		warningReport();
    		throw new IllegalStateException("THINK didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
        HashSet<String> titles = new HashSet<String>();
        String title;
        String line;
        String buf;
        boolean headerRead = false;
        DataColumnSpec[] allColSpecs;
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        DataTableSpec outputSpec;
        BufferedDataContainer cont =null;
        RowKey key =null;
        int count =0;
        int ilast=0;
        int rowcount =0;
        char chr='0';
        while ((line = in.readLine()) != null) {
        	exec.checkCanceled();
// Extract column headers
            if ( !headerRead ) {
                headerRead = true;
 // Create column definitions
                allColSpecs = new DataColumnSpec[2];
                allColSpecs[0] = 
                    new DataColumnSpecCreator("Interactions",StringCell.TYPE).createSpec();
                allColSpecs[1] = 
                    new DataColumnSpecCreator("Count",IntCell.TYPE).createSpec();
                outputSpec = new DataTableSpec(allColSpecs);
                cont = exec.createDataContainer(outputSpec);
            }
// Load data
            else if ( !line.equalsIgnoreCase("Ligand,Interactions,Count")) {
                DataCell[] cells = new DataCell[2];
                rowcount++;
                count =0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else
                        chr = ',';
                    if ( chr == ',' ) { 
                        if ( count == 0 ) {
                            title = line.substring(0,i);
                            if (title.length() == 0) {
                                title = "Mol " + rowcount;
                            }
                            if (titles.contains(title)) {
                                title = title + "-" + rowcount;
                            }
                            titles.add(title);
                            key = new RowKey(title);                           
                        }
                        else if ( count == 1 ){
                        	buf = line.substring(ilast,i);
                            cells[0] = new StringCell(buf);
                            
                        }
                        else if ( count == 2 ){
                            try {
                                int n_value = Integer.valueOf(line.substring(ilast,i)).intValue();
                                cells[1] = new IntCell(n_value);
                            } catch (Throwable t) {
                                cells[1] = DataType.getMissingCell();
                            }
                        }
                        count ++;
                        ilast = i + 1;
                    }
                }
                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
             }
        }

        cont.close();
        in.close();

        return cont.getTable();
 
    }
    private BufferedDataTable readSitesFile(final ExecutionContext exec) throws Exception {

    	String cwdString = m_working.getStringValue();
        File outFile = new File (cwdString, "sites.csv");

    	if (!(outFile.exists() && outFile.isFile())) {
    		warningReport();
    		throw new IllegalStateException("THINK didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
        HashSet<String> titles = new HashSet<String>();
        String title;
        String line;
        String buf;
        boolean headerRead = false;
        DataColumnSpec[] allColSpecs;
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        DataTableSpec outputSpec;
        BufferedDataContainer cont =null;
        RowKey key =null;
        int count =0;
        int ilast=0;
        int rowcount =0;
        char chr='0';
        while ((line = in.readLine()) != null) {
        	exec.checkCanceled();
// Extract column headers
            if ( !headerRead ) {
                headerRead = true;
 // Create column definitions
                allColSpecs = new DataColumnSpec[3];
                allColSpecs[0] = 
                    new DataColumnSpecCreator("Interactions",IntCell.TYPE).createSpec();
                allColSpecs[1] = 
                    new DataColumnSpecCreator("Residues",IntCell.TYPE).createSpec();
                allColSpecs[2] = 
                    new DataColumnSpecCreator("File",StringCell.TYPE).createSpec();
                outputSpec = new DataTableSpec(allColSpecs);
                cont = exec.createDataContainer(outputSpec);
            }
// Load data
            else if ( !line.equalsIgnoreCase("Site,Interactions,Residues,File")) {
                DataCell[] cells = new DataCell[3];
                rowcount++;
                count =0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else
                        chr = ',';
                    if ( chr == ',' ) { 
                        if ( count == 0 ) {
                            title = line.substring(0,i);
                            if (title.length() == 0) {
                                title = "Site " + rowcount;
                            }
                            if (titles.contains(title)) {
                                title = title + "-" + rowcount;
                            }
                            titles.add(title);
                            key = new RowKey(title);                           
                        }
                        else if ( count == 3 ){
                        	buf = line.substring(ilast,i);
                            cells[2] = new StringCell(buf);
                            
                        }
                        else {
                            try {
                                int n_value = Integer.valueOf(line.substring(ilast,i)).intValue();
                                cells[count-1] = new IntCell(n_value);
                            } catch (Throwable t) {
                                cells[count-1] = DataType.getMissingCell();
                            }
                        }
                        count ++;
                        ilast = i + 1;
                    }
                }
                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
             }
        }

        cont.close();
        in.close();

        return cont.getTable();
 
    }

    private BufferedDataTable readSDFile(final ExecutionContext exec) throws Exception {

    	String cwdString = m_working.getStringValue();
        String line;
        String fieldName;
        String doubleFields[] = new String[100];
        String stringFields[]= new String[100];
        int smileCount=0;
        int keyCount=0;
        int doubleCount=0;
        int stringCount=0;
        File outFile = new File (cwdString, "ligands.sdf");
       	if (!(outFile.exists() && outFile.isFile())) {
       		DataColumnSpec[] allColSpecs = new DataColumnSpec[1];
            allColSpecs[0] = 
                new DataColumnSpecCreator("Molecule",SdfCell.TYPE).createSpec();
            DataTableSpec outputSpec = new DataTableSpec(allColSpecs);
            BufferedDataContainer cont = exec.createDataContainer(outputSpec);
            cont.close();
            return cont.getTable();

        }

     	// Scan file for fields and data types
    	
        BufferedReader looksee = new BufferedReader(new FileReader(outFile));
        while ((line = looksee.readLine()) != null) {
            if (line.startsWith("$$$$")) 
            	break;
            else if (line.startsWith("> <")) {
            	int l = line.length() - 1;
            	fieldName = line.substring(3,l);
            	line = looksee.readLine();
            	if ( fieldName.equals("SMILES")) {
            		smileCount = 1;
            	}
            	else if ( fieldName.equals("KEYS")) {
            		keyCount = 1;
            	}
            	else {
            		try {
            			double d_value = Double.valueOf(line).doubleValue();
            			doubleFields[doubleCount] = fieldName;
            			doubleCount ++;
            		}
            		catch (Throwable t){
            			stringFields[stringCount] = fieldName;
            			stringCount++;
            		}
            	}
            }	
        }	        
        looksee.close();
        
        // Create the table
        DataColumnSpec[] allColSpecs;
        allColSpecs = new DataColumnSpec[stringCount+doubleCount+smileCount+keyCount+1];
        int count =0;
        int keyStart=0;
        int doubleStart=0;
        int stringStart=0;
        allColSpecs[0] = 
            new DataColumnSpecCreator("Molecule",SdfCell.TYPE).createSpec();
        if ( smileCount == 1 ) {
        	count ++;
            allColSpecs[count] = 
            	new DataColumnSpecCreator("SMILES",SmilesCell.TYPE).createSpec();
        }
        if ( keyCount == 1 ) {
        	keyCount = count + 1;
        	count ++;
            allColSpecs[count] = 
            	new DataColumnSpecCreator("KEYS",StringCell.TYPE).createSpec();
        }
        if ( stringCount > 0 ) {
        	stringStart=count + 1;
        	for (int i=0; i<stringCount; i++ ) {
        		count ++;
                allColSpecs[count] = 
        		new DataColumnSpecCreator(stringFields[i],StringCell.TYPE).createSpec();
        	}
        }
        if ( doubleCount > 0 ) {
           	doubleStart=count + 1;      	
        	for (int i=0; i<doubleCount; i++ ) {
        		count ++;
                allColSpecs[count] = 
        		new DataColumnSpecCreator(doubleFields[i],DoubleCell.TYPE).createSpec();
        	}
        }
        DataTableSpec outputSpec = new DataTableSpec(allColSpecs);
        BufferedDataContainer cont = exec.createDataContainer(outputSpec);
                
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        
        HashSet<String> titles = new HashSet<String>();

        StringBuilder buf = new StringBuilder(4096);
        boolean titleRead = false;
        String title = "";
    		
        DataCell[] cells = new DataCell[count+1];
        int molcount=0;
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();

            if (!titleRead) {
                title = line;
                if (title.length() == 0) {
                    title = "Mol " + molcount;
                }
                if (titles.contains(title)) {
                    title = title + "-" + molcount;
                }
                titles.add(title);
                titleRead = true;
            }
            buf.append(line).append('\n');

            if (line.startsWith("$$$$")) {
            	molcount ++;
                cells[0] = new SdfCell(buf.toString());
                RowKey key = new RowKey(title);

                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
                buf.delete(0, buf.length());
                titleRead = false;
            }
            else if ( line.startsWith("> <")) {
            	int l = line.length() - 1;
            	fieldName = line.substring(3,l);
            	line = in.readLine();
                buf.append(line).append('\n');
            	if ( smileCount == 1 & fieldName.equals("SMILES")) {
            		cells[1] = new SmilesCell(line);
            	}
            	else if ( keyCount == 1 & fieldName.equals("KEYS")) {
                    cells[keyStart]= new StringCell(line);
            	}
            	else {
            		for (int i=0; i<doubleCount; i++ ) {
            			if ( doubleFields[i].equals(fieldName)) {
            				try {
            				double d_value = Double.valueOf(line).doubleValue();
            				cells[doubleStart+i]= new DoubleCell(d_value);
            				} catch (Throwable t) {
            					cells[doubleStart+i] = DataType.getMissingCell();
            				}
            				break;
            			}
            		}
            		for (int i=0; i<stringCount; i++ ) {
            			if ( stringFields[i].equals(fieldName)) {
            				cells[stringStart+i]= new StringCell(line);
            				break;
            			}
            		}
            	}
            }
        }
        cont.close();
        in.close();
 
        return cont.getTable();
 
  
    }

    /**
     * @see org.knime.core.node.NodeModel#reset()
     */
    @Override
    protected void reset() {
        // TODO Code executed on reset.
        // Models build during execute are cleared here.
        // Also data handled in load/saveInternals will be erased here.
    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #configure(org.knime.core.data.DataTableSpec[])
     */
    @Override
    protected DataTableSpec[] configure(final DataTableSpec[] inSpecs)
            throws InvalidSettingsException {
        
        // check if user settings are available, fit to the incoming
        // table structure, and the incoming types are feasible for the node
        // to execute. If the node can execute in its current state return
        // the spec of its output data table(s) (if you can, otherwise an array
        // with null elements), or throw an exception with a useful user message

    	if ( m_search.getStringValue().equals("") ) {
            throw new InvalidSettingsException(
            "PDB file not specified");       		
    	}
        if ( System.getenv("THINK_EXEC") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_EXEC is unset - It should be set to the folder containing the THINK software.");       	  		
    	}
        if ( System.getenv("THINK_WORKING") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_WORKING is not set to the folder in which working files will be created.");       	  		
    	}

        return new DataTableSpec[]{null, null, null};
    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #saveSettingsTo(org.knime.core.node.NodeSettings)
     */
    @Override
    protected void saveSettingsTo(final NodeSettingsWO settings) {

        // TODO save user settings to the config object.
		
		m_search.saveSettingsTo(settings);
		m_ligands.saveSettingsTo(settings);
		m_pdb.saveSettingsTo(settings);
		m_findsites.saveSettingsTo(settings);
		m_lipophile.saveSettingsTo(settings);
		m_radius.saveSettingsTo(settings);
		m_tolerance.saveSettingsTo(settings);
		m_require.saveSettingsTo(settings);
		m_centre1.saveSettingsTo(settings);
		m_centre2.saveSettingsTo(settings);
		m_centre3.saveSettingsTo(settings);
		m_centre4.saveSettingsTo(settings);
		m_working.saveSettingsTo(settings);

    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #loadValidatedSettingsFrom(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void loadValidatedSettingsFrom(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO load (valid) settings from the config object.
        // It can be safely assumed that the settings are valided by the 
        // method below.
        
        m_search.loadSettingsFrom(settings);
        m_ligands.loadSettingsFrom(settings);
        m_pdb.loadSettingsFrom(settings);
        m_radius.loadSettingsFrom(settings);
        m_tolerance.loadSettingsFrom(settings);
        m_findsites.loadSettingsFrom(settings);
        m_lipophile.loadSettingsFrom(settings);
        m_require.loadSettingsFrom(settings);
        m_centre1.loadSettingsFrom(settings);
        m_centre2.loadSettingsFrom(settings);
        m_centre3.loadSettingsFrom(settings);
        m_centre4.loadSettingsFrom(settings);
        m_working.loadSettingsFrom(settings);

    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #validateSettings(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void validateSettings(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO check if the settings could be applied to our model
        // e.g. if the count is in a certain range (which is ensured by the
        // SettingsModel).
        // Do not actually set any values of any member variables.

        m_search.validateSettings(settings);
        m_ligands.validateSettings(settings);
        m_pdb.validateSettings(settings);
        m_findsites.validateSettings(settings);
        m_lipophile.validateSettings(settings);
        m_radius.validateSettings(settings);
        m_tolerance.validateSettings(settings);
        m_require.validateSettings(settings);
        m_centre1.validateSettings(settings);
        m_centre2.validateSettings(settings);
        m_centre3.validateSettings(settings);
        m_centre4.validateSettings(settings);
        m_working.validateSettings(settings);

    }
    
    /**
     * @see org.knime.core.node.NodeModel #loadInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void loadInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
        
		// TODO load internal data. 
		// Everything handed to output ports is loaded automatically (data
		// returned by the execute method, models loaded in loadModelContent,
		// and user settings set through loadSettingsFrom - is all taken care 
		// of). Load here only the other internals that need to be restored
		// (e.g. data used by the views).

    }
    
    /**
     * @see org.knime.core.node.NodeModel #saveInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void saveInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
       
       	// TODO save internal models. 
		// Everything written to output ports is saved automatically (data
		// returned by the execute method, models saved in the saveModelContent,
		// and user settings saved through saveSettingsTo - is all taken care 
		// of). Save here only the other internals that need to be preserved
		// (e.g. data used by the views).

    }

}
