package com.treweren.denovo;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;

import org.knime.base.node.io.filereader.FileAnalyzer;
import org.knime.base.node.io.filereader.FileReaderNodeSettings;
import org.knime.base.node.io.filereader.FileReaderSettings;
import org.knime.base.node.io.filereader.FileTable;
import org.knime.base.node.io.filereader.SettingsStatus;
import org.knime.chem.types.SdfCell;
import org.knime.chem.types.SdfValue;
import org.knime.chem.types.SmilesCell;
import org.knime.chem.types.SmilesValue;
import org.knime.core.data.DataCell;
import org.knime.core.data.DataColumnSpec;
import org.knime.core.data.DataColumnSpecCreator;
import org.knime.core.data.DataRow;
import org.knime.core.data.DataType;
import org.knime.core.data.DataTableSpec;
import org.knime.core.data.def.DefaultRow;
import org.knime.core.data.def.DoubleCell;
import org.knime.core.data.def.IntCell;
import org.knime.core.data.def.StringCell;
import org.knime.core.data.RowKey;
import org.knime.core.node.BufferedDataContainer;
import org.knime.core.node.BufferedDataTable;
import org.knime.core.node.CanceledExecutionException;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelDouble;
import org.knime.core.node.defaultnodesettings.SettingsModelDoubleRange;
import org.knime.core.node.defaultnodesettings.SettingsModelInteger;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelString;
import org.knime.core.node.port.PortType;
import org.knime.core.node.ExecutionContext;
import org.knime.core.node.ExecutionMonitor;
import org.knime.core.node.InvalidSettingsException;
import org.knime.core.node.NodeLogger;
import org.knime.core.node.NodeModel;
import org.knime.core.node.NodeSettingsRO;
import org.knime.core.node.NodeSettingsWO;
import org.knime.core.util.MutableBoolean;







/**
 * This is the model implementation of DeNovo.
 * Generates de novo derivatives
 *
 * @author Treweren Consultants
 */
public class DeNovoNodeModel extends NodeModel {
    
    // the logger instance
    private static final NodeLogger logger = NodeLogger
            .getLogger(DeNovoNodeModel.class);
    private static int n_Column = 0;
    private static String m_sdfColumn = null;
    private static String m_smilesColumn = null;
    private static String m_stringColumn = null;
    private static final int CANCEL_CHECK_INTERVAL = 1000;
       
    /** the settings key which is used to retrieve and 
        store the settings (from the dialog or from a settings file)    
       (package visibility to be usable from the dialog). */
	static final String CFGKEY_MODE = "Mode";
	static final String CFGKEY_COUNT = "Count";
	static final String CFGKEY_ATTEMPTS = "Attempts";
	static final String CFGKEY_CURB = "Curb";
	static final String CFGKEY_GROUP = "Group";
	static final String CFGKEY_SMILES = "Smiles";
	static final String CFGKEY_ROW1 = "Row1";
	static final String CFGKEY_ROW = "Row";
	static final String CFGKEY_TRANSFORM = "Transform";
	static final String CFGKEY_WORKING = "Working";
 	static final String CFGKEY_CENTRES = "Centres";
 	static final String CFGKEY_MASS = "Mass";
	static final String CFGKEY_XSA = "XSA";
	static final String CFGKEY_ROTBONDS = "Rot-Bonds";
	static final String CFGKEY_GROUP1 = "Group 1";
	static final String CFGKEY_GROUP2 = "Group 2";
	static final String CFGKEY_GROUP3 = "Group 3";
	static final String CFGKEY_GROUP4 = "Group 4";

	/** default values */
	static final String DEFAULT_MODE = "2D";
	static final int DEFAULT_COUNT = 100;
	static final int DEFAULT_ATTEMPTS = 1000;
	static final int DEFAULT_CURB = 2;
	static final String DEFAULT_GROUP = "None";
	static final Boolean DEFAULT_SMILES = false;
	static final Boolean DEFAULT_ROW1 = true;
	static final int DEFAULT_ROW = 1;
	static final String DEFAULT_TRANSFORM="THINK_EXEC:tranform.smi";
	static final String DEFAULT_WORKING = System.getenv("THINK_WORKING");
	static final int DEFAULT_CENTRE1 = 2;
	static final int DEFAULT_CENTRE2 = 9;
	static final double DEFAULT_MASS1 = 150.;
	static final double DEFAULT_MASS2 = 800.;
	static final double DEFAULT_XSA1 = 20.;
	static final double DEFAULT_XSA2 = 240.;
	static final int DEFAULT_ROTBONDS1 = 0;
	static final int DEFAULT_ROTBONDS2 = 10;
	static final String DEFAULT_GROUP1 = "";
	static final String DEFAULT_GROUP2 = "";
	static final String DEFAULT_GROUP3 = "";
	static final String DEFAULT_GROUP4 = "";

	
	// example value: the models count variable filled from the dialog 
	// and used in the models execution method. The default components of the
	// dialog work with "SettingsModels".

	private final SettingsModelString m_mode =
		new SettingsModelString(DeNovoNodeModel.CFGKEY_MODE,
                    DeNovoNodeModel.DEFAULT_MODE);
	
	private final SettingsModelIntegerBounded m_count =
		new SettingsModelIntegerBounded(DeNovoNodeModel.CFGKEY_COUNT,
                    DeNovoNodeModel.DEFAULT_COUNT,
                    Integer.MIN_VALUE, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_attempts =
		new SettingsModelIntegerBounded(DeNovoNodeModel.CFGKEY_ATTEMPTS,
                    DeNovoNodeModel.DEFAULT_ATTEMPTS,
                    Integer.MIN_VALUE, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_curb =
		new SettingsModelIntegerBounded(DeNovoNodeModel.CFGKEY_CURB,
                    DeNovoNodeModel.DEFAULT_CURB,
                    Integer.MIN_VALUE, Integer.MAX_VALUE);

	private final SettingsModelString m_group =
		new SettingsModelString(DeNovoNodeModel.CFGKEY_GROUP,
                    DeNovoNodeModel.DEFAULT_GROUP);
	
	private final SettingsModelDoubleRange m_centres =
		new SettingsModelDoubleRange(DeNovoNodeModel.CFGKEY_CENTRES, DeNovoNodeModel.DEFAULT_CENTRE1, DeNovoNodeModel.DEFAULT_CENTRE2 );


	private final SettingsModelDoubleRange m_mass =
		new SettingsModelDoubleRange(DeNovoNodeModel.CFGKEY_MASS, DeNovoNodeModel.DEFAULT_MASS1, DeNovoNodeModel.DEFAULT_MASS2   );

	private final SettingsModelDoubleRange m_xsa =
		new SettingsModelDoubleRange(DeNovoNodeModel.CFGKEY_XSA, DeNovoNodeModel.DEFAULT_XSA1, DeNovoNodeModel.DEFAULT_XSA2 );

	private final SettingsModelDoubleRange m_rotbonds =
		new SettingsModelDoubleRange(DeNovoNodeModel.CFGKEY_ROTBONDS, DeNovoNodeModel.DEFAULT_ROTBONDS1, DeNovoNodeModel.DEFAULT_ROTBONDS2 );

	private final SettingsModelBoolean m_smiles =
		new SettingsModelBoolean(DeNovoNodeModel.CFGKEY_SMILES,
                    DeNovoNodeModel.DEFAULT_SMILES );

	private final SettingsModelBoolean m_row1 =
		new SettingsModelBoolean(DeNovoNodeModel.CFGKEY_ROW1,
                    DeNovoNodeModel.DEFAULT_ROW1 );

	private final SettingsModelIntegerBounded m_row =
			new SettingsModelIntegerBounded(DeNovoNodeModel.CFGKEY_ROW,
	                    DeNovoNodeModel.DEFAULT_ROW,
	                    1, Integer.MAX_VALUE);

	private final SettingsModelString m_transform =
		new SettingsModelString(DeNovoNodeModel.CFGKEY_TRANSFORM,
                    DeNovoNodeModel.DEFAULT_TRANSFORM);
	
	private final SettingsModelString m_working =
		new SettingsModelString(DeNovoNodeModel.CFGKEY_WORKING, DeNovoNodeModel.DEFAULT_WORKING );

	private final SettingsModelString m_group1 =
			new SettingsModelString(DeNovoNodeModel.CFGKEY_GROUP1,
	                    DeNovoNodeModel.DEFAULT_GROUP1);
	private final SettingsModelString m_group2 =
			new SettingsModelString(DeNovoNodeModel.CFGKEY_GROUP2,
	                    DeNovoNodeModel.DEFAULT_GROUP2);
	private final SettingsModelString m_group3 =
			new SettingsModelString(DeNovoNodeModel.CFGKEY_GROUP3,
	                    DeNovoNodeModel.DEFAULT_GROUP3);
	private final SettingsModelString m_group4 =
			new SettingsModelString(DeNovoNodeModel.CFGKEY_GROUP4,
	                    DeNovoNodeModel.DEFAULT_GROUP4);
	

    /**
     * Constructor for the node model.
     */
    public static final PortType OPTIONAL_PORT_TYPE = new PortType(BufferedDataTable.class, true);

    protected DeNovoNodeModel() {
    
        // Two incoming ports (second optional) and one outgoing port
        super(createOPOs(2, 2), createOPOs(1));
    }
	private static PortType[] createOPOs(final int nrDataPorts, final int... optionalPortsIds)
   	{
   	    PortType[] portTypes = new PortType[nrDataPorts];
   	    Arrays.fill(portTypes, BufferedDataTable.TYPE);        
    	    if (optionalPortsIds.length > 0) {
   	        for (int portId : optionalPortsIds) {
   	            if ((portId - 1) < nrDataPorts) {
   	                portTypes[portId - 1] = OPTIONAL_PORT_TYPE;
   	            }
   	        }
   	    }
   	    return portTypes;
   	}
    @Override
    protected DataTableSpec[] configure(final DataTableSpec[] inSpecs)
            throws InvalidSettingsException {
        if ( System.getenv("THINK_EXEC") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_EXEC is unset - It should be set to the folder containing the THINK software.");       	  		
    	}
        if ( System.getenv("THINK_WORKING") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_WORKING is not set to the folder in which working files will be created.");       	  		
    	}
        DataTableSpec inpdb = inSpecs[1];
        if ( inpdb != null ) {
            if ( inpdb.getNumColumns() < 3 ) {
                throw new InvalidSettingsException(
                "Site input table does not have 3 columns");       	           	
            }
        	if ( !inpdb.getColumnSpec(2).getName().equals("File")) { 
                throw new InvalidSettingsException(
                "Site input table does not have third column named 'File' ");       	
        	}
        }
        DataTableSpec in = inSpecs[0];
        n_Column = columnSelect (in);
        return new DataTableSpec[] {null};
    }
    protected int columnSelect ( final DataTableSpec in ) throws  InvalidSettingsException {

        StringBuilder warningMessage = new StringBuilder();
        int n_column=0;
        int sdfColCount = 0;
        int n_sdf=0;
        int smilesColCount = 0;
        int n_smiles=0;
        int stringColCount = 0;
        int n_string=0;
        for (int i = 0; i < in.getNumColumns(); i++) {
            DataColumnSpec s = in.getColumnSpec(i);
            if (s.getType().isCompatible(SdfValue.class)) {
                if (m_sdfColumn == null) {
                    m_sdfColumn = in.getColumnSpec(i).getName();
                    n_sdf = i;
                }
                sdfColCount++;
            }
            else if ( s.getType().isCompatible(SmilesValue.class)) {
            	if (m_smilesColumn == null ) {
             		m_smilesColumn = in.getColumnSpec(i).getName();
             		n_smiles=i;
              	}
               	smilesColCount++;
            }
            else if ( in.getColumnSpec(i).getName().equalsIgnoreCase("smiles"))  {
            	if (m_stringColumn == null ) {
             		m_stringColumn = in.getColumnSpec(i).getName();
             		n_string=i;
              	}
               	stringColCount++;
            }
        }
        if ( smilesColCount == 0 & sdfColCount == 0 & stringColCount == 0) {
            throw new InvalidSettingsException(
                    "No column with Smiles or SD file compatible type in molecule input");       	
        }
        else if ( ( m_smiles.getBooleanValue() | sdfColCount == 0 ) & smilesColCount > 0 ) {
        	m_sdfColumn = null;
        	m_stringColumn = null;
        	n_column = n_smiles;
        	if (smilesColCount > 1) {
                warningMessage.append("More than one Smiles compatible column in " 
                        + "molecule input, using column \"" + m_smilesColumn + "\".");
        	}
        }
        else if ( sdfColCount > 0 ) {
           	m_smilesColumn = null;
        	m_stringColumn = null;
        	n_column = n_sdf;
        	if (sdfColCount > 1) {
                warningMessage.append("More than one SDF compatible column in " 
                        + "molecule input, using column \"" + m_sdfColumn + "\".");
        	}
        }
        else if ( stringColCount > 0 ) {
        	m_smilesColumn = null;
        	m_sdfColumn = null;
        	n_column = n_string;
           	if (stringColCount > 1) {
                warningMessage.append("More than one Smiles string compatible column in " 
                        + "molecule input, using column \"" + m_stringColumn + "\".");
           	}
        }
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
        return n_column;
    }

    /**
     * @see org.knime.core.node.NodeModel #execute(BufferedDataTable[],
     *      ExecutionContext)
     */
    @Override
    protected BufferedDataTable[] execute(final BufferedDataTable[] inData,
            final ExecutionContext exec) throws Exception {

        notifyViews(null);

        final String cwdString =  m_working.getStringValue();
        // clean up 
        File outputReport = new File(cwdString, "output.log"); 
        outputReport.delete();
        File outputFile = new File(cwdString, "novel.sdf"); 
        outputFile.delete();

        BufferedDataTable in = inData[0];
        if ( n_Column == 0 ) {
        	DataTableSpec InSpec = in.getDataTableSpec();
        	n_Column = columnSelect(InSpec);
        }
        
        String pdbString=null;
        BufferedDataTable inpdb = inData[1];
        if ( inpdb != null ){
          final double count = inpdb.getRowCount(); // floating point operations
          if ( count > 1 ) {
              setWarningMessage("Using only using site " + m_row.getIntValue());
          }
          int rowCount=0;
          for (DataRow r : inpdb ) {
              exec.checkCanceled();
              rowCount++;
              DataCell c = r.getCell(2);
              if (!c.isMissing() && rowCount == m_row.getIntValue()){
              	pdbString = c.toString();
                break;
             }
           }

        }

        
       // write out input file
        File cwdFile = new File (cwdString);
               String fileString = null; 
        if ( m_sdfColumn == null ) {
            fileString = "denovo.smiles"; 
        }
        else {
            fileString = "denovo.sdf"; 
        }
        File inFile = new File(cwdString, fileString); 
        BufferedWriter outWriter = new BufferedWriter(new FileWriter(inFile));
        int colIndex;
        if ( m_sdfColumn != null ) {
        	colIndex = in.getDataTableSpec().findColumnIndex(m_sdfColumn);
        }
        else if ( m_smilesColumn != null ) {
        	colIndex = in.getDataTableSpec().findColumnIndex(m_smilesColumn);        
        }
        else {
            colIndex = in.getDataTableSpec().findColumnIndex(m_stringColumn);        
        }
        	  
        final double count = in.getRowCount(); // floating point operations
        if ( count > 1 & m_row1.getBooleanValue() ) {
            setWarningMessage("Using only first row/molecule");
         }
        int i = 0; 
        int missingCount = 0;
        for (DataRow r : in) {
            exec.checkCanceled();
            DataCell c = r.getCell(colIndex);
            if (c.isMissing()) {
                missingCount++;
            } 
            else if ( m_sdfColumn != null ) {
                SdfValue v = (SdfValue)c;
                String toString = v.toString();
                outWriter.write(toString);
                if (!toString.trim().endsWith("$$$$")) {
                    outWriter.newLine();
                    outWriter.append("$$$$");
                    outWriter.newLine();
                }
            }
            else if  ( m_smilesColumn != null ){
                SmilesValue v = (SmilesValue)c;
                String toString = v.toString() + " " + r.getKey();
                outWriter.write(toString);        	
                outWriter.newLine();
            }
            else {
            	String toString = c.toString() + " " + r.getKey();
            	outWriter.write(toString);        	
            	outWriter.newLine();
            }
            i++;
            if ( m_row1.getBooleanValue() ) break;
        }

        outWriter.close();
        if (missingCount > 0) {
            setWarningMessage("Skipped " + missingCount 
                    + " row(s) because of missing values");
        }

        // execute THINK
        int exitVal;
        try {
            exec.setProgress(0., "Starting THINK");
            Runtime rt = Runtime.getRuntime();
            // prepare the command string
            String s_mode = m_mode.getStringValue();
            int n_count = m_count.getIntValue();
            int n_curb = m_curb.getIntValue();
            String s_group = m_group.getStringValue();
            int n_attempts = m_attempts.getIntValue();
            String s_transform = m_transform.getStringValue();
            double n_centre1 = m_centres.getMinRange();
            double n_centre2 = m_centres.getMaxRange();
            double d_mass1 = m_mass.getMinRange();
            double d_mass2 = m_mass.getMaxRange();
            double d_xsa1 = m_xsa.getMinRange();
            double d_xsa2 = m_xsa.getMaxRange();
            double n_rotbonds1 = m_rotbonds.getMinRange();
            double n_rotbonds2 = m_rotbonds.getMaxRange();
            String s_group1 = m_group1.getStringValue();
            String s_group2 = m_group2.getStringValue();
            String s_group3 = m_group3.getStringValue();
            String s_group4 = m_group4.getStringValue();
            
            String cmdString = System.getenv("THINK_EXEC") + "think THINK_EXEC:denovo.log output.log " + fileString + " \"" + s_mode + "\" " + n_count + " " + n_attempts + " " + n_curb + " " + s_group + " \"" + s_transform
            + "\" " + n_centre1 + " " + n_centre2 + " " + d_mass1 + " " + d_mass2 + " " + d_xsa1 + " " + d_xsa2 + " " + n_rotbonds1 + " " + n_rotbonds2 +" \"" + pdbString + "\"" + " \"" + s_group1 + " \""+ " \"" + s_group2 + " \""+ " \"" + s_group3 + " \""+ " \"" + s_group4 + " \"";
            // Go!
            logger.info("THINK command line: '" + cmdString + "'");
            final Process proc = rt.exec(cmdString, null, cwdFile);

            final MutableBoolean procDone = new MutableBoolean(false);
            new Thread(new Runnable() {
                public void run() {
                    synchronized (procDone) {
                        while (!procDone.booleanValue()) {
                            try {
                                exec.checkCanceled();
                            } catch (CanceledExecutionException cee) {
                                // blow away the running external process
                                proc.destroy();
                                return;
                            }
                            try {
                               	procDone.wait(CANCEL_CHECK_INTERVAL);
                                // Read progress file and set progress value       
                               	double d= 0.;
                                File progressReport = new File(cwdString, "progress0.dat");
                                if ( progressReport.exists() && progressReport.isFile() ) {
                               		if ( progressReport.length() > 0 ) {
                               			BufferedReader in = new BufferedReader(new FileReader(progressReport));
                               			String line;
                               			if ( (line = in.readLine()) != null) {
                               			  d = Double.valueOf(line).doubleValue();
                               			}               
                              			if ( (line = in.readLine()) == null) {
                              				line = "Running THINK";
                              			}
                               			in.close();
                                        exec.setProgress( d, line.toString());
                               		}
                                }
                            } catch (InterruptedException e) {
                                // do nothing
                            } catch ( FileNotFoundException e) {
                                // do nothing
                            } catch ( IOException e) {
                                // do nothing
                            }
                        }
                    }
                }

            }).start();
           

            // wait until the external process finishes.
            exitVal = proc.waitFor();

            synchronized (procDone) {
                // this should terminate the check cancel thread
                procDone.setValue(true);
            }

            exec.checkCanceled();

            exec.setProgress("Wrapping up");
            logger.info("THINK terminated with exit code: " + exitVal);
        } catch (Throwable t) {
        	warningReport();
        	logger.error("THINK failed (with exception)", t);
            throw new Exception(t);
        }

        if (exitVal != 0) {
            // before we return barfing, we save the output in the failing list
           	warningReport();
            throw new IllegalStateException("THINK failed (error code "
                    + exitVal + ")");
        }

        exec.setProgress("Creating KNIME tables");
        BufferedDataTable outTable = readSDFile(exec);

        return new BufferedDataTable[]{outTable};

    }

    private void warningReport( ) throws Exception {

    	StringBuilder warningMessage = new StringBuilder();
        String cwdString = m_working.getStringValue();
        File outputReport = new File(cwdString, "output.log"); 

        if ( outputReport.exists() && outputReport.isFile() ) {
       		if ( outputReport.length() > 0 ) {
       			BufferedReader in = new BufferedReader(new FileReader(outputReport));
       			String line;
       			while ((line = in.readLine()) != null) {
       				warningMessage.append( line + "\n");
       			}               
       			in.close();
       		}
       		else { 
                warningMessage.append( "Report from THINK is blank");      	     			
       		}
        }
       	else {
            warningMessage.append( "No report from THINK");      	
       	}	
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
   }
    private BufferedDataTable readSDFile(final ExecutionContext exec) throws Exception {

    	String cwdString = m_working.getStringValue();
        String line;
        String fieldName;
        String doubleFields[] = new String[100];
        String stringFields[]= new String[100];
        int smileCount=0;
        int keyCount=0;
        int doubleCount=0;
        int stringCount=0;
        File outFile = new File (cwdString, "novel.sdf");

    	if (!(outFile.exists() && outFile.isFile())) {
    		warningReport();
    		throw new IllegalStateException("THINK didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
    	// Scan file for fields and data types
    	
        BufferedReader looksee = new BufferedReader(new FileReader(outFile));
        while ((line = looksee.readLine()) != null) {
            if (line.startsWith("$$$$")) 
            	break;
            else if (line.startsWith("> <")) {
            	int l = line.length() - 1;
            	fieldName = line.substring(3,l);
            	line = looksee.readLine();
            	if ( fieldName.equals("SMILES")) {
            		smileCount = 1;
            	}
            	else if ( fieldName.equals("KEYS")) {
            		keyCount = 1;
            	}
            	else {
            		try {
            			double d_value = Double.valueOf(line).doubleValue();
            			doubleFields[doubleCount] = fieldName;
            			doubleCount ++;
            		}
            		catch (Throwable t){
            			stringFields[stringCount] = fieldName;
            			stringCount++;
            		}
            	}
            }	
        }	        
        looksee.close();
        
        // Create the table
        DataColumnSpec[] allColSpecs;
        allColSpecs = new DataColumnSpec[stringCount+doubleCount+smileCount+keyCount+1];
        int count =0;
        int keyStart=0;
        int doubleStart=0;
        int stringStart=0;
        allColSpecs[0] = 
            new DataColumnSpecCreator("Molecule",SdfCell.TYPE).createSpec();
        if ( smileCount == 1 ) {
        	count ++;
            allColSpecs[count] = 
            	new DataColumnSpecCreator("SMILES",SmilesCell.TYPE).createSpec();
        }
        if ( keyCount == 1 ) {
        	keyCount = count + 1;
        	count ++;
            allColSpecs[count] = 
            	new DataColumnSpecCreator("KEYS",StringCell.TYPE).createSpec();
        }
        if ( stringCount > 0 ) {
        	stringStart=count + 1;
        	for (int i=0; i<stringCount; i++ ) {
        		count ++;
                allColSpecs[count] = 
        		new DataColumnSpecCreator(stringFields[i],StringCell.TYPE).createSpec();
        	}
        }
        if ( doubleCount > 0 ) {
           	doubleStart=count + 1;      	
        	for (int i=0; i<doubleCount; i++ ) {
        		count ++;
                allColSpecs[count] = 
        		new DataColumnSpecCreator(doubleFields[i],DoubleCell.TYPE).createSpec();
        	}
        }
        DataTableSpec outputSpec = new DataTableSpec(allColSpecs);
        BufferedDataContainer cont = exec.createDataContainer(outputSpec);
                
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        
        HashSet<String> titles = new HashSet<String>();

        StringBuilder buf = new StringBuilder(4096);
        boolean titleRead = false;
        String title = "";
    		
        DataCell[] cells = new DataCell[count+1];
        int molcount=0;
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();

            if (!titleRead) {
                title = line;
                if (title.length() == 0) {
                    title = "Mol " + molcount;
                }
                if (titles.contains(title)) {
                    title = title + "-" + molcount;
                }
                titles.add(title);
                titleRead = true;
            }
            buf.append(line).append('\n');

            if (line.startsWith("$$$$")) {
            	molcount ++;
                cells[0] = new SdfCell(buf.toString());
                RowKey key = new RowKey(title);

                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
                buf.delete(0, buf.length());
                titleRead = false;
            }
            else if ( line.startsWith("> <")) {
            	int l = line.length() - 1;
            	fieldName = line.substring(3,l);
            	line = in.readLine();
                buf.append(line).append('\n');
            	if ( smileCount == 1 & fieldName.equals("SMILES")) {
            		cells[1] = new SmilesCell(line);
            	}
            	else if ( keyCount == 1 & fieldName.equals("KEYS")) {
                    cells[keyStart]= new StringCell(line);
            	}
            	else {
            		for (int i=0; i<doubleCount; i++ ) {
            			if ( doubleFields[i].equals(fieldName)) {
            				try {
            				double d_value = Double.valueOf(line).doubleValue();
            				cells[doubleStart+i]= new DoubleCell(d_value);
            				} catch (Throwable t) {
            					cells[doubleStart+i] = DataType.getMissingCell();
            				}
            				break;
            			}
            		}
            		for (int i=0; i<stringCount; i++ ) {
            			if ( stringFields[i].equals(fieldName)) {
            				cells[stringStart+i]= new StringCell(line);
            				break;
            			}
            		}
            	}
            }
        }
        cont.close();
        in.close();
 
        return cont.getTable();
 
  
    }

    private BufferedDataTable readOutputFile(final ExecutionContext exec)
            throws Exception { 

    	String cwdString =  m_working.getStringValue();
        File outFile = new File (cwdString, "denovo.csv");

    	if (!(outFile.exists() && outFile.isFile())) {
           	warningReport();
            throw new IllegalStateException("THINK didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
        
        HashSet<String> titles = new HashSet<String>();
        String title;
        String line;
    	boolean headerRead = false;
    	DataColumnSpec[] allColSpecs;
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        DataTableSpec outputSpec;
        BufferedDataContainer cont =null;
      	RowKey key =null;
      	int count =0;
      	int colcount=0;
      	int rowcount =0;
      	char chr='0';
        int keys = 0;
       	while ((line = in.readLine()) != null) {
            exec.checkCanceled();
// Extract column headers (NAME and SMILES only)
            if ( !headerRead ) {
            	headerRead = true;
            	colcount =0;
            	for ( int i=0; i<line.length(); i++ ) {
            		if ( line.charAt(i) == ',' ) colcount ++;
            	}
// Create column definitions
//                allColSpecs = new DataColumnSpec[1];
                allColSpecs = new DataColumnSpec[colcount];
                 count =0;
               	int ilast=0;
               	for ( int i=0; i<=line.length(); i++ ) {
             		if ( i < line.length() ) {
               			chr = line.charAt(i);
               		}
               		else {
               			chr = ',';
               		}
            		if ( chr == ',' ) {
             			if  ( count == 1 ) {
                   			allColSpecs[count-1] = 
                    		new DataColumnSpecCreator(line.substring(ilast,i),SmilesCell.TYPE).createSpec();
                   			//break;
             			}
             			else if  ( line.substring(ilast,i).equals("KEYS") ) {
             				keys = count;
                   			allColSpecs[count-1] = 
                    		new DataColumnSpecCreator(line.substring(ilast,i),StringCell.TYPE).createSpec();
               			}
            			else if ( count > 1 ) {
            				allColSpecs[count-1] = 
            				new DataColumnSpecCreator(line.substring(ilast,i), DoubleCell.TYPE).createSpec();
            			}
            			count ++;
            			ilast = i + 1;
            		}
              	}
               	outputSpec = new DataTableSpec(allColSpecs);
               	cont = exec.createDataContainer(outputSpec);
               	}
// Load data
            else {
     
                DataCell[] cells = new DataCell[colcount];
               	rowcount++;
                count =0;              	
               	double d_value;
               	int ilast=0;
                for ( int i=0; i<=line.length(); i++ ) {
               		if ( i < line.length() ) {
               			chr = line.charAt(i);
               		}
               		else {
               			chr = ',';
               		}
            		if ( chr == ',' ) { 
            			if ( count == 0 ) {
            				title = line.substring(0,i);
            	            if (title.length() == 0) {
            	            	title = "Mol " + rowcount;
            	            }
            	            if (titles.contains(title)) {
            	            	title = title + "-" + rowcount;
            	            }
            	            titles.add(title);
            				key = new RowKey(title);           				
            			}
            			else if  ( count == 1 ) {
            				cells[count-1] = new SmilesCell(line.substring(ilast,i));
//            				break;
            			}
            			else if  ( count == keys ) {
            				cells[count-1] = new StringCell(line.substring(ilast,i));
            			}
            			else if ( count <= colcount ) {
            			try {
            					d_value = Double.valueOf(line.substring(ilast,i)).doubleValue();
            					cells[count-1] = new DoubleCell(d_value);
            				} catch (Throwable t) {
            					cells[count-1] = DataType.getMissingCell();
            				}
             			}
            			count ++;
            			ilast = i + 1;
            		}
               	}
                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
            }
        }
        cont.close();
        in.close();

        return cont.getTable();

    }

    /**
     * @see org.knime.core.node.NodeModel#reset()
     */
    @Override
    protected void reset() {
        // TODO Code executed on reset.
        // Models build during execute are cleared here.
        // Also data handled in load/saveInternals will be erased here.
    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #saveSettingsTo(org.knime.core.node.NodeSettings)
     */
    @Override
    protected void saveSettingsTo(final NodeSettingsWO settings) {

        // TODO save user settings to the config object.
		
		m_mode.saveSettingsTo(settings);
		m_count.saveSettingsTo(settings);
		m_curb.saveSettingsTo(settings);
		m_group.saveSettingsTo(settings);
	    m_attempts.saveSettingsTo(settings);
		m_transform.saveSettingsTo(settings);
		m_centres.saveSettingsTo(settings);
		m_mass.saveSettingsTo(settings);
		m_xsa.saveSettingsTo(settings);
		m_rotbonds.saveSettingsTo(settings);
		m_smiles.saveSettingsTo(settings);
		m_row1.saveSettingsTo(settings);
		m_row.saveSettingsTo(settings);
		m_working.saveSettingsTo(settings);
		m_group1.saveSettingsTo(settings);
		m_group2.saveSettingsTo(settings);
		m_group3.saveSettingsTo(settings);
		m_group4.saveSettingsTo(settings);

    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #loadValidatedSettingsFrom(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void loadValidatedSettingsFrom(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO load (valid) settings from the config object.
        // It can be safely assumed that the settings are valided by the 
        // method below.
        
        m_mode.loadSettingsFrom(settings);
        m_count.loadSettingsFrom(settings);
        m_curb.loadSettingsFrom(settings);
        m_group.loadSettingsFrom(settings);
        m_attempts.loadSettingsFrom(settings);
        m_transform.loadSettingsFrom(settings);
        m_centres.loadSettingsFrom(settings);
        m_mass.loadSettingsFrom(settings);
        m_xsa.loadSettingsFrom(settings);
        m_rotbonds.loadSettingsFrom(settings);
        m_smiles.loadSettingsFrom(settings);
        m_row1.loadSettingsFrom(settings);
        m_row.loadSettingsFrom(settings);
        m_working.loadSettingsFrom(settings);
        m_group1.loadSettingsFrom(settings);
        m_group2.loadSettingsFrom(settings);
        m_group3.loadSettingsFrom(settings);
        m_group4.loadSettingsFrom(settings); 
    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #validateSettings(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void validateSettings(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO check if the settings could be applied to our model
        // e.g. if the count is in a certain range (which is ensured by the
        // SettingsModel).
        // Do not actually set any values of any member variables.

        m_mode.validateSettings(settings);
        m_count.validateSettings(settings);
        m_curb.validateSettings(settings);
        m_group.validateSettings(settings);
        m_attempts.validateSettings(settings);
        m_transform.validateSettings(settings);
        m_centres.validateSettings(settings);
        m_mass.validateSettings(settings);
        m_xsa.validateSettings(settings);
        m_rotbonds.validateSettings(settings);
        m_smiles.validateSettings(settings);
        m_row1.validateSettings(settings);
        m_row.validateSettings(settings);
        m_working.validateSettings(settings);
        m_group1.validateSettings(settings);
        m_group2.validateSettings(settings);
        m_group3.validateSettings(settings);
        m_group4.validateSettings(settings);
 
    }
    
    /**
     * @see org.knime.core.node.NodeModel #loadInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void loadInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
        
		// TODO load internal data. 
		// Everything handed to output ports is loaded automatically (data
		// returned by the execute method, models loaded in loadModelContent,
		// and user settings set through loadSettingsFrom - is all taken care 
		// of). Load here only the other internals that need to be restored
		// (e.g. data used by the views).

    }
    
    /**
     * @see org.knime.core.node.NodeModel #saveInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void saveInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
       
       	// TODO save internal models. 
		// Everything written to output ports is saved automatically (data
		// returned by the execute method, models saved in the saveModelContent,
		// and user settings saved through saveSettingsTo - is all taken care 
		// of). Save here only the other internals that need to be preserved
		// (e.g. data used by the views).

    }

}
