package com.treweren.denovo;

import java.awt.GridLayout;
import java.awt.FlowLayout;

import javax.swing.JFileChooser;

import org.knime.core.node.KNIMEConstants;
import org.knime.core.node.defaultnodesettings.DefaultNodeSettingsPane;
import org.knime.core.node.defaultnodesettings.DialogComponentBoolean;
import org.knime.core.node.defaultnodesettings.DialogComponentFileChooser;
import org.knime.core.node.defaultnodesettings.DialogComponentNumber;
import org.knime.core.node.defaultnodesettings.DialogComponentDoubleRange;
import org.knime.core.node.defaultnodesettings.DialogComponentString;
import org.knime.core.node.defaultnodesettings.DialogComponentStringSelection;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelDoubleBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelDoubleRange;
import org.knime.core.node.defaultnodesettings.SettingsModelString;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;





/**
 * <code>NodeDialog</code> for the "DeNovo" Node.
 * Generates de novo derivatives
 *
 * This node dialog derives from {@link DefaultNodeSettingsPane} which allows
 * creation of a simple dialog with standard components. If you need a more 
 * complex dialog please derive directly from 
 * {@link org.knime.core.node.NodeDialogPane}.
 * 
 * @author Treweren Consultants
 */
public class DeNovoNodeDialog extends DefaultNodeSettingsPane {

    /**
     * New pane for configuring DeNovo node dialog.
     * This is just a suggestion to demonstrate possible default dialog
     * components.
     */
    protected DeNovoNodeDialog() {
        super();
        DialogComponentStringSelection c_mode = new DialogComponentStringSelection(
                new SettingsModelString(
                        DeNovoNodeModel.CFGKEY_MODE,
                        DeNovoNodeModel.DEFAULT_MODE),
                        "Derivative mode", "2D", "3D", "3D in Protein", "3D in Map");
        addDialogComponent(c_mode);
        c_mode.setToolTipText("The derivatives may be unconstrained (2D) or use substitutions only to maintain the core (3D) with optional protein or pharmacophore volume constraints");
        c_mode.getComponentPanel().setLayout( new FlowLayout(2));
        
        DialogComponentNumber c_count = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        DeNovoNodeModel.CFGKEY_COUNT,
                        DeNovoNodeModel.DEFAULT_COUNT,
                        Integer.MIN_VALUE, Integer.MAX_VALUE),
                        "Desired number of derivatives", /*step*/ 1, /*componentwidth*/ 5);
        
        addDialogComponent(c_count);
        c_count.getComponentPanel().setLayout( new FlowLayout(2));
        c_count.setToolTipText("THINK attempts to generate the specified number of drug-like derivatives");

        DialogComponentNumber c_attempts = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        DeNovoNodeModel.CFGKEY_ATTEMPTS,
                        DeNovoNodeModel.DEFAULT_ATTEMPTS,
                        Integer.MIN_VALUE, Integer.MAX_VALUE),
                        "Maximum number of attempts", /*step*/ 1, /*componentwidth*/ 5);
        
        addDialogComponent(c_attempts);
        c_attempts.getComponentPanel().setLayout( new FlowLayout(2));
        c_attempts.setToolTipText("The number of attempts is limited to restrict the time spent considering randomly selected transformations");

        DialogComponentNumber c_curb = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        DeNovoNodeModel.CFGKEY_CURB,
                        DeNovoNodeModel.DEFAULT_CURB,
                        Integer.MIN_VALUE, Integer.MAX_VALUE),
                        "Curb on non-H atom insertions", /*step*/ 1, /*componentwidth*/ 5);

        DialogComponentStringSelection c_group = new DialogComponentStringSelection(
                new SettingsModelString(
                        DeNovoNodeModel.CFGKEY_GROUP,
                        DeNovoNodeModel.DEFAULT_GROUP),
                        "Group restriction on modified atoms", "None", "1", "2", "3", "4", "5", "6", "7", "8", "9");
        addDialogComponent(c_group);
        c_group.setToolTipText("The changes may be restricted to atoms with specific group numbers (set using the Display node)");
        c_group.getComponentPanel().setLayout( new FlowLayout(2));
        
        addDialogComponent(c_curb);
        c_curb.getComponentPanel().setLayout( new FlowLayout(2));
        c_curb.setToolTipText("The probability for transformations which insert more non-H atoms than the curb value is scaled progressively to zero at twice the curb value");

        
        DialogComponentDoubleRange c_centres = new DialogComponentDoubleRange (
        		new SettingsModelDoubleRange (DeNovoNodeModel.CFGKEY_CENTRES, DeNovoNodeModel.DEFAULT_CENTRE1, DeNovoNodeModel.DEFAULT_CENTRE2),
        		 0., 10., 1., 0., 20., 1., "Interaction Centres");
        addDialogComponent(c_centres);        		
        c_centres.setToolTipText("Eliminate molecules with an interaction centres count outside the range");
        c_centres.getComponentPanel().setLayout( new FlowLayout(2));
        
        DialogComponentDoubleRange c_mass = new DialogComponentDoubleRange(
        		new SettingsModelDoubleRange( DeNovoNodeModel.CFGKEY_MASS, DeNovoNodeModel.DEFAULT_MASS1, DeNovoNodeModel.DEFAULT_MASS2),
        		0.,1000., 10., 0., 1000., 10., "Molecular mass");
        
        addDialogComponent(c_mass);
        c_mass.setToolTipText("Eliminate molecules with a molecular mass outside the range");
        c_mass.getComponentPanel().setLayout( new FlowLayout(2));
        
        DialogComponentDoubleRange c_xsa = new DialogComponentDoubleRange(
        		new SettingsModelDoubleRange( DeNovoNodeModel.CFGKEY_XSA, DeNovoNodeModel.DEFAULT_XSA1, DeNovoNodeModel.DEFAULT_XSA2),
        		0.,1000., 10., 0., 1000., 10., "N+O surface area");
        
        addDialogComponent(c_xsa);
        c_xsa.setToolTipText("Eliminate molecules with a N+O surface area outside the range");
        c_xsa.getComponentPanel().setLayout( new FlowLayout(2));
        
        DialogComponentDoubleRange c_rotbonds = new DialogComponentDoubleRange(
        		new SettingsModelDoubleRange( DeNovoNodeModel.CFGKEY_ROTBONDS, DeNovoNodeModel.DEFAULT_ROTBONDS1, DeNovoNodeModel.DEFAULT_ROTBONDS2),
        		0.,100., 1., 0., 100., 1., "Rotatable bonds");
        
        addDialogComponent(c_rotbonds);
        c_rotbonds.setToolTipText("Eliminate molecules with a rotatable bond count outside the range");
        c_rotbonds.getComponentPanel().setLayout( new FlowLayout(2));

        createNewTab("Groups");
        
        DialogComponentString c_group1 = new DialogComponentString(
                new SettingsModelString (
                    DeNovoNodeModel.CFGKEY_GROUP1,
                    DeNovoNodeModel.DEFAULT_GROUP1),
                    "Atoms for group 1" );
        addDialogComponent(c_group1);
        c_group1.getComponentPanel().setLayout( new FlowLayout(0));
        c_group1.setToolTipText("Comma separated list of atoms for group eg C16,H17,H18; N(24),H(25); (24:25)");

        DialogComponentString c_group2 = new DialogComponentString(
                new SettingsModelString (
                    DeNovoNodeModel.CFGKEY_GROUP2,
                    DeNovoNodeModel.DEFAULT_GROUP2),
                    "Atoms for group 2" );
        addDialogComponent(c_group2);
        c_group2.getComponentPanel().setLayout( new FlowLayout(0));
        c_group2.setToolTipText("Comma separated list of atoms for group eg C16,H17,H18; N(24),H(25); (24:25)");

        DialogComponentString c_group3 = new DialogComponentString(
                new SettingsModelString (
                    DeNovoNodeModel.CFGKEY_GROUP3,
                    DeNovoNodeModel.DEFAULT_GROUP3),
                    "Atoms for group 3" );
        addDialogComponent(c_group3);
        c_group3.getComponentPanel().setLayout( new FlowLayout(0));
        c_group3.setToolTipText("Comma separated list of atoms for group eg C16,H17,H18; N(24),H(25); (24:25");

        DialogComponentString c_group4 = new DialogComponentString(
                new SettingsModelString (
                    DeNovoNodeModel.CFGKEY_GROUP4,
                    DeNovoNodeModel.DEFAULT_GROUP4),
                    "Atoms for group 4" );
        addDialogComponent(c_group4);
        c_group4.getComponentPanel().setLayout( new FlowLayout(0));
        c_group4.setToolTipText("Comma separated list of atoms for group eg C16,H17,H18; N(24),H(25); (24:25");

     
        createNewTab("THINK setup");
 
        DialogComponentBoolean c_column = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        DeNovoNodeModel.CFGKEY_SMILES,
                        DeNovoNodeModel.DEFAULT_SMILES),
                        "Use SMILES in preference to SD column" );
        addDialogComponent(c_column);
        c_column.getComponentPanel().setLayout( new FlowLayout(0));
        c_column.setToolTipText("The molecule data is transferred to THINK using SD or SMILES format depending on this setting");
        
        DialogComponentBoolean c_row = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        DeNovoNodeModel.CFGKEY_ROW1,
                        DeNovoNodeModel.DEFAULT_ROW1),
                        "Use first row/molecule (not all)" );
        
        addDialogComponent(c_row);
        c_row.getComponentPanel().setLayout( new FlowLayout(0));
        c_row.setToolTipText("The data transferred may be limited to the first row using this control");
 
        DialogComponentNumber c_site = new DialogComponentNumber(
                new SettingsModelIntegerBounded (
                        DeNovoNodeModel.CFGKEY_ROW,
                        DeNovoNodeModel.DEFAULT_ROW, 0, Integer.MAX_VALUE),
                        "Site row",1, 2 );
        
        addDialogComponent(c_site);
        c_site.getComponentPanel().setLayout( new FlowLayout(0));
        c_site.setToolTipText("The data transferred will be limited to one site");
 
        DialogComponentFileChooser c_transform = new DialogComponentFileChooser(
                new SettingsModelString (
                        DeNovoNodeModel.CFGKEY_TRANSFORM,
                        DeNovoNodeModel.DEFAULT_TRANSFORM ),
                        "Transform file", JFileChooser.OPEN_DIALOG, "smi" );
        addDialogComponent(c_transform );
        c_transform.getComponentPanel().setLayout( new FlowLayout(2));
        c_transform.setToolTipText("A customised set of transformations will be loaded from this file");

        DialogComponentString c_working = new DialogComponentString(
                new SettingsModelString (
                        DeNovoNodeModel.CFGKEY_WORKING,
                        DeNovoNodeModel.DEFAULT_WORKING),
                        "Working folder" );
        addDialogComponent(c_working);
        c_working.getComponentPanel().setLayout( new FlowLayout(2));
        c_working.setToolTipText("The temporary files used to transfer the data to and from THINK are placed in this folder");

                
    }
}
