package com.treweren.pharmacophores;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.HashSet;
import java.util.HashMap;
import java.util.TreeMap;
import java.util.Map;
import java.util.ArrayList;
import java.util.Properties;


import org.knime.base.node.io.filereader.FileAnalyzer;
import org.knime.base.node.io.filereader.FileReaderNodeSettings;
import org.knime.base.node.io.filereader.FileReaderSettings;
import org.knime.base.node.io.filereader.FileTable;
import org.knime.base.node.io.filereader.SettingsStatus;
import org.knime.chem.types.SdfCell;
import org.knime.chem.types.SdfValue;
import org.knime.chem.types.SmilesValue;
import org.knime.core.data.DataCell;
import org.knime.core.data.DataColumnSpec;
import org.knime.core.data.DataColumnSpecCreator;
import org.knime.core.data.DataColumnProperties;
import org.knime.core.data.DataRow;
import org.knime.core.data.DataTableSpec;
import org.knime.core.data.DataType;
import org.knime.core.data.RowKey;
import org.knime.core.data.def.DefaultRow;
import org.knime.core.data.def.DoubleCell;
import org.knime.core.data.def.IntCell;
import org.knime.core.data.def.StringCell;
import org.knime.core.node.BufferedDataContainer;
import org.knime.core.node.BufferedDataTable;
import org.knime.core.node.CanceledExecutionException;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelDoubleBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelString;
import org.knime.core.node.ExecutionContext;
import org.knime.core.node.ExecutionMonitor;
import org.knime.core.node.InvalidSettingsException;
import org.knime.core.node.NodeLogger;
import org.knime.core.node.NodeModel;
import org.knime.core.node.NodeSettingsRO;
import org.knime.core.node.NodeSettingsWO;
import org.knime.core.util.MutableBoolean;



/**
 * This is the model implementation of Pharmacophores.
 * Compute most common pharmacophores for a set of active molecules
 *
 * @author Treweren Consultants
 */
public class PharmacophoresNodeModel extends NodeModel {
    
    // the logger instance
    private static final NodeLogger logger = NodeLogger
            .getLogger(PharmacophoresNodeModel.class);
    private static int n_Column = 0;
    private static String m_sdfColumn = null;
    private static String m_smilesColumn = null;
    private static String m_stringColumn = null;
    private static final int CANCEL_CHECK_INTERVAL = 1000;
             
    /** the settings key which is used to retrieve and 
        store the settings (from the dialog or from a settings file)    
       (package visibility to be usable from the dialog). */
	static final String CFGKEY_COUNT = "Count";
	static final String CFGKEY_EMPTY = "Empty";
	static final String CFGKEY_TOLERANCE = "Tolerance";
	static final String CFGKEY_CENTRES = "Interactions";
	static final String CFGKEY_OUTPUT = "Output";
	static final String CFGKEY_VALID = "Validate";
	static final String CFGKEY_SMILES = "Smiles";
	static final String CFGKEY_WORKING = "Working";
	static final String CFGKEY_SINGLE = "Single";
	static final String CFGKEY_CONJUGATED = "Conjugated";
	static final String CFGKEY_CROWDED = "Crowded";
	static final String CFGKEY_ALPHA = "Alpha";
	static final String CFGKEY_AMIDE = "Amide";
	static final String CFGKEY_RING = "Ring";
	static final String CFGKEY_GENERATION = "Generation";
	static final String CFGKEY_SAMPLES = "Samples";
	static final String CFGKEY_CPKRATIO = "CPKratio";
	static final String CFGKEY_TIMELIMIT = "TimeLimit";
	static final String CFGKEY_SYSLIMIT = "SysLimit";
	static final String CFGKEY_BONDLIMIT = "BondLimit";

	/** initial default count value. */
	static final int DEFAULT_COUNT = 100;
	static final double DEFAULT_EMPTY = 0.01;
	static final double DEFAULT_TOLERANCE = 0.5;
	static final double DEFAULT_VALID = 90.;
	static final String DEFAULT_OUTPUT= "3D Coordinates";
	static final int DEFAULT_CENTRES = 4;
	static final boolean DEFAULT_SMILES = false;
	static final String DEFAULT_WORKING = System.getenv("THINK_WORKING");
	static final int DEFAULT_SINGLE = 3;
	static final int DEFAULT_CONJUGATED = 2;
	static final int DEFAULT_CROWDED = 2;
	static final int DEFAULT_ALPHA = 6;
	static final int DEFAULT_AMIDE = 0;
	static final int DEFAULT_RING = 0;
	static final String DEFAULT_GENERATION = "Systematic";
	static final int DEFAULT_SAMPLES = 1000;
	static final double DEFAULT_CPKRATIO = 0.6;
	static final double DEFAULT_TIMELIMIT = 60.;
	static final int DEFAULT_SYSLIMIT = 0;
	static final int DEFAULT_BONDLIMIT = 10;

	// example value: the models count variable filled from the dialog 
	// and used in the models execution method. The default components of the
	// dialog work with "SettingsModels".
	
	private final SettingsModelIntegerBounded m_count =
		new SettingsModelIntegerBounded(PharmacophoresNodeModel.CFGKEY_COUNT,
                    PharmacophoresNodeModel.DEFAULT_COUNT,
                    Integer.MIN_VALUE, Integer.MAX_VALUE);

	private final SettingsModelDoubleBounded m_empty =
		new SettingsModelDoubleBounded(PharmacophoresNodeModel.CFGKEY_EMPTY,
                    PharmacophoresNodeModel.DEFAULT_EMPTY,
                    0., 100.0);

	private final SettingsModelDoubleBounded m_tolerance =
		new SettingsModelDoubleBounded(PharmacophoresNodeModel.CFGKEY_TOLERANCE,
                    PharmacophoresNodeModel.DEFAULT_TOLERANCE,
                    0., 5.0);

	private final SettingsModelDoubleBounded m_valid =
		new SettingsModelDoubleBounded(PharmacophoresNodeModel.CFGKEY_VALID,
                    PharmacophoresNodeModel.DEFAULT_VALID,
                    0., 100.0);
		
	private final SettingsModelString m_output =
		new SettingsModelString(PharmacophoresNodeModel.CFGKEY_OUTPUT,
                    PharmacophoresNodeModel.DEFAULT_OUTPUT);

	private final SettingsModelIntegerBounded m_centres =
		new SettingsModelIntegerBounded(PharmacophoresNodeModel.CFGKEY_CENTRES,
                    PharmacophoresNodeModel.DEFAULT_CENTRES, 3, 4 );

	private final SettingsModelBoolean m_smiles =
		new SettingsModelBoolean(PharmacophoresNodeModel.CFGKEY_SMILES,
                    PharmacophoresNodeModel.DEFAULT_SMILES );

	private final SettingsModelString m_working =
		new SettingsModelString(PharmacophoresNodeModel.CFGKEY_WORKING, PharmacophoresNodeModel.DEFAULT_WORKING );

	private final SettingsModelIntegerBounded m_single =
		new SettingsModelIntegerBounded(PharmacophoresNodeModel.CFGKEY_SINGLE,
                    PharmacophoresNodeModel.DEFAULT_SINGLE, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_conjugated =
		new SettingsModelIntegerBounded(PharmacophoresNodeModel.CFGKEY_CONJUGATED,
                    PharmacophoresNodeModel.DEFAULT_CONJUGATED, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_crowded =
		new SettingsModelIntegerBounded(PharmacophoresNodeModel.CFGKEY_CROWDED,
                    PharmacophoresNodeModel.DEFAULT_CROWDED, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_alpha =
		new SettingsModelIntegerBounded(PharmacophoresNodeModel.CFGKEY_ALPHA,
                    PharmacophoresNodeModel.DEFAULT_ALPHA, 0, Integer.MAX_VALUE);


	private final SettingsModelIntegerBounded m_amide =
		new SettingsModelIntegerBounded(PharmacophoresNodeModel.CFGKEY_AMIDE,
                    PharmacophoresNodeModel.DEFAULT_AMIDE, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_ring =
		new SettingsModelIntegerBounded(PharmacophoresNodeModel.CFGKEY_RING,
                    PharmacophoresNodeModel.DEFAULT_RING, 0, Integer.MAX_VALUE);
	
	private final SettingsModelString m_generation =
		new SettingsModelString(PharmacophoresNodeModel.CFGKEY_GENERATION,
                    PharmacophoresNodeModel.DEFAULT_GENERATION);

	private final SettingsModelIntegerBounded m_samples =
		new SettingsModelIntegerBounded(PharmacophoresNodeModel.CFGKEY_SAMPLES,
                    PharmacophoresNodeModel.DEFAULT_SAMPLES, 0, Integer.MAX_VALUE);

	private final SettingsModelDoubleBounded m_cpkratio =
		new SettingsModelDoubleBounded(PharmacophoresNodeModel.CFGKEY_CPKRATIO,
                    PharmacophoresNodeModel.DEFAULT_CPKRATIO,
                    0., 5.0);
	
	private final SettingsModelDoubleBounded m_timelimit =
		new SettingsModelDoubleBounded(PharmacophoresNodeModel.CFGKEY_TIMELIMIT,
                    PharmacophoresNodeModel.DEFAULT_TIMELIMIT,
                    0., 1440.);

	private final SettingsModelIntegerBounded m_syslimit =
		new SettingsModelIntegerBounded(PharmacophoresNodeModel.CFGKEY_SYSLIMIT,
                    PharmacophoresNodeModel.DEFAULT_SYSLIMIT, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_bondlimit =
		new SettingsModelIntegerBounded(PharmacophoresNodeModel.CFGKEY_BONDLIMIT,
                    PharmacophoresNodeModel.DEFAULT_BONDLIMIT, 0, Integer.MAX_VALUE);


     /**
     * Constructor for the node model.
     */
    protected PharmacophoresNodeModel() {
    
        // One incoming port and three outgoing ports 
    	super(1, 3);
    }

    /**
     * @see org.knime.core.node.NodeModel #execute(BufferedDataTable[],
     *      ExecutionContext)
     */
    @Override
    protected DataTableSpec[] configure(final DataTableSpec[] inSpecs) throws InvalidSettingsException {

        if ( System.getenv("THINK_EXEC") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_EXEC is unset - It should be set to the folder containing the THINK software.");       	  		
    	}
        if ( System.getenv("THINK_WORKING") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_WORKING is not set to the folder in which working files will be created.");       	  		
    	}

        DataTableSpec in = inSpecs[0];
        n_Column = columnSelect (in);
        return new DataTableSpec[] {null,null,null};
    }
    protected int columnSelect ( final DataTableSpec in ) throws  InvalidSettingsException {

        StringBuilder warningMessage = new StringBuilder();
        int n_column=0;
        int sdfColCount = 0;
        int n_sdf=0;
        int smilesColCount = 0;
        int n_smiles=0;
        int stringColCount = 0;
        int n_string=0;
        for (int i = 0; i < in.getNumColumns(); i++) {
            DataColumnSpec s = in.getColumnSpec(i);
            if (s.getType().isCompatible(SdfValue.class)) {
                if (m_sdfColumn == null) {
                    m_sdfColumn = in.getColumnSpec(i).getName();
                    n_sdf = i;
                }
                sdfColCount++;
            }
            else if ( s.getType().isCompatible(SmilesValue.class)) {
            	if (m_smilesColumn == null ) {
             		m_smilesColumn = in.getColumnSpec(i).getName();
             		n_smiles=i;
              	}
               	smilesColCount++;
            }
            else if ( in.getColumnSpec(i).getName().equalsIgnoreCase("smiles"))  {
            	if (m_stringColumn == null ) {
             		m_stringColumn = in.getColumnSpec(i).getName();
             		n_string=i;
              	}
               	stringColCount++;
            }
        }
        if ( smilesColCount == 0 & sdfColCount == 0 & stringColCount == 0) {
            throw new InvalidSettingsException(
                    "No column with Smiles or SD file compatible type");       	
        }
        else if ( ( m_smiles.getBooleanValue() | sdfColCount == 0 ) & smilesColCount > 0 ) {
        	m_sdfColumn = null;
        	m_stringColumn = null;
        	n_column = n_smiles;
        	if (smilesColCount > 1) {
                warningMessage.append("More than one Smiles compatible column in " 
                        + "input, using column \"" + m_smilesColumn + "\".");
        	}
        }
        else if ( sdfColCount > 0 ) {
           	m_smilesColumn = null;
        	m_stringColumn = null;
        	n_column = n_sdf;
        	if (sdfColCount > 1) {
                warningMessage.append("More than one SDF compatible column in " 
                        + "input, using column \"" + m_sdfColumn + "\".");
        	}
        }
        else if ( stringColCount > 0 ) {
        	m_smilesColumn = null;
        	m_sdfColumn = null;
        	n_column = n_string;
           	if (stringColCount > 1) {
                warningMessage.append("More than one Smiles string compatible column in " 
                        + "input, using column \"" + m_stringColumn + "\".");
           	}
        }
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
        return n_column;
    }
 
    /**
     * @see org.knime.core.node.NodeModel #execute(BufferedDataTable[],
     *      ExecutionContext)
     */
    @Override
    protected BufferedDataTable[] execute(final BufferedDataTable[] inData,
            final ExecutionContext exec) throws Exception {

        notifyViews(null);

        final String cwdString = m_working.getStringValue();
        // clean up 
        File outputReport = new File(cwdString, "output.log"); 
        outputReport.delete();
        File outputFile1a = new File(cwdString, "trainers.phc"); 
        outputFile1a.delete();
        File outputFile1b = new File(cwdString, "trainers_maps.sdf"); 
        outputFile1b.delete();
        File outputFile2 = new File(cwdString, "trainers.pro"); 
        outputFile2.delete();
        File outputFile3 = new File(cwdString, "trainers.phm"); 
        outputFile3.delete();
        File progressFile = new File(cwdString, "progress0.dat"); 
        progressFile.delete();
        BufferedDataTable in = inData[0];

        if ( n_Column == 0 ) {
        	DataTableSpec InSpec = in.getDataTableSpec();
        	n_Column = columnSelect(InSpec);
        }

 
        // write out input file
        String fileString = null; 
        if ( m_sdfColumn == null ) {
            fileString = "trainers.smiles"; 
        }
        else {
            fileString = "trainers.sdf"; 
        }
        File inFile = new File(cwdString, fileString); 
        BufferedWriter outWriter = new BufferedWriter(new FileWriter(inFile));
         int colIndex;
        if ( m_sdfColumn != null ) {
        	colIndex = in.getDataTableSpec().findColumnIndex(m_sdfColumn);
        }
        else if ( m_smilesColumn != null ) {
        	colIndex = in.getDataTableSpec().findColumnIndex(m_smilesColumn);        
        }
        else {
            colIndex = in.getDataTableSpec().findColumnIndex(m_stringColumn);        
        }
       	  
        final double count = in.getRowCount(); // floating point operations
        int i = 0; 
        int missingCount = 0;
        for (DataRow r : in) {
            exec.checkCanceled();
            DataCell c = r.getCell(colIndex);
            if (c.isMissing()) {
                missingCount++;
            } 
            else if ( m_sdfColumn != null ) {
                SdfValue v = (SdfValue)c;
                String toString = v.toString();
                outWriter.write(toString);
                if (!toString.trim().endsWith("$$$$")) {
                    outWriter.newLine();
                    outWriter.append("$$$$");
                    outWriter.newLine();
                }
            }
            else if ( m_smilesColumn != null ) {
                SmilesValue v = (SmilesValue)c;
                String toString = v.toString() + " " + r.getKey();
                outWriter.write(toString);        	
                outWriter.newLine();
            }
            else {
            	String toString = c.toString() + " " + r.getKey();
            	outWriter.write(toString);        	
            	outWriter.newLine();
            }

           i++;
        }

        outWriter.close();
        if (missingCount > 0) {
            setWarningMessage("Skipped " + missingCount 
                    + " row(s) because of missing values");
        }

        // execute THINK
        File cwdFile = new File (cwdString);
        int exitVal;
        String s_output = "3D";
        try {
            exec.setProgress(0., "Starting THINK");
            Runtime rt = Runtime.getRuntime();
            int n_count = m_count.getIntValue();
            double d_empty = m_empty.getDoubleValue();
            double d_tolerance = m_tolerance.getDoubleValue();
            double d_valid = m_valid.getDoubleValue();
            int n_centres = m_centres.getIntValue();
            if (m_output.getStringValue().equals("3D coord & maps") ) {
            	s_output = "MAPS";
            }
            else if ( m_output.getStringValue().equals("Profile") ) { 
               	s_output="PROFILE";
            }
            else if ( m_output.getStringValue().equals("Pharmacophores") ) { 
               	s_output="PHARMS";
        
            }
            int n_single = m_single.getIntValue();
            int n_conjugated = m_conjugated.getIntValue();
            int n_crowded = m_crowded.getIntValue();
            int n_alpha = m_alpha.getIntValue();
            int n_amide = m_amide.getIntValue();
            int n_ring = m_ring.getIntValue();
            String s_generation = m_generation.getStringValue();
            int n_samples = m_samples.getIntValue();
            double d_cpkratio = m_cpkratio.getDoubleValue();
            double d_timelimit = m_timelimit.getDoubleValue();
            int n_syslimit = m_syslimit.getIntValue();
            int n_bondlimit = m_bondlimit.getIntValue();

            // prepare the command string
            String cmdString = System.getenv("THINK_EXEC") + "think THINK_EXEC:pharms.log output.log " + fileString + " " + n_count + " " + d_empty + " " + d_tolerance + " " + n_centres + " " + s_output + " " + d_valid
            + " " + n_single + " " + n_conjugated + " " + n_crowded + " " + n_alpha  + " " + n_amide + " " + n_ring
            + " " + s_generation + " " + n_samples + " " + d_cpkratio + " " + d_timelimit + " " + n_syslimit + " " + n_bondlimit;

            // Go!
            logger.info("THINK command line: '" + cmdString + "'");
            final Process proc = rt.exec(cmdString, null, cwdFile);

            final MutableBoolean procDone = new MutableBoolean(false);
            new Thread(new Runnable() {
                public void run() {
                    synchronized (procDone) {
                        while (!procDone.booleanValue()) {
                            try {
                                exec.checkCanceled();
                            } catch (CanceledExecutionException cee) {
                                // blow away the running external process
                                proc.destroy();
                                return;
                            }
                            try {
                               	procDone.wait(CANCEL_CHECK_INTERVAL);
                                // Read progress file and set progress value       
                               	double d= 0.;
                                File progressReport = new File(cwdString, "progress0.dat");
                                if ( progressReport.exists() && progressReport.isFile() ) {
                               		if ( progressReport.length() > 0 ) {
                               			BufferedReader in = new BufferedReader(new FileReader(progressReport));
                               			String line;
                               			if ( (line = in.readLine()) != null) {
                               			  d = Double.valueOf(line).doubleValue();
                               			}               
                             			if ( (line = in.readLine()) == null) {
                              				line = "Running THINK";
                              			}
                               			in.close();
                                        exec.setProgress( d, line.toString());
                               		}
                                }
                            } catch (InterruptedException e) {
                                // do nothing
                            } catch ( FileNotFoundException e) {
                                // do nothing
                            } catch ( IOException e) {
                                // do nothing
                            }
                        }
                    }
                }

            }).start();

            // wait until the external process finishes.
            exitVal = proc.waitFor();

            synchronized (procDone) {
                // this should terminate the check cancel thread
                procDone.setValue(true);
            }

            exec.checkCanceled();

            exec.setProgress("Wrapping up");
            logger.info("THINK terminated with exit code: " + exitVal);
        } catch (Throwable t) {
        	warningReport();
            logger.error("THINK failed (with exception)", t);
            throw new Exception(t);
        }

        if (exitVal != 0) {
        	warningReport();
            // before we return barfing, we save the output in the failing list
            throw new IllegalStateException("THINK failed (error code "
                    + exitVal + ")");
        }

        exec.setProgress("Creating KNIME tables");
        BufferedDataTable coordTable = null;
        BufferedDataTable profileTable = null;
        BufferedDataTable pharmTable = null;
        if ( s_output == "PHARMS") {         
        	pharmTable = readPharmFile("trainers.phm", exec);
           	profileTable = dummyProfileFile( exec);
        	coordTable = dummyOutputFile(exec);        
        }
        else if ( s_output == "PROFILE") {
        	pharmTable = dummyPharmFile(exec);
        	profileTable = readProfileFile("trainers.pro", exec);
        	coordTable = dummyOutputFile(exec);        }
        else {
           	pharmTable = dummyPharmFile(exec);
           	profileTable = dummyProfileFile(exec);
        	coordTable = readOutputFile(exec);
        }
 
        return new BufferedDataTable[]{coordTable, profileTable, pharmTable };

    }

    private void warningReport( ) throws Exception {

    	StringBuilder warningMessage = new StringBuilder();
        String cwdString = m_working.getStringValue();
        File outputReport = new File(cwdString, "output.log"); 

        if ( outputReport.exists() && outputReport.isFile() ) {
       		if ( outputReport.length() > 0 ) {
       			BufferedReader in = new BufferedReader(new FileReader(outputReport));
       			String line;
       			while ((line = in.readLine()) != null) {
       				warningMessage.append( line + "\n");
       			}               
       			in.close();
       		}
       		else { 
                warningMessage.append( "Report from THINK is blank");      	     			
       		}
        }
       	else {
            warningMessage.append( "No report from THINK");      	
       	}	
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
   }
    
    private BufferedDataTable dummyPharmFile(final ExecutionContext exec)
    throws Exception {
        DataColumnSpec[] allColSpecs = new DataColumnSpec[1];
      	allColSpecs[0]=new DataColumnSpecCreator("Disabled", DoubleCell.TYPE).createSpec(); 
        DataTableSpec outputSpec = new DataTableSpec(allColSpecs);
        BufferedDataContainer cont = exec.createDataContainer(outputSpec);
        cont.close();
        return cont.getTable();
    }
    
    private BufferedDataTable dummyProfileFile(final ExecutionContext exec)
    throws Exception {
        DataColumnSpec[] allColSpecs = new DataColumnSpec[1];
      	allColSpecs[0]=new DataColumnSpecCreator("Disabled", DoubleCell.TYPE).createSpec(); 
        DataTableSpec outputSpec = new DataTableSpec(allColSpecs);
        BufferedDataContainer cont = exec.createDataContainer(outputSpec);
        cont.close();
        return cont.getTable();
    }
    
    private BufferedDataTable dummyOutputFile(final ExecutionContext exec)
    throws Exception {
        DataColumnSpec[] allColSpecs = new DataColumnSpec[1];
      	allColSpecs[0]=new DataColumnSpecCreator("Disabled", SdfCell.TYPE).createSpec();
        DataTableSpec outputSpec = new DataTableSpec(allColSpecs);
        BufferedDataContainer cont = exec.createDataContainer(outputSpec);
        cont.close();
        return cont.getTable();
    }

    
    private BufferedDataTable readPharmFile(String outString, final ExecutionContext exec)
    throws Exception {

    	String cwdString = m_working.getStringValue();
     	int nocols=0;
    	File outFile = new File (cwdString, outString);

    	if (!(outFile.exists() && outFile.isFile())) {
    		warningReport();
    		throw new IllegalStateException("THINK didn't produce any output"
            + " at the specified location ('"
            + outFile.getAbsolutePath() + "')");
    	}
    	// Scan file and count number of pharmacophores = number of columns
        BufferedReader looksee =  new BufferedReader(new FileReader(outFile));
        String line;
        ArrayList pharms = new ArrayList();
        int npharms=0;
        
        while ((line = looksee.readLine()) != null) {
          exec.checkCanceled();
          if ( !line.startsWith("#") ) {
        	  int ilast =0;
              for ( int i=0; i<line.length(); i++ ) {
                  if ( line.charAt(i) == ',' ) {
                	  if ( ilast == 0 ){
                		  ilast = i+1;
                	  }
                	  else if ( !pharms.contains ( line.substring(ilast,i) ) ) { 
                       	  npharms = npharms + 1;
                       	  pharms.add( line.substring(ilast,i) );
                    	  break;
                	  }
                  }
              }

          }
        }
        looksee.close();
        
        DataColumnSpec[] allColSpecs = new DataColumnSpec[npharms];
        for ( int n=0; n<npharms; n++) {
        	allColSpecs[n]=new DataColumnSpecCreator(pharms.get(n).toString(), DoubleCell.TYPE).createSpec(); 
        }
        DataTableSpec outputSpec = new DataTableSpec(allColSpecs);
        BufferedDataContainer cont = exec.createDataContainer(outputSpec);

        BufferedReader in =  new BufferedReader(new FileReader(outFile));
        HashSet<String> titles = new HashSet<String>();
        char chr='0';
        String title=" ";
        int count=0;
        double[] d_values = new double[npharms];
         while ((line = in.readLine()) != null) {
            exec.checkCanceled();
            if ( !line.startsWith("#") ) {
            	int ilast = 0;
            	int ncol = -1;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else {
                        chr = ',';
                    }
                    if ( chr == ',' ) {
// Row ID: Molecule name
                    	if ( ilast == 0 ) {
// Add completed row?                    		
                    		if ( !line.substring(ilast,i).equals(title)) {
                    			count ++;
                    			if ( !title.equals(" ") ) {
                                	RowKey key = new RowKey(title);
                                    DataCell[] cells = new DataCell[npharms];
                                    for (int n=0; n<npharms; n++) {
                                    	cells[n] = new DoubleCell(d_values[n]);
                                    }
                                  	DefaultRow row = new DefaultRow(key, cells);
                                    cont.addRowToTable(row);
                    			}
                   				title = line.substring(ilast,i); 
                    			if (titles.contains(title)) {
                    				title = title + "-" + count;
                    			}
                    			for (int n=0; n<npharms; n++ ){
                    				d_values[n]= 0;
                    			}
                    		}
                    		titles.add(title);
                    	}
// Pharmacophore column
                    	else if ( ncol == -1 ) {
                    		ncol = pharms.lastIndexOf( line.substring(ilast,i) );
                     	}
// Population
                    	else {
                    		d_values[ncol] = Double.valueOf(line.substring(ilast,i)).doubleValue();
                    	}
                    	ilast = i+1;
                    }
                }
            }
        }
// Add last molecule         
		count ++;
		if ( !title.equals(" ") ) {
           	RowKey key = new RowKey(title);
               DataCell[] cells = new DataCell[npharms];
               for (int n=0; n<npharms; n++) {
               	cells[n] = new DoubleCell(d_values[n]);
               }
           	DefaultRow row = new DefaultRow(key, cells);
            cont.addRowToTable(row);
		}
 
        cont.close();
        return cont.getTable();
    }

    private BufferedDataTable readProfileFile(String outString, final ExecutionContext exec)
    throws Exception {

    	String cwdString = m_working.getStringValue();
     	int nocols=0;
    	File outFile = new File (cwdString, outString);

    	if (!(outFile.exists() && outFile.isFile())) {
    		warningReport();
    		throw new IllegalStateException("THINK didn't produce any output"
            + " at the specified location ('"
            + outFile.getAbsolutePath() + "')");
    	}
        BufferedReader in =  new BufferedReader(new FileReader(outFile));
        String line = " ";

        Map Headers = new HashMap();
        if ( (line = in.readLine()) != null) {
        	Headers.put("Header 1",line);      	
        }
        if ( (line = in.readLine()) != null) {
        	Headers.put("Header 2",line);      	
        }
        if ( (line = in.readLine()) != null) {
        	Headers.put("Header 3",line);      	
        }
    	DataColumnProperties ColProp = new DataColumnProperties( Headers );
        DataColumnSpec[] allColSpecs = new DataColumnSpec[1];
      	DataColumnSpecCreator colCreator =  new DataColumnSpecCreator("Population", DoubleCell.TYPE);
       	colCreator.setProperties(ColProp);
       	allColSpecs[0] = colCreator.createSpec();
        DataTableSpec outputSpec = new DataTableSpec(allColSpecs);
        BufferedDataContainer cont = exec.createDataContainer(outputSpec);

        HashSet<String> titles = new HashSet<String>();
        char chr='0';
        String title=" ";
        int count=0;
        double d_value=0.;
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();
            if ( !line.startsWith("#") ) {
            	int ilast = 0;
            	int ncol = -1;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else {
                        chr = ',';
                    }
                    if ( chr == ',' ) {
// Row ID: Pharm name
                    	if ( ilast == 0 ) {                 		
                   			count ++;
                			title = line.substring(ilast,i); 
                    		if (titles.contains(title)) {
                    			title = title + "-" + count;
                    		}
                    		titles.add(title);
                    	}
// Population
                    	else {
                    		d_value = Double.valueOf(line.substring(ilast,i)).doubleValue();
                    	}
                    	ilast = i+1;
                    }
                }
// Add row
        		RowKey key = new RowKey(title);
                DataCell[] cells = new DataCell[1];
               	cells[0] = new DoubleCell(d_value);
              	DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
            }
        }
  
        cont.close();
        return cont.getTable();
    }

    private BufferedDataTable readOutputFile(final ExecutionContext exec)
            throws Exception {

    	String cwdString = m_working.getStringValue();
    	String outString=null;
    	int nocols=0;
    	if (m_output.getStringValue().equals("3D coord & maps")) {
    		outString= "trainers_maps.sdf";
    		nocols = 3;
        }
    	else {
    		outString = "trainers.phc";
    		nocols = 2;
    	}
        File outFile = new File (cwdString, outString);

    	if (!(outFile.exists() && outFile.isFile())) {
    		warningReport();
    		throw new IllegalStateException("THINK didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
        DataColumnSpec[] allColSpecs = new DataColumnSpec[nocols];
        allColSpecs[0] = 
            new DataColumnSpecCreator("Molecule", SdfCell.TYPE).createSpec();
        allColSpecs[1] = 
            new DataColumnSpecCreator("Pharm-Count", DoubleCell.TYPE).createSpec();
    	if ( nocols > 2 ) {
    		allColSpecs[2] = 
            new DataColumnSpecCreator("Validation", DoubleCell.TYPE).createSpec();
    	}
  
        DataTableSpec outputSpec = new DataTableSpec(allColSpecs);
        BufferedDataContainer cont = exec.createDataContainer(outputSpec);
    
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        
        HashSet<String> titles = new HashSet<String>();
 
        String line;
        StringBuilder buf = new StringBuilder(4096);
        boolean titleRead = false;
        boolean countRead = false;
        boolean validRead = false;
        String title = "";
        int count = 1;
        double pharm_count=0.;
        double valid =0.;
        exec.setMessage("Reading molecule #" + count);
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();

            if (!titleRead) {
                title = line;
                if (title.length() == 0) {
                    title = "Mol " + count;
                }
                if (titles.contains(title)) {
                    title = title + "-" + count;
                }
                titles.add(title);
                titleRead = true;
            }
            else if ( countRead ) {
              pharm_count = Double.valueOf(line).doubleValue();
              countRead = false;
            }
            else if ( validRead ) {
                valid = Double.valueOf(line).doubleValue();
                validRead = false;
            }
            buf.append(line).append('\n');

            if (line.startsWith("$$$$")) {
                DataCell[] cells = new DataCell[nocols];
 
            	cells[0] = new SdfCell(buf.toString());
            	cells[1] = new DoubleCell(pharm_count);
                	if ( nocols > 2 ) {
               		cells[2] = new DoubleCell(valid);
               	}
              	RowKey key = new RowKey(title);
 
                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
                buf.delete(0, buf.length());
                titleRead = false;
                count++;
                exec.setMessage("Reading molecule #" + count);
             }
            else if (line.startsWith("> <Pharm-Count>")) {
            	countRead = true;
            }
            else if (line.startsWith("> <Validation>")) {
            	validRead = true;
            }
        }
        cont.close();
        in.close();

        return cont.getTable();
  
    }

    /**
     * @see org.knime.core.node.NodeModel#reset()
     */
    @Override
    protected void reset() {
        // TODO Code executed on reset.
        // Models build during execute are cleared here.
        // Also data handled in load/saveInternals will be erased here.
  /*  	 m_count.setIntValue(DEFAULT_COUNT);
         m_empty.setDoubleValue(DEFAULT_EMPTY);
         m_tolerance.setDoubleValue(DEFAULT_TOLERANCE);
         m_centres.setIntValue(DEFAULT_CENTRES);
         m_valid.setDoubleValue(DEFAULT_VALID);
         m_maps.setBooleanValue(DEFAULT_MAPS);
         m_smiles.setBooleanValue(DEFAULT_SMILES);
         m_working.setStringValue(DEFAULT_WORKING);
         m_single.setIntValue(DEFAULT_SINGLE);
         m_conjugated.setIntValue(DEFAULT_CONJUGATED);
         m_crowded.setIntValue(DEFAULT_CROWDED);
         m_alpha.setIntValue(DEFAULT_ALPHA);
         m_amide.setIntValue(DEFAULT_AMIDE);
         m_ring.setIntValue(DEFAULT_RING);
         m_generation.setStringValue(DEFAULT_GENERATION);
         m_samples.setIntValue(DEFAULT_SAMPLES);
         m_cpkratio.setDoubleValue(DEFAULT_CPKRATIO);
         m_timelimit.setDoubleValue(DEFAULT_TIMELIMIT);
         m_syslimit.setIntValue(DEFAULT_SYSLIMIT);
         m_bondlimit.setIntValue(DEFAULT_BONDLIMIT);
         */
    }

  
     /**
     * @see org.knime.core.node.NodeModel
     *      #saveSettingsTo(org.knime.core.node.NodeSettings)
     */
    @Override
    protected void saveSettingsTo(final NodeSettingsWO settings) {

        // TODO save user settings to the config object.
		
		m_count.saveSettingsTo(settings);
		m_empty.saveSettingsTo(settings);
		m_tolerance.saveSettingsTo(settings);
		m_centres.saveSettingsTo(settings);
		m_valid.saveSettingsTo(settings);
		m_output.saveSettingsTo(settings);
		m_smiles.saveSettingsTo(settings);
		m_working.saveSettingsTo(settings);
		m_single.saveSettingsTo(settings);
		m_conjugated.saveSettingsTo(settings);
		m_crowded.saveSettingsTo(settings);
		m_alpha.saveSettingsTo(settings);
		m_amide.saveSettingsTo(settings);
		m_ring.saveSettingsTo(settings);
		m_generation.saveSettingsTo(settings);
		m_samples.saveSettingsTo(settings);
		m_cpkratio.saveSettingsTo(settings);
		m_timelimit.saveSettingsTo(settings);
		m_syslimit.saveSettingsTo(settings);
		m_bondlimit.saveSettingsTo(settings);

    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #loadValidatedSettingsFrom(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void loadValidatedSettingsFrom(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO load (valid) settings from the config object.
        // It can be safely assumed that the settings are valided by the 
        // method below.
        
        m_count.loadSettingsFrom(settings);
        m_empty.loadSettingsFrom(settings);
        m_tolerance.loadSettingsFrom(settings);
        m_centres.loadSettingsFrom(settings);
        m_valid.loadSettingsFrom(settings);
        m_output.loadSettingsFrom(settings);
        m_smiles.loadSettingsFrom(settings);
        m_working.loadSettingsFrom(settings);
        m_single.loadSettingsFrom(settings);
        m_conjugated.loadSettingsFrom(settings);
        m_crowded.loadSettingsFrom(settings);
        m_alpha.loadSettingsFrom(settings);
        m_amide.loadSettingsFrom(settings);
        m_ring.loadSettingsFrom(settings);
        m_generation.loadSettingsFrom(settings);
        m_samples.loadSettingsFrom(settings);
        m_cpkratio.loadSettingsFrom(settings);
        m_timelimit.loadSettingsFrom(settings);
        m_syslimit.loadSettingsFrom(settings);
        m_bondlimit.loadSettingsFrom(settings);

    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #validateSettings(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void validateSettings(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO check if the settings could be applied to our model
        // e.g. if the count is in a certain range (which is ensured by the
        // SettingsModel).
        // Do not actually set any values of any member variables.

        m_count.validateSettings(settings);
        m_empty.validateSettings(settings);
        m_tolerance.validateSettings(settings);
        m_centres.validateSettings(settings);
        m_valid.validateSettings(settings);
        m_output.validateSettings(settings);
        m_smiles.validateSettings(settings);
        m_working.validateSettings(settings);
        m_single.validateSettings(settings);
        m_conjugated.validateSettings(settings);
        m_crowded.validateSettings(settings);
        m_alpha.validateSettings(settings);
        m_amide.validateSettings(settings);
        m_ring.validateSettings(settings);
        m_generation.validateSettings(settings);
        m_samples.validateSettings(settings);
        m_cpkratio.validateSettings(settings);
        m_timelimit.validateSettings(settings);
        m_syslimit.validateSettings(settings);
        m_bondlimit.validateSettings(settings);

    }
    
    /**
     * @see org.knime.core.node.NodeModel #loadInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void loadInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
        
		// TODO load internal data. 
		// Everything handed to output ports is loaded automatically (data
		// returned by the execute method, models loaded in loadModelContent,
		// and user settings set through loadSettingsFrom - is all taken care 
		// of). Load here only the other internals that need to be restored
		// (e.g. data used by the views).

    }
    
    /**
     * @see org.knime.core.node.NodeModel #saveInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void saveInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
       
       	// TODO save internal models. 
		// Everything written to output ports is saved automatically (data
		// returned by the execute method, models saved in the saveModelContent,
		// and user settings saved through saveSettingsTo - is all taken care 
		// of). Save here only the other internals that need to be preserved
		// (e.g. data used by the views).

    }

}
