package com.treweren.pharmacophores;

import java.awt.FlowLayout;

import org.knime.core.node.defaultnodesettings.DefaultNodeSettingsPane;
import org.knime.core.node.defaultnodesettings.DialogComponentBoolean;
import org.knime.core.node.defaultnodesettings.DialogComponentNumber;
import org.knime.core.node.defaultnodesettings.DialogComponentString;
import org.knime.core.node.defaultnodesettings.DialogComponentStringSelection;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelDoubleBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelString;

/**
 * <code>NodeDialog</code> for the "Pharmacophores" Node.
 * Compute most common pharmacophores for a set of active molecules
 *
 * This node dialog derives from {@link DefaultNodeSettingsPane} which allows
 * creation of a simple dialog with standard components. If you need a more 
 * complex dialog please derive directly from 
 * {@link org.knime.core.node.NodeDialogPane}.
 * 
 * @author Treweren Consultants
 */
public class PharmacophoresNodeDialog extends DefaultNodeSettingsPane {

    /**
     * New pane for configuring Pharmacophores node dialog.
     * This is just a suggestion to demonstrate possible default dialog
     * components.
     */
    protected PharmacophoresNodeDialog() {
        super();
        
        DialogComponentStringSelection c_output = new DialogComponentStringSelection(
                new SettingsModelString(
                        PharmacophoresNodeModel.CFGKEY_OUTPUT,
                        PharmacophoresNodeModel.DEFAULT_OUTPUT),
                        "Output", "3D coordinates", "3D coord & maps", "Profile", "Pharmacophores");
        addDialogComponent(c_output);
        c_output.setToolTipText("3D coordinates or 3D coord & maps are required for Search;  Profile is used by FocusedSet and DiverseSet; and Pharmacophores by general statistic nodes.");
        c_output.getComponentPanel().setLayout( new FlowLayout(2));

        DialogComponentNumber c_count = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        PharmacophoresNodeModel.CFGKEY_COUNT,
                        PharmacophoresNodeModel.DEFAULT_COUNT,
                        Integer.MIN_VALUE, Integer.MAX_VALUE),
                        "Limit for 3D coordinate pharmacophores", /*step*/ 10, /*componentwidth*/ 4);
        addDialogComponent(c_count);
        c_count.setToolTipText("The most common pharmacophores are retained");
        c_count.getComponentPanel().setLayout( new FlowLayout(2));
        
        DialogComponentNumber c_empty = new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                        PharmacophoresNodeModel.CFGKEY_EMPTY,
                        PharmacophoresNodeModel.DEFAULT_EMPTY, 0., 1. ),
                        "Ignore populations below", .05, 5 );
        addDialogComponent(c_empty );
        c_empty.setToolTipText("For a set of active molecules, pharmacophores with smaller populations are unlikely to be associated with activity");
        c_empty.getComponentPanel().setLayout( new FlowLayout(2));
 
        DialogComponentNumber c_centres = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                    PharmacophoresNodeModel.CFGKEY_CENTRES,
                    PharmacophoresNodeModel.DEFAULT_CENTRES,
                    3, 4),
                    "Number of pharmacophore centres", /*step*/ 1, /*componentwidth*/ 1);
        addDialogComponent(c_centres);
        c_centres.setToolTipText("The number of interaction centres in the pharmacophore");
        c_centres.getComponentPanel().setLayout( new FlowLayout(2));

        DialogComponentNumber c_tolerance = new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                        PharmacophoresNodeModel.CFGKEY_TOLERANCE,
                        PharmacophoresNodeModel.DEFAULT_TOLERANCE, 0., 1. ),
                        "Distance tolerance", .05, 5 );
        
        addDialogComponent(c_tolerance);
        c_tolerance.setToolTipText("Set the maximum separation of the interaction centre from the matched atom");
        c_tolerance.getComponentPanel().setLayout( new FlowLayout(2));
 
        DialogComponentNumber c_valid = new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                        PharmacophoresNodeModel.CFGKEY_VALID,
                        PharmacophoresNodeModel.DEFAULT_VALID, 0., 100. ),
                        "Minimum Validation % for maps", 5., 5 );
        addDialogComponent(c_valid );
        c_valid.setToolTipText("To reject pharmacophores which are exhibited in fewer than the specified percentage of the molecules - can arise due to different binding modes");
        c_valid.getComponentPanel().setLayout( new FlowLayout(2));

       createNewTab("Bond rotations");
        
       DialogComponentNumber c_single = new DialogComponentNumber(
               new SettingsModelIntegerBounded(
                       PharmacophoresNodeModel.CFGKEY_SINGLE,
                       PharmacophoresNodeModel.DEFAULT_SINGLE,
                       0, Integer.MAX_VALUE),
                       "Single bond increments", /*step*/ 1, /*componentwidth*/ 2);
       addDialogComponent(c_single);
       c_single.setToolTipText("Single bonds are incremented by 360/n degrees where n is the number of steps (0=no rotation)");
       c_single.getComponentPanel().setLayout( new FlowLayout(2));
       
       DialogComponentNumber c_conjugated = new DialogComponentNumber(
               new SettingsModelIntegerBounded(
                       PharmacophoresNodeModel.CFGKEY_CONJUGATED,
                       PharmacophoresNodeModel.DEFAULT_CONJUGATED,
                       0, Integer.MAX_VALUE),
                       "Conjugated bond increments", /*step*/ 1, /*componentwidth*/ 2);
       addDialogComponent(c_conjugated);
       c_conjugated.setToolTipText("Conjugated (csp2-csp2) bonds are incremented by 360/n degrees where n is the number of steps");
       c_conjugated.getComponentPanel().setLayout( new FlowLayout(2));

       DialogComponentNumber c_crowded = new DialogComponentNumber(
               new SettingsModelIntegerBounded(
                       PharmacophoresNodeModel.CFGKEY_CROWDED,
                       PharmacophoresNodeModel.DEFAULT_CROWDED,
                       0, Integer.MAX_VALUE),
                       "Crowded bond increments", /*step*/ 1, /*componentwidth*/ 2);
       addDialogComponent(c_crowded);
       c_crowded.setToolTipText("Crowded (csp2-csp2) bonds are incremented by 360/n degrees where n is the number of steps");
       c_crowded.getComponentPanel().setLayout( new FlowLayout(2));

       DialogComponentNumber c_alpha = new DialogComponentNumber(
               new SettingsModelIntegerBounded(
                   PharmacophoresNodeModel.CFGKEY_ALPHA,
                   PharmacophoresNodeModel.DEFAULT_ALPHA,
                   0, Integer.MAX_VALUE),
                   "Alpha bond increments", /*step*/ 1, /*componentwidth*/ 2);
       addDialogComponent(c_alpha);
       c_alpha.setToolTipText("Alpha (csp2-csp3) bonds are incremented by 360/n degrees where n is the number of steps");
       c_alpha.getComponentPanel().setLayout( new FlowLayout(2));

       DialogComponentNumber c_amide = new DialogComponentNumber(
               new SettingsModelIntegerBounded(
                       PharmacophoresNodeModel.CFGKEY_AMIDE,
                       PharmacophoresNodeModel.DEFAULT_AMIDE,
                       0, Integer.MAX_VALUE),
                       "Amide bond increments", /*step*/ 1, /*componentwidth*/ 2);
       
       addDialogComponent(c_amide);
       c_amide.setToolTipText("Amide bonds are incremented by 360/n degrees where n is the number of steps");
       c_amide.getComponentPanel().setLayout( new FlowLayout(2));

       DialogComponentNumber c_ring = new DialogComponentNumber(
               new SettingsModelIntegerBounded(
                       PharmacophoresNodeModel.CFGKEY_RING,
                       PharmacophoresNodeModel.DEFAULT_RING,
                       0, Integer.MAX_VALUE),
                       "Ring bond increments", /*step*/ 1, /*componentwidth*/ 2); 
       addDialogComponent(c_ring);
       c_ring.setToolTipText("Ring bonds are incremented by 360/n degrees where n is the number of steps");
       c_ring.getComponentPanel().setLayout( new FlowLayout(2));


        createNewTab("Advanced");
        DialogComponentStringSelection c_generation = new DialogComponentStringSelection(
                new SettingsModelString(
                        PharmacophoresNodeModel.CFGKEY_GENERATION,
                        PharmacophoresNodeModel.DEFAULT_GENERATION),
                        "Conformer Mode", "Systematic", "Sample", "Random");
        addDialogComponent(c_generation);
        c_generation.setToolTipText("For the most flexible molecules sample or random modes take less time");
        c_generation.getComponentPanel().setLayout( new FlowLayout(2));

        DialogComponentNumber c_samples = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        PharmacophoresNodeModel.CFGKEY_SAMPLES,
                        PharmacophoresNodeModel.DEFAULT_SAMPLES,
                        1, Integer.MAX_VALUE),
                        "Number of sample/random conformations", /*step*/ 100, /*componentwidth*/ 4);
        addDialogComponent(c_samples);
        c_samples.setToolTipText("The number of conformations should be sufficiently large to be representative");
        c_samples.getComponentPanel().setLayout( new FlowLayout(2));
        
        DialogComponentNumber c_cpkratio = new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                        PharmacophoresNodeModel.CFGKEY_CPKRATIO,
                        PharmacophoresNodeModel.DEFAULT_CPKRATIO, 0., 5. ),
                        "Contact to VdW radius ratio", .1, 3 );
        
        addDialogComponent(c_cpkratio);
        c_cpkratio.setToolTipText("Conformers with separation distances between distances less than the sum of the contact radii are skipped");
        c_cpkratio.getComponentPanel().setLayout( new FlowLayout(2));
        
        DialogComponentNumber c_timelimit = new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                        PharmacophoresNodeModel.CFGKEY_TIMELIMIT,
                        PharmacophoresNodeModel.DEFAULT_TIMELIMIT, 0., 1440. ),
                        "Molecule time limit (m)", 15., 4 );
        addDialogComponent(c_timelimit );
        c_timelimit.setToolTipText("In order to progress other molecules, the search for a molecule which takes longer than the time limit is abandoned");
        c_timelimit.getComponentPanel().setLayout( new FlowLayout(2));

        createNewGroup("Systematic limits");
        DialogComponentNumber c_syslimit = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        PharmacophoresNodeModel.CFGKEY_SYSLIMIT,
                        PharmacophoresNodeModel.DEFAULT_SYSLIMIT,
                        0, Integer.MAX_VALUE),
                        "Number of conformations", /*step*/ 100, /*componentwidth*/ 4);
        addDialogComponent(c_syslimit);
        c_syslimit.setToolTipText("If the estimated number of conformers to be generated exceeds the systematic limit, sampling is used instead");
        c_syslimit.getComponentPanel().setLayout( new FlowLayout(2));
        
        DialogComponentNumber c_bondlimit = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        PharmacophoresNodeModel.CFGKEY_BONDLIMIT,
                        PharmacophoresNodeModel.DEFAULT_BONDLIMIT,
                        0, Integer.MAX_VALUE),
                        "Number of rotating bonds", /*step*/ 1, /*componentwidth*/ 2);
        addDialogComponent(c_bondlimit);
        c_bondlimit.setToolTipText("If the estimated number of rotatable bonds exceeds the systematic limit, sampling is used instead");
        c_bondlimit.getComponentPanel().setLayout( new FlowLayout(2));

        closeCurrentGroup();

        createNewTab("THINK setup");         

        DialogComponentBoolean c_column = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        PharmacophoresNodeModel.CFGKEY_SMILES,
                        PharmacophoresNodeModel.DEFAULT_SMILES),
                        "Use SMILES in preference to SD column" );
        addDialogComponent(c_column);
        c_column.getComponentPanel().setLayout( new FlowLayout(0));
        c_column.setToolTipText("The molecule data is transferred to THINK using SD or SMILES format depending on this setting");

        DialogComponentString c_working = new DialogComponentString(
                new SettingsModelString (
                        PharmacophoresNodeModel.CFGKEY_WORKING,
                        PharmacophoresNodeModel.DEFAULT_WORKING),
                        "Working folder" );
        addDialogComponent(c_working);
        c_working.getComponentPanel().setLayout( new FlowLayout(2));
        c_working.setToolTipText("The temporary files used to transfer the data to and from THINK are placed in this folder");
       
                   
    }
}
