package com.treweren.search;

import java.awt.FlowLayout;
import javax.swing.JFileChooser;

import org.knime.core.node.defaultnodesettings.DefaultNodeSettingsPane;
import org.knime.core.node.defaultnodesettings.DialogComponentBoolean;
import org.knime.core.node.defaultnodesettings.DialogComponentFileChooser;
import org.knime.core.node.defaultnodesettings.DialogComponentNumber;
import org.knime.core.node.defaultnodesettings.DialogComponentString;
import org.knime.core.node.defaultnodesettings.DialogComponentStringSelection;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelDoubleBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelString;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;



/**
 * <code>NodeDialog</code> for the "Search" Node.
 * Search
 *
 * This node dialog derives from {@link DefaultNodeSettingsPane} which allows
 * creation of a simple dialog with standard components. If you need a more 
 * complex dialog please derive directly from 
 * {@link org.knime.core.node.NodeDialogPane}.
 * 
 * @author Treweren Consultants
 */
public class SearchNodeDialog extends DefaultNodeSettingsPane {

    /**
     * New pane for configuring Search node dialog.
     * This is just a suggestion to demonstrate possible default dialog
     * components.
     */
    protected SearchNodeDialog() {
        super();
        DialogComponentFileChooser c_search = new DialogComponentFileChooser(
                new SettingsModelString (
                        SearchNodeModel.CFGKEY_SEARCH,
                        SearchNodeModel.DEFAULT_SEARCH ),
                        "Search file", JFileChooser.OPEN_DIALOG, "smi", "csv", "smiles", "sdf" );
        addDialogComponent(c_search);
        c_search.setToolTipText("The molecules being searched are in this file");
        c_search.setBorderTitle("Search molecules in file");

        DialogComponentStringSelection c_mode = new DialogComponentStringSelection(
                new SettingsModelString(
                        SearchNodeModel.CFGKEY_MODE,
                        SearchNodeModel.DEFAULT_MODE),
                        "Search mode", "Exact", "2D", "3D", "Pharmacophore");
        addDialogComponent(c_mode);
        c_mode.setToolTipText("Choose the type of search to be performed");
        c_mode.getComponentPanel().setLayout( new FlowLayout(2));

        DialogComponentBoolean c_filter = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        SearchNodeModel.CFGKEY_FILTER,
                        SearchNodeModel.DEFAULT_FILTER),
                        "Apply drug-like filters" );
        addDialogComponent(c_filter);
        c_filter.setToolTipText("Eliminates molecules which have properties which are not drug-like");
        c_filter.getComponentPanel().setLayout( new FlowLayout(0));
        
        final DialogComponentBoolean c_parent = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        SearchNodeModel.CFGKEY_PARENT,
                        SearchNodeModel.DEFAULT_PARENT),
                        "Convert salts to parent form" );
        addDialogComponent(c_parent);
        c_parent.setToolTipText("Converts molecules to parent(neutral) form and eliminates counter-ions(salts)");
        c_parent.getComponentPanel().setLayout( new FlowLayout(0));
     
        createNewGroup("3D and Pharmacophore searches only");
        
        DialogComponentNumber c_tolerance = new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                        SearchNodeModel.CFGKEY_TOLERANCE,
                        SearchNodeModel.DEFAULT_TOLERANCE, 0., 1. ),
                        "Distance tolerance", .05, 5 );
        
        addDialogComponent(c_tolerance);
        c_tolerance.setToolTipText("Set the maximum separation of the interaction centre from the matched atom");
        c_tolerance.getComponentPanel().setLayout( new FlowLayout(2));
 
        DialogComponentNumber c_count = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        SearchNodeModel.CFGKEY_COUNT,
                        SearchNodeModel.DEFAULT_COUNT,
                        Integer.MIN_VALUE, Integer.MAX_VALUE),
                        "Pharmacophore search map number (0=all)", /*step*/ 1, /*componentwidth*/ 2);
        
        addDialogComponent(c_count);
        c_count.setToolTipText("Choose the map constraint number when appropriate");
        c_count.getComponentPanel().setLayout( new FlowLayout(2));

        closeCurrentGroup();
 
        createNewTab("Bond rotations");
        DialogComponentNumber c_single = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        SearchNodeModel.CFGKEY_SINGLE,
                        SearchNodeModel.DEFAULT_SINGLE,
                        0, Integer.MAX_VALUE),
                        "Single bond increments", /*step*/ 1, /*componentwidth*/ 2);
        addDialogComponent(c_single);
        c_single.setToolTipText("Single bonds are incremented by 360/n degrees where n is the number of steps (0=no rotation)");
        c_single.getComponentPanel().setLayout( new FlowLayout(2));
        
        DialogComponentNumber c_conjugated = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        SearchNodeModel.CFGKEY_CONJUGATED,
                        SearchNodeModel.DEFAULT_CONJUGATED,
                        0, Integer.MAX_VALUE),
                        "Conjugated bond increments", /*step*/ 1, /*componentwidth*/ 2);
        addDialogComponent(c_conjugated);
        c_conjugated.setToolTipText("Conjugated (csp2-csp2) bonds are incremented by 360/n degrees where n is the number of steps");
        c_conjugated.getComponentPanel().setLayout( new FlowLayout(2));

        DialogComponentNumber c_crowded = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        SearchNodeModel.CFGKEY_CROWDED,
                        SearchNodeModel.DEFAULT_CROWDED,
                        0, Integer.MAX_VALUE),
                        "Crowded bond increments", /*step*/ 1, /*componentwidth*/ 2);
        addDialogComponent(c_crowded);
        c_crowded.setToolTipText("Crowded (csp2-csp2) bonds are incremented by 360/n degrees where n is the number of steps");
        c_crowded.getComponentPanel().setLayout( new FlowLayout(2));
 
        DialogComponentNumber c_alpha = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                    SearchNodeModel.CFGKEY_ALPHA,
                    SearchNodeModel.DEFAULT_ALPHA,
                    0, Integer.MAX_VALUE),
                    "Alpha bond increments", /*step*/ 1, /*componentwidth*/ 2);
        addDialogComponent(c_alpha);
        c_alpha.setToolTipText("Alpha (csp2-csp3) bonds are incremented by 360/n degrees where n is the number of steps");
        c_alpha.getComponentPanel().setLayout( new FlowLayout(2));

        DialogComponentNumber c_amide = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        SearchNodeModel.CFGKEY_AMIDE,
                        SearchNodeModel.DEFAULT_AMIDE,
                        0, Integer.MAX_VALUE),
                        "Amide bond increments", /*step*/ 1, /*componentwidth*/ 2);
        
        addDialogComponent(c_amide);
        c_amide.setToolTipText("Amide bonds are incremented by 360/n degrees where n is the number of steps");
        c_amide.getComponentPanel().setLayout( new FlowLayout(2));

        DialogComponentNumber c_ring = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        SearchNodeModel.CFGKEY_RING,
                        SearchNodeModel.DEFAULT_RING,
                        0, Integer.MAX_VALUE),
                        "Ring bond increments", /*step*/ 1, /*componentwidth*/ 2); 
        addDialogComponent(c_ring);
        c_ring.setToolTipText("Ring bonds are incremented by 360/n degrees where n is the number of steps");
        c_ring.getComponentPanel().setLayout( new FlowLayout(2));

     
        createNewTab("Advanced");
        
        createNewGroup("3D and Pharmacophore searches only");
        DialogComponentStringSelection c_generation = new DialogComponentStringSelection(
                new SettingsModelString(
                        SearchNodeModel.CFGKEY_GENERATION,
                        SearchNodeModel.DEFAULT_GENERATION),
                        "Conformer Mode", "Systematic", "Sample", "Random");
        addDialogComponent(c_generation);
        c_generation.setToolTipText("For the most flexible molecules sample or random modes take less time");
        c_generation.getComponentPanel().setLayout( new FlowLayout(2));

        DialogComponentNumber c_samples = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        SearchNodeModel.CFGKEY_SAMPLES,
                        SearchNodeModel.DEFAULT_SAMPLES,
                        1, Integer.MAX_VALUE),
                        "Number of sample/random conformations", /*step*/ 100, /*componentwidth*/ 4);
        addDialogComponent(c_samples);
        c_samples.setToolTipText("The number of conformations should be sufficiently large to be representative");
        c_samples.getComponentPanel().setLayout( new FlowLayout(2));
        
        DialogComponentNumber c_cpkratio = new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                        SearchNodeModel.CFGKEY_CPKRATIO,
                        SearchNodeModel.DEFAULT_CPKRATIO, 0., 5. ),
                        "Contact to VdW radius ratio", .1, 3 );
        
        addDialogComponent(c_cpkratio);
        c_cpkratio.setToolTipText("Conformers with separation distances between distances less than the sum of the contact radii are skipped");
        c_cpkratio.getComponentPanel().setLayout( new FlowLayout(2));
        
        DialogComponentNumber c_timelimit = new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                        SearchNodeModel.CFGKEY_TIMELIMIT,
                        SearchNodeModel.DEFAULT_TIMELIMIT, 0., 1440. ),
                        "Molecule time limit (m)", 15., 4 );
        addDialogComponent(c_timelimit );
        c_timelimit.setToolTipText("In order to progress other molecules, the search for a molecule which takes longer than the time limit is abandoned");
        c_timelimit.getComponentPanel().setLayout( new FlowLayout(2));
 
        DialogComponentBoolean c_conformers = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        SearchNodeModel.CFGKEY_CONFORMERS,
                        SearchNodeModel.DEFAULT_CONFORMERS),
                        "Retain all conformers (not best)" );
        addDialogComponent(c_conformers);
        c_conformers.setToolTipText("Many queries have more than one conformation which match");
        c_conformers.getComponentPanel().setLayout( new FlowLayout(0));
        
        closeCurrentGroup();
      
        createNewGroup("Systematic limits");
        DialogComponentNumber c_syslimit = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        SearchNodeModel.CFGKEY_SYSLIMIT,
                        SearchNodeModel.DEFAULT_SYSLIMIT,
                        0, Integer.MAX_VALUE),
                        "Number of conformations", /*step*/ 100, /*componentwidth*/ 4);
        addDialogComponent(c_syslimit);
        c_syslimit.setToolTipText("If the estimated number of conformers to be generated exceeds the systematic limit, sampling is used instead");
        c_syslimit.getComponentPanel().setLayout( new FlowLayout(2));
        
        DialogComponentNumber c_bondlimit = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        SearchNodeModel.CFGKEY_BONDLIMIT,
                        SearchNodeModel.DEFAULT_BONDLIMIT,
                        0, Integer.MAX_VALUE),
                        "Number of rotating bonds", /*step*/ 1, /*componentwidth*/ 2);
        addDialogComponent(c_bondlimit);
        c_bondlimit.setToolTipText("If the estimated number of rotatable bonds exceeds the systematic limit, sampling is used instead");
        c_bondlimit.getComponentPanel().setLayout( new FlowLayout(2));

        closeCurrentGroup();
        
        createNewTab("THINK setup");         

        DialogComponentBoolean c_column = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        SearchNodeModel.CFGKEY_SMILES,
                        SearchNodeModel.DEFAULT_SMILES),
                        "Use SMILES in preference to SD column" );
        addDialogComponent(c_column);
        c_column.getComponentPanel().setLayout( new FlowLayout(0));
        c_column.setToolTipText("The molecule data is transferred to THINK using SD or SMILES format depending on this setting");
        
        DialogComponentBoolean c_row = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        SearchNodeModel.CFGKEY_ROW1,
                        SearchNodeModel.DEFAULT_ROW1),
                        "Use first row/molecule (not all)" );
        
        addDialogComponent(c_row);
        c_row.getComponentPanel().setLayout( new FlowLayout(0));
        c_row.setToolTipText("The data transferred may be limited to the first row using this control");
 
        DialogComponentBoolean c_restart = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        SearchNodeModel.CFGKEY_RESTART,
                        SearchNodeModel.DEFAULT_RESTART),
                        "Restart search (instead of continuing previous search)" );
        addDialogComponent(c_restart);
        c_restart.setToolTipText("Check point files (searchn.dat) are deleted before THINK restarts");
        c_restart.getComponentPanel().setLayout( new FlowLayout(0));

         
        DialogComponentNumber c_process = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        SearchNodeModel.CFGKEY_PROCESS,
                        SearchNodeModel.DEFAULT_PROCESS,
                        1, 36 ),
                        "Maximum number of THINK processes", /*step*/ 1, /*componentwidth*/ 2);
        
        addDialogComponent(c_process);
        c_process.getComponentPanel().setLayout( new FlowLayout(2));
        c_process.setToolTipText("The search is divided between the THINK processes");
  
        DialogComponentString c_working = new DialogComponentString(
                new SettingsModelString (
                        SearchNodeModel.CFGKEY_WORKING,
                        SearchNodeModel.DEFAULT_WORKING),
                        "Working folder" );
        addDialogComponent(c_working);
        c_working.getComponentPanel().setLayout( new FlowLayout(2));
        c_working.setToolTipText("The temporary files used to transfer the data to and from THINK are placed in this folder");
        
                     
    }
}
