package com.treweren.fingerprint;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.util.HashSet;
import java.io.BufferedReader;
import java.io.FileReader;

import org.knime.base.node.io.filereader.FileAnalyzer;
import org.knime.base.node.io.filereader.FileReaderNodeSettings;
import org.knime.base.node.io.filereader.FileReaderSettings;
import org.knime.base.node.io.filereader.FileTable;
import org.knime.base.node.io.filereader.SettingsStatus;
import org.knime.chem.types.SdfValue;
import org.knime.chem.types.SmilesValue;
import org.knime.core.data.DataCell;
import org.knime.core.data.DataColumnSpec;
import org.knime.core.data.DataColumnSpecCreator;
import org.knime.core.data.DataRow;
import org.knime.core.data.DataTableSpec;
import org.knime.core.data.RowKey;
import org.knime.core.data.def.DefaultRow;
import org.knime.core.data.def.DoubleCell;
import org.knime.core.data.def.IntCell;
import org.knime.core.data.def.StringCell;
import org.knime.core.node.BufferedDataContainer;
import org.knime.core.node.BufferedDataTable;
import org.knime.core.node.CanceledExecutionException;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelString;
import org.knime.core.node.ExecutionContext;
import org.knime.core.node.ExecutionMonitor;
import org.knime.core.node.InvalidSettingsException;
import org.knime.core.node.NodeLogger;
import org.knime.core.node.NodeModel;
import org.knime.core.node.NodeSettingsRO;
import org.knime.core.node.NodeSettingsWO;

import org.knime.core.data.RowKey;
import org.knime.core.node.BufferedDataContainer;
import org.knime.core.util.MutableBoolean;
import org.knime.core.data.DataColumnSpecCreator;
import org.knime.chem.types.SmilesCell;
import org.knime.core.data.def.DoubleCell;
import org.knime.core.data.def.IntCell;
import org.knime.core.data.def.StringCell;
import org.knime.core.data.def.DefaultRow;
import org.knime.core.data.DataType;


/**
 * This is the model implementation of Fingerprint.
 * Calculation of functional group fingerprint
 *
 * @author Treweren Consultants
 */
public class FingerprintNodeModel extends NodeModel {
    
    // the logger instance
    private static final NodeLogger logger = NodeLogger
            .getLogger(FingerprintNodeModel.class);
    private static int n_Column=0;
    private static String m_sdfColumn = null;
    private static String m_smilesColumn = null;
    private static String m_stringColumn = null;
    private static final int CANCEL_CHECK_INTERVAL = 1000;
        
    /** the settings key which is used to retrieve and 
        store the settings (from the dialog or from a settings file)    
       (package visibility to be usable from the dialog). */
	static final String CFGKEY_SMILES = "Smiles";
    static final String CFGKEY_FPMODE = "FingerprintMode";
	static final String CFGKEY_WORKING = "Working";

	/** initial default SMILES usage. */
	static final Boolean DEFAULT_SMILES = false;
	static final String DEFAULT_WORKING =  System.getenv("THINK_WORKING");
 
	/** initial default fingerprint mode. */
	static final Boolean DEFAULT_FPMODE = false;

	// example value: the models count variable filled from the dialog 
	// and used in the models execution method. The default components of the
	// dialog work with "SettingsModels".
	private final SettingsModelBoolean m_smiles =
		new SettingsModelBoolean(FingerprintNodeModel.CFGKEY_SMILES,
                    FingerprintNodeModel.DEFAULT_SMILES );
	private final SettingsModelBoolean m_fingerprint =
		new SettingsModelBoolean(FingerprintNodeModel.CFGKEY_FPMODE,
                    FingerprintNodeModel.DEFAULT_FPMODE );
	private final SettingsModelString m_working =
		new SettingsModelString(FingerprintNodeModel.CFGKEY_WORKING, FingerprintNodeModel.DEFAULT_WORKING );

	

    /**
     * Constructor for the node model.
     */
    protected FingerprintNodeModel() {
    
        // TODO one incoming port and one outgoing port is assumed
        super(1, 1);
    }
    @Override
    protected DataTableSpec[] configure(final DataTableSpec[] inSpecs) throws InvalidSettingsException {

        if ( System.getenv("THINK_EXEC") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_EXEC is unset - It should be set to the folder containing the THINK software.");       	  		
    	}
        if ( System.getenv("THINK_WORKING") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_WORKING is not set to the folder in which working files will be created.");       	  		
    	}

        DataTableSpec in = inSpecs[0];
        n_Column = columnSelect (in);
        return new DataTableSpec[] {null};
    }
    protected int columnSelect ( final DataTableSpec in ) throws  InvalidSettingsException {

        StringBuilder warningMessage = new StringBuilder();
        int n_column=0;
        int sdfColCount = 0;
        int n_sdf=0;
        int smilesColCount = 0;
        int n_smiles=0;
        int stringColCount = 0;
        int n_string=0;
        for (int i = 0; i < in.getNumColumns(); i++) {
            DataColumnSpec s = in.getColumnSpec(i);
            if (s.getType().isCompatible(SdfValue.class)) {
                if (m_sdfColumn == null) {
                    m_sdfColumn = in.getColumnSpec(i).getName();
                    n_sdf = i;
                }
                sdfColCount++;
            }
            else if ( s.getType().isCompatible(SmilesValue.class)) {
            	if (m_smilesColumn == null ) {
             		m_smilesColumn = in.getColumnSpec(i).getName();
             		n_smiles=i;
              	}
               	smilesColCount++;
            }
            else if ( in.getColumnSpec(i).getName().equalsIgnoreCase("smiles"))  {
            	if (m_stringColumn == null ) {
             		m_stringColumn = in.getColumnSpec(i).getName();
             		n_string=i;
              	}
               	stringColCount++;
            }
        }
        if ( smilesColCount == 0 & sdfColCount == 0 & stringColCount == 0) {
            throw new InvalidSettingsException(
                    "No column with Smiles or SD file compatible type");       	
        }
        else if ( ( m_smiles.getBooleanValue() | sdfColCount == 0 ) & smilesColCount > 0 ) {
        	m_sdfColumn = null;
        	m_stringColumn = null;
        	n_column = n_smiles;
        	if (smilesColCount > 1) {
                warningMessage.append("More than one Smiles compatible column in " 
                        + "input, using column \"" + m_smilesColumn + "\".");
        	}
        }
        else if ( sdfColCount > 0 ) {
           	m_smilesColumn = null;
        	m_stringColumn = null;
        	n_column = n_sdf;
        	if (sdfColCount > 1) {
                warningMessage.append("More than one SDF compatible column in " 
                        + "input, using column \"" + m_sdfColumn + "\".");
        	}
        }
        else if ( stringColCount > 0 ) {
        	m_smilesColumn = null;
        	m_sdfColumn = null;
        	n_column = n_string;
           	if (stringColCount > 1) {
                warningMessage.append("More than one Smiles string compatible column in " 
                        + "input, using column \"" + m_stringColumn + "\".");
           	}
        }
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
        return n_column;
    }

 
    /**
     * @see org.knime.core.node.NodeModel #execute(BufferedDataTable[],
     *      ExecutionContext)
     */
    @Override
    protected BufferedDataTable[] execute(final BufferedDataTable[] inData,
            final ExecutionContext exec) throws Exception {

        notifyViews(null);

        final String cwdString = m_working.getStringValue();
        // clean up 
        File outputReport = new File(cwdString, "output.log"); 
        outputReport.delete();
        File outputFile = new File(cwdString, "fingerprint.key"); 
        outputFile.delete();
        BufferedDataTable in = inData[0];

        if ( n_Column == 0 ) {
        	DataTableSpec InSpec = in.getDataTableSpec();
        	n_Column = columnSelect(InSpec);
        }

 
        // write out input file
        String fileString = null; 
        if ( m_sdfColumn == null ) {
            fileString = "fingerprint.smiles"; 
        }
        else {
            fileString = "fingerprint.sdf"; 
        }
        File inFile = new File(cwdString, fileString); 
        BufferedWriter outWriter = new BufferedWriter(new FileWriter(inFile));
        int colIndex;
        if ( m_sdfColumn != null ) {
        	colIndex = in.getDataTableSpec().findColumnIndex(m_sdfColumn);
        }
        else if ( m_smilesColumn != null ) {
        	colIndex = in.getDataTableSpec().findColumnIndex(m_smilesColumn);        
        }
        else {
            colIndex = in.getDataTableSpec().findColumnIndex(m_stringColumn);        
        }
        	  
        final double count = in.getRowCount(); // floating point operations
        int i = 0; 
        int missingCount = 0;
        for (DataRow r : in) {
            exec.checkCanceled();
            DataCell c = r.getCell(colIndex);
            if (c.isMissing()) {
                missingCount++;
            } 
            else if ( m_sdfColumn != null ) {
                SdfValue v = (SdfValue)c;
                String toString = v.toString();
                outWriter.write(toString);
                if (!toString.trim().endsWith("$$$$")) {
                    outWriter.newLine();
                    outWriter.append("$$$$");
                    outWriter.newLine();
                }
            }
            else if ( m_smilesColumn != null ) {
                SmilesValue v = (SmilesValue)c;
                String toString = v.toString() + " " + r.getKey();
                outWriter.write(toString);        	
                outWriter.newLine();
            }
            else {
            	String toString = c.toString() + " " + r.getKey();
            	outWriter.write(toString);        	
            	outWriter.newLine();
            }
            i++;
        }

        outWriter.close();
        if (missingCount > 0) {
            setWarningMessage("Skipped " + missingCount 
                    + " row(s) because of missing values");
        }

        // execute THINK
        File cwdFile = new File (cwdString);
        int exitVal;
        try {
            exec.setProgress(0., "Starting THINK");
            Runtime rt = Runtime.getRuntime();
            // prepare the command string
            String cmdString = System.getenv("THINK_EXEC") + "think THINK_EXEC:fingerprint.log output.log " + fileString;
            // Go!
            logger.info("THINK command line: '" + cmdString + "'");
            final Process proc = rt.exec(cmdString, null, cwdFile);
            final MutableBoolean procDone = new MutableBoolean(false);
            new Thread(new Runnable() {
                public void run() {
                    synchronized (procDone) {
                        while (!procDone.booleanValue()) {
                            try {
                                exec.checkCanceled();
                            } catch (CanceledExecutionException cee) {
                                // blow away the running external process
                                proc.destroy();
                                return;
                            }
                            try {
                               	procDone.wait(CANCEL_CHECK_INTERVAL);
                                // Read progress file and set progress value       
                               	double d= 0.;
                                File progressReport = new File(cwdString, "progress0.dat");
                                if ( progressReport.exists() && progressReport.isFile() ) {
                               		if ( progressReport.length() > 0 ) {
                               			BufferedReader in = new BufferedReader(new FileReader(progressReport));
                               			String line;
                               			if ( (line = in.readLine()) != null) {
                               			  d = Double.valueOf(line).doubleValue();
                               			}               
                              			if ( (line = in.readLine()) == null) {
                              				line = "Running THINK";
                              			}
                               			in.close();
                                        exec.setProgress( d, line.toString());
                               		}
                                }
                            } catch (InterruptedException e) {
                                // do nothing
                            } catch ( FileNotFoundException e) {
                                // do nothing
                            } catch ( IOException e) {
                                // do nothing
                            }
                        }
                    }
                }

            }).start();
           

            // wait until the external process finishes.
            exitVal = proc.waitFor();

            synchronized (procDone) {
                // this should terminate the check cancel thread
                procDone.setValue(true);
            }

            exec.checkCanceled();

            exec.setProgress("Wrapping up");
            logger.info("THINK terminated with exit code: " + exitVal);
        } catch (Throwable t) {
            logger.error("THINK failed (with exception)", t);
            warningReport();
            throw new Exception(t);
        }

        if (exitVal != 0) {
            // before we return barfing, we save the output in the failing list
        	warningReport();
            throw new IllegalStateException("THINK failed (error code "
                    + exitVal + ")");
        }

        exec.setProgress("Creating KNIME tables");
        BufferedDataTable outTable = readOutputFile(exec);

        return new BufferedDataTable[]{outTable};

    }

    private void warningReport( ) throws Exception {

    	StringBuilder warningMessage = new StringBuilder();
        String cwdString = m_working.getStringValue();
        File outputReport = new File(cwdString, "output.log"); 

        if ( outputReport.exists() && outputReport.isFile() ) {
       		if ( outputReport.length() > 0 ) {
       			BufferedReader in = new BufferedReader(new FileReader(outputReport));
       			String line;
       			while ((line = in.readLine()) != null) {
       				warningMessage.append( line + "\n");
       			}               
       			in.close();
       		}
       		else { 
                warningMessage.append( "Report from THINK is blank");      	     			
       		}
        }
       	else {
            warningMessage.append( "No report from THINK");      	
       	}	
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
   }

    private BufferedDataTable readOutputFile(final ExecutionContext exec) throws Exception { 

        String cwdString =  m_working.getStringValue();
        File outFile = new File (cwdString, "fingerprint.key");
        if (!(outFile.exists() && outFile.isFile())) {
        	warningReport();
            throw new IllegalStateException("THINK didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
        HashSet<String> titles = new HashSet<String>();
        String title;
        String line;
        boolean headerRead = false;
        DataColumnSpec[] allColSpecs;
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        DataTableSpec outputSpec;
        BufferedDataContainer cont =null;
        RowKey key =null;
        int count =0;
        int rowcount =0;
        char chr='0';
        int keys = 0;
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();
// Extract column headers
            if ( !headerRead ) {
                headerRead = true;
                count =0;
                for ( int i=0; i<line.length(); i++ ) {
                    if ( line.charAt(i) == ',' ) count ++;
                }
                if (m_fingerprint.getBooleanValue() ) {
                	count = count + 255;
                }
 // Create column definitions
                allColSpecs = new DataColumnSpec[count];
                count =0;
                int ilast=0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else {
                        chr = ',';
                    }
                    if ( chr == ',' ) {
                        if  ( count == 1 ) {
                            allColSpecs[count-1] = 
                            new DataColumnSpecCreator(line.substring(ilast,i),SmilesCell.TYPE).createSpec();
                        }
                        else if  ( line.substring(ilast,i).equals("KEYS") ) {
                            keys = count;
                            if ( m_fingerprint.getBooleanValue()) {
                               	for (int j=0; j<256; j++ ) {
                            	    allColSpecs[count-1] = 
                            	 	new DataColumnSpecCreator("Key#"+j,IntCell.TYPE).createSpec();
                                	count ++;
                               	}
                               	count --;
                            }
                            else {
                            	allColSpecs[count-1] = 
                            		new DataColumnSpecCreator(line.substring(ilast,i),StringCell.TYPE).createSpec();
                            }
                        }
                        else if ( count > 1 ) {
                            allColSpecs[count-1] = 
                            new DataColumnSpecCreator(line.substring(ilast,i), DoubleCell.TYPE).createSpec();
                        }
                        count ++;
                        ilast = i + 1;
                    }
                }
                outputSpec = new DataTableSpec(allColSpecs);
                cont = exec.createDataContainer(outputSpec);
            }
// Load data
            else {
                DataCell[] cells = new DataCell[count-1];
                rowcount++;
                count =0;                  
                double d_value;
                int ilast=0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else
                        chr = ',';
                    if ( chr == ',' ) { 
                        if ( count == 0 ) {
                            title = line.substring(0,i);
                            if (title.length() == 0) {
                                title = "Mol " + rowcount;
                            }
                            if (titles.contains(title)) {
                                title = title + "-" + rowcount;
                            }
                            titles.add(title);
                            key = new RowKey(title);                           
                        }
                        else if  ( count == 1 ) {
                            cells[count-1] = new SmilesCell(line.substring(ilast,i));
                        }
                        else if  ( count == keys ) {
                            if ( m_fingerprint.getBooleanValue()) {
                            	int n=0;
                            	String hex = "0123456789ABCDEF" ;
                            	for (int j=0; j<64; j++ ) {
                            		int m = hex.indexOf(line.substring(ilast+j,ilast+j+1));
                                 	int b3 = 0;
                              		if ( m >= 8) {
                                 		b3 = 1;
                                 		m = m - 8;
                            		}
                                 	int b2 = 0;
                              		if ( m >= 4) {
                                 		b2 = 1;
                                 		m = m - 4;
                            		}
                                 	int b1 = 0;
                              		if ( m >= 2) {
                                 		b1 = 1;
                                 		m = m - 2;
                            		}
                                 	int b0 = 0;
                            		if ( m >= 1) {
                                 		b0 = 1;
                                 		m = m - 1;
                            		}
                               		cells[count-1] = new IntCell(b0);
                               		count++;
                               		cells[count-1] = new IntCell(b1);
                               		count++;
                               		cells[count-1] = new IntCell(b2);
                               		count++;
                               		cells[count-1] = new IntCell(b3);
                               		count++;
                            	}
                            	count--;
                            }
                            else {
                            	cells[count-1] = new StringCell(line.substring(ilast,i));
                            }
                        }
                        else {
                        try {
                            d_value = Double.valueOf(line.substring(ilast,i)).doubleValue();
                            cells[count-1] = new DoubleCell(d_value);
                        } catch (Throwable t) {
                            cells[count-1] = DataType.getMissingCell();
                        }
                        }
                        count ++;
                        ilast = i + 1;
                    }
                }
                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
            }
        }
        cont.close();
        in.close();

        return cont.getTable();
    }
   /**
     * @see org.knime.core.node.NodeModel#reset()
     */
    @Override
    protected void reset() {
        // TODO Code executed on reset.
        // Models build during execute are cleared here.
        // Also data handled in load/saveInternals will be erased here.
    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #configure(org.knime.core.data.DataTableSpec[])
     */

    /**
     * @see org.knime.core.node.NodeModel
     *      #saveSettingsTo(org.knime.core.node.NodeSettings)
     */
    @Override
    protected void saveSettingsTo(final NodeSettingsWO settings) {

        // TODO save user settings to the config object.
		
		m_smiles.saveSettingsTo(settings);
		m_fingerprint.saveSettingsTo(settings);
		m_working.saveSettingsTo(settings);

    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #loadValidatedSettingsFrom(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void loadValidatedSettingsFrom(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO load (valid) settings from the config object.
        // It can be safely assumed that the settings are valided by the 
        // method below.
        
        m_smiles.loadSettingsFrom(settings);
        m_fingerprint.loadSettingsFrom(settings);
        m_working.loadSettingsFrom(settings);

    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #validateSettings(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void validateSettings(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO check if the settings could be applied to our model
        // e.g. if the count is in a certain range (which is ensured by the
        // SettingsModel).
        // Do not actually set any values of any member variables.

        m_smiles.validateSettings(settings);
        m_fingerprint.validateSettings(settings);
        m_working.validateSettings(settings);

    }
    
    /**
     * @see org.knime.core.node.NodeModel #loadInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void loadInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
        
		// TODO load internal data. 
		// Everything handed to output ports is loaded automatically (data
		// returned by the execute method, models loaded in loadModelContent,
		// and user settings set through loadSettingsFrom - is all taken care 
		// of). Load here only the other internals that need to be restored
		// (e.g. data used by the views).

    }
    
    /**
     * @see org.knime.core.node.NodeModel #saveInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void saveInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
       
       	// TODO save internal models. 
		// Everything written to output ports is saved automatically (data
		// returned by the execute method, models saved in the saveModelContent,
		// and user settings saved through saveSettingsTo - is all taken care 
		// of). Save here only the other internals that need to be preserved
		// (e.g. data used by the views).

    }

}
