package com.treweren.focusedset;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.HashSet;

import org.knime.chem.types.SdfValue;
import org.knime.chem.types.SmilesCell;
import org.knime.chem.types.SmilesValue;
import org.knime.core.data.DataCell;
import org.knime.core.data.DataColumnProperties;
import org.knime.core.data.DataColumnSpec;
import org.knime.core.data.DataColumnSpecCreator;
import org.knime.core.data.DataRow;
import org.knime.core.data.DataTableSpec;
import org.knime.core.data.DataType;
import org.knime.core.data.RowKey;
import org.knime.core.data.def.DefaultRow;
import org.knime.core.data.def.DoubleCell;
import org.knime.core.data.def.IntCell;
import org.knime.core.data.def.StringCell;
import org.knime.core.node.BufferedDataContainer;
import org.knime.core.node.BufferedDataTable;
import org.knime.core.node.CanceledExecutionException;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelDoubleBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelString;
import org.knime.core.node.ExecutionContext;
import org.knime.core.node.ExecutionMonitor;
import org.knime.core.node.InvalidSettingsException;
import org.knime.core.node.NodeLogger;
import org.knime.core.node.NodeModel;
import org.knime.core.node.NodeSettingsRO;
import org.knime.core.node.NodeSettingsWO;
import org.knime.core.util.MutableBoolean;


/**
 * This is the model implementation of FocusedSet.
 * Select molecules by overlapping pharmacophores
 *
 * @author Treweren Consultants
 */
public class FocusedSetNodeModel extends NodeModel {
    
    // the logger instance
    private static final NodeLogger logger = NodeLogger
            .getLogger(FocusedSetNodeModel.class);
    private static int n_Column = 0;
    private static String m_sdfColumn = null;
    private static String m_smilesColumn = null;
    private static String m_stringColumn = null;
    private static final int CANCEL_CHECK_INTERVAL = 1000;
     
    /** the settings key which is used to retrieve and 
        store the settings (from the dialog or from a settings file)    
       (package visibility to be usable from the dialog). */
	static final String CFGKEY_COUNT = "Count";
	static final String CFGKEY_CENTRES = "Centres";
	static final String CFGKEY_TOLERANCE = "Tolerance";
	static final String CFGKEY_SMILES = "Smiles";
	static final String CFGKEY_WORKING = "Working";
	static final String CFGKEY_SINGLE = "Single";
	static final String CFGKEY_CONJUGATED = "Conjugated";
	static final String CFGKEY_CROWDED = "Crowded";
	static final String CFGKEY_ALPHA = "Alpha";
	static final String CFGKEY_AMIDE = "Amide";
	static final String CFGKEY_RING = "Ring";
	static final String CFGKEY_GENERATION = "Generation";
	static final String CFGKEY_SAMPLES = "Samples";
	static final String CFGKEY_CPKRATIO = "CPKratio";
	static final String CFGKEY_TIMELIMIT = "TimeLimit";
	static final String CFGKEY_SYSLIMIT = "SysLimit";
	static final String CFGKEY_BONDLIMIT = "BondLimit";

	/** initial default count value. */
	static final int DEFAULT_COUNT = 10;
	static final int DEFAULT_CENTRES = 4;
	static final double DEFAULT_TOLERANCE = 0.5;
	static final boolean DEFAULT_SMILES = false;
	static final String DEFAULT_WORKING = System.getenv("THINK_WORKING");
	static final int DEFAULT_SINGLE = 3;
	static final int DEFAULT_CONJUGATED = 2;
	static final int DEFAULT_CROWDED = 2;
	static final int DEFAULT_ALPHA = 6;
	static final int DEFAULT_AMIDE = 0;
	static final int DEFAULT_RING = 0;
	static final String DEFAULT_GENERATION = "Systematic";
	static final int DEFAULT_SAMPLES = 1000;
	static final double DEFAULT_CPKRATIO = 0.6;
	static final double DEFAULT_TIMELIMIT = 60.;
	static final int DEFAULT_SYSLIMIT = 0;
	static final int DEFAULT_BONDLIMIT = 10;

	// example value: the models count variable filled from the dialog 
	// and used in the models execution method. The default components of the
	// dialog work with "SettingsModels".
	
	private final SettingsModelIntegerBounded m_count =
		new SettingsModelIntegerBounded(FocusedSetNodeModel.CFGKEY_COUNT,
                    FocusedSetNodeModel.DEFAULT_COUNT,
                    1, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_centres =
		new SettingsModelIntegerBounded(FocusedSetNodeModel.CFGKEY_CENTRES,
                    FocusedSetNodeModel.DEFAULT_CENTRES, 3, 4 );

	private final SettingsModelDoubleBounded m_tolerance =
		new SettingsModelDoubleBounded(FocusedSetNodeModel.CFGKEY_TOLERANCE,
                    FocusedSetNodeModel.DEFAULT_TOLERANCE,
                    0., 5.0);

	private final SettingsModelBoolean m_smiles =
		new SettingsModelBoolean(FocusedSetNodeModel.CFGKEY_SMILES,
                    FocusedSetNodeModel.DEFAULT_SMILES );

	private final SettingsModelString m_working =
		new SettingsModelString(FocusedSetNodeModel.CFGKEY_WORKING, FocusedSetNodeModel.DEFAULT_WORKING );

	private final SettingsModelIntegerBounded m_single =
		new SettingsModelIntegerBounded(FocusedSetNodeModel.CFGKEY_SINGLE,
                    FocusedSetNodeModel.DEFAULT_SINGLE, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_conjugated =
		new SettingsModelIntegerBounded(FocusedSetNodeModel.CFGKEY_CONJUGATED,
                    FocusedSetNodeModel.DEFAULT_CONJUGATED, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_crowded =
		new SettingsModelIntegerBounded(FocusedSetNodeModel.CFGKEY_CROWDED,
                    FocusedSetNodeModel.DEFAULT_CROWDED, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_alpha =
		new SettingsModelIntegerBounded(FocusedSetNodeModel.CFGKEY_ALPHA,
                    FocusedSetNodeModel.DEFAULT_ALPHA, 0, Integer.MAX_VALUE);


	private final SettingsModelIntegerBounded m_amide =
		new SettingsModelIntegerBounded(FocusedSetNodeModel.CFGKEY_AMIDE,
                    FocusedSetNodeModel.DEFAULT_AMIDE, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_ring =
		new SettingsModelIntegerBounded(FocusedSetNodeModel.CFGKEY_RING,
                    FocusedSetNodeModel.DEFAULT_RING, 0, Integer.MAX_VALUE);
	
	private final SettingsModelString m_generation =
		new SettingsModelString(FocusedSetNodeModel.CFGKEY_GENERATION,
                    FocusedSetNodeModel.DEFAULT_GENERATION);

	private final SettingsModelIntegerBounded m_samples =
		new SettingsModelIntegerBounded(FocusedSetNodeModel.CFGKEY_SAMPLES,
                    FocusedSetNodeModel.DEFAULT_SAMPLES, 0, Integer.MAX_VALUE);

	private final SettingsModelDoubleBounded m_cpkratio =
		new SettingsModelDoubleBounded(FocusedSetNodeModel.CFGKEY_CPKRATIO,
                    FocusedSetNodeModel.DEFAULT_CPKRATIO,
                    0., 5.0);
	
	private final SettingsModelDoubleBounded m_timelimit =
		new SettingsModelDoubleBounded(FocusedSetNodeModel.CFGKEY_TIMELIMIT,
                    FocusedSetNodeModel.DEFAULT_TIMELIMIT,
                    0., 1440.);

	private final SettingsModelIntegerBounded m_syslimit =
		new SettingsModelIntegerBounded(FocusedSetNodeModel.CFGKEY_SYSLIMIT,
                    FocusedSetNodeModel.DEFAULT_SYSLIMIT, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_bondlimit =
		new SettingsModelIntegerBounded(FocusedSetNodeModel.CFGKEY_BONDLIMIT,
                    FocusedSetNodeModel.DEFAULT_BONDLIMIT, 0, Integer.MAX_VALUE);


    /**
     * Constructor for the node model.
     */
    protected FocusedSetNodeModel() {
    
        // Two incoming port and one outgoing port (need to add revised pharm o/p)
        super(2, 1);
    }

    /**
     * @see org.knime.core.node.NodeModel #execute(BufferedDataTable[],
     *      ExecutionContext)
     */
    @Override
    protected DataTableSpec[] configure(final DataTableSpec[] inSpecs) throws InvalidSettingsException {

        if ( System.getenv("THINK_EXEC") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_EXEC is unset - It should be set to the folder containing the THINK software.");       	  		
    	}
        if ( System.getenv("THINK_WORKING") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_WORKING is not set to the folder in which working files will be created.");       	  		
    	}

        DataTableSpec in = inSpecs[0];
        n_Column = columnSelect (in);
        return new DataTableSpec[] {null};
    }
    protected int columnSelect ( final DataTableSpec in ) throws  InvalidSettingsException {

        StringBuilder warningMessage = new StringBuilder();
        int n_column=0;
        int sdfColCount = 0;
        int n_sdf=0;
        int smilesColCount = 0;
        int n_smiles=0;
        int stringColCount = 0;
        int n_string=0;
        for (int i = 0; i < in.getNumColumns(); i++) {
            DataColumnSpec s = in.getColumnSpec(i);
            if (s.getType().isCompatible(SdfValue.class)) {
                if (m_sdfColumn == null) {
                    m_sdfColumn = in.getColumnSpec(i).getName();
                    n_sdf = i;
                }
                sdfColCount++;
            }
            else if ( s.getType().isCompatible(SmilesValue.class)) {
            	if (m_smilesColumn == null ) {
             		m_smilesColumn = in.getColumnSpec(i).getName();
             		n_smiles=i;
              	}
               	smilesColCount++;
            }
            else if ( in.getColumnSpec(i).getName().equalsIgnoreCase("smiles"))  {
            	if (m_stringColumn == null ) {
             		m_stringColumn = in.getColumnSpec(i).getName();
             		n_string=i;
              	}
               	stringColCount++;
            }
        }
        if ( smilesColCount == 0 & sdfColCount == 0 & stringColCount == 0) {
            throw new InvalidSettingsException(
                    "No column with Smiles or SD file compatible type");       	
        }
        else if ( ( m_smiles.getBooleanValue() | sdfColCount == 0 ) & smilesColCount > 0 ) {
        	m_sdfColumn = null;
        	m_stringColumn = null;
        	n_column = n_smiles;
        	if (smilesColCount > 1) {
                warningMessage.append("More than one Smiles compatible column in " 
                        + "input, using column \"" + m_smilesColumn + "\".");
        	}
        }
        else if ( sdfColCount > 0 ) {
           	m_smilesColumn = null;
        	m_stringColumn = null;
        	n_column = n_sdf;
        	if (sdfColCount > 1) {
                warningMessage.append("More than one SDF compatible column in " 
                        + "input, using column \"" + m_sdfColumn + "\".");
        	}
        }
        else if ( stringColCount > 0 ) {
        	m_smilesColumn = null;
        	m_sdfColumn = null;
        	n_column = n_string;
           	if (stringColCount > 1) {
                warningMessage.append("More than one Smiles string compatible column in " 
                        + "input, using column \"" + m_stringColumn + "\".");
           	}
        }
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
        return n_column;
    }
   @Override
    protected BufferedDataTable[] execute(final BufferedDataTable[] inData,
            final ExecutionContext exec) throws Exception {
       notifyViews(null);

       final String cwdString = m_working.getStringValue();
       // clean up 
       File outputReport = new File(cwdString, "output.log"); 
       outputReport.delete();
       File outputFile = new File(cwdString, "focusedset.csv"); 
       outputFile.delete();
       File progressFile = new File(cwdString, "progress0.dat"); 
       progressFile.delete();
       BufferedDataTable in = inData[0];

       if ( n_Column == 0 ) {
       	DataTableSpec InSpec = in.getDataTableSpec();
       	n_Column = columnSelect(InSpec);
       }


       // write out input file
       String fileString = null; 
       if ( m_sdfColumn == null ) {
           fileString = "focusedset.smiles"; 
       }
       else {
           fileString = "focusedset.sdf"; 
       }
       File inFile = new File(cwdString, fileString); 
       BufferedWriter outWriter = new BufferedWriter(new FileWriter(inFile));
        int colIndex;
       if ( m_sdfColumn != null ) {
       	colIndex = in.getDataTableSpec().findColumnIndex(m_sdfColumn);
       }
       else if ( m_smilesColumn != null ) {
       	colIndex = in.getDataTableSpec().findColumnIndex(m_smilesColumn);        
       }
       else {
           colIndex = in.getDataTableSpec().findColumnIndex(m_stringColumn);        
       }
      	  
       final double count = in.getRowCount(); // floating point operations
       int i = 0; 
       int missingCount = 0;
       for (DataRow r : in) {
           exec.checkCanceled();
           DataCell c = r.getCell(colIndex);
           if (c.isMissing()) {
               missingCount++;
           } 
           else if ( m_sdfColumn != null ) {
               SdfValue v = (SdfValue)c;
               String toString = v.toString();
               outWriter.write(toString);
               if (!toString.trim().endsWith("$$$$")) {
                   outWriter.newLine();
                   outWriter.append("$$$$");
                   outWriter.newLine();
               }
           }
           else if ( m_smilesColumn != null ) {
               SmilesValue v = (SmilesValue)c;
               String toString = v.toString() + " " + r.getKey();
               outWriter.write(toString);        	
               outWriter.newLine();
           }
           else {
           	String toString = c.toString() + " " + r.getKey();
           	outWriter.write(toString);        	
           	outWriter.newLine();
           }

          i++;
       }

       outWriter.close();
       if (missingCount > 0) {
           setWarningMessage("Skipped " + missingCount 
                   + " row(s) because of missing values");
       }
// Write output pharmacophore profile
       int n_pop = -1;
       String head1="";
       String head2="";
       String head3="";
       BufferedDataTable pro = inData[1];
       DataTableSpec proSpec = pro.getDataTableSpec();
       for (int j = 0; j < proSpec.getNumColumns(); j++) {
           DataColumnSpec s = proSpec.getColumnSpec(j);
           if ( proSpec.getColumnSpec(j).getName().equals("Population")) {
        	   n_pop = j;
        	   DataColumnProperties ColProp = proSpec.getColumnSpec(j).getProperties();
        	   head1 = ColProp.getProperty("Header 1","# 3 11     1 Y  ????? N");
        	   head2 = ColProp.getProperty("Header 2","#3.0 4.0 5.0 6.0 7.0 9.0 11.0 14.0 17.0 20.0 9.9E35");
        	   head3 = ColProp.getProperty("Header 3","#  4");
        	   break;
           }
       }
       
       String proString =  "focusedset.pro"; 
       File proFile = new File(cwdString, proString); 
       BufferedWriter proWriter = new BufferedWriter(new FileWriter(proFile));
       boolean rows=false;
       if ( n_pop >= 0) {
    	   for (DataRow r : pro) {
    		   exec.checkCanceled();
    		   DataCell c = r.getCell(n_pop);
    		   if (c.isMissing()) {
    			   missingCount++;
    		   }
    		   else {
    			   if ( !rows ){
    				   rows = true;
// Write headers    			   
    				   proWriter.write(head1);        	
    				   proWriter.newLine();       	   
    				   proWriter.write(head2);        	
    				   proWriter.newLine();       	   
  	       			   proWriter.write(head3);        	
   	    			   proWriter.newLine();       	     					   
    			   }
    			   String toString = r.getKey() + "," + c.toString();
    			   proWriter.write(toString);        	
    			   proWriter.newLine();       	   
    		   }
    	   }
       }
       proWriter.close();
       
       // execute THINK
       File cwdFile = new File (cwdString);
       int exitVal;
       try {
           exec.setProgress(0., "Starting THINK");
           Runtime rt = Runtime.getRuntime();
           int n_count = m_count.getIntValue();
           double d_tolerance = m_tolerance.getDoubleValue();
           int n_centres = m_centres.getIntValue();
           String s_profile = "NONE";
           if ( rows ) {
        	   s_profile = "focusedset.pro";
           }
           int n_single = m_single.getIntValue();
           int n_conjugated = m_conjugated.getIntValue();
           int n_crowded = m_crowded.getIntValue();
           int n_alpha = m_alpha.getIntValue();
           int n_amide = m_amide.getIntValue();
           int n_ring = m_ring.getIntValue();
           String s_generation = m_generation.getStringValue();
           int n_samples = m_samples.getIntValue();
           double d_cpkratio = m_cpkratio.getDoubleValue();
           double d_timelimit = m_timelimit.getDoubleValue();
           int n_syslimit = m_syslimit.getIntValue();
           int n_bondlimit = m_bondlimit.getIntValue();
           
           // prepare the command string
           String cmdString = System.getenv("THINK_EXEC") + "think THINK_EXEC:focusedset.log output.log " + fileString + " " + n_count + " " + d_tolerance + " " + n_centres + " " + s_profile + " "
           + n_single + " " + n_conjugated + " " + n_crowded + " " + n_alpha  + " " + n_amide + " " + n_ring
           + " " + s_generation + " " + n_samples + " " + d_cpkratio + " " + d_timelimit + " " + n_syslimit + " " + n_bondlimit;

           // Go!
           logger.info("THINK command line: '" + cmdString + "'");
           final Process proc = rt.exec(cmdString, null, cwdFile);

           final MutableBoolean procDone = new MutableBoolean(false);
           new Thread(new Runnable() {
               public void run() {
                   synchronized (procDone) {
                       while (!procDone.booleanValue()) {
                           try {
                               exec.checkCanceled();
                           } catch (CanceledExecutionException cee) {
                               // blow away the running external process
                               proc.destroy();
                               return;
                           }
                           try {
                              	procDone.wait(CANCEL_CHECK_INTERVAL);
                               // Read progress file and set progress value       
                              	double d= 0.;
                               File progressReport = new File(cwdString, "progress0.dat");
                               if ( progressReport.exists() && progressReport.isFile() ) {
                              		if ( progressReport.length() > 0 ) {
                              			BufferedReader in = new BufferedReader(new FileReader(progressReport));
                              			String line;
                              			if ( (line = in.readLine()) != null) {
                              			  d = Double.valueOf(line).doubleValue();
                              			}               
                            			if ( (line = in.readLine()) == null) {
                             				line = "Running THINK";
                             			}
                              			in.close();
                                       exec.setProgress( d, line.toString());
                              		}
                               }
                           } catch (InterruptedException e) {
                               // do nothing
                           } catch ( FileNotFoundException e) {
                               // do nothing
                           } catch ( IOException e) {
                               // do nothing
                           }
                       }
                   }
               }

           }).start();

           // wait until the external process finishes.
           exitVal = proc.waitFor();

           synchronized (procDone) {
               // this should terminate the check cancel thread
               procDone.setValue(true);
           }

           exec.checkCanceled();

           exec.setProgress("Wrapping up");
           logger.info("THINK terminated with exit code: " + exitVal);
       } catch (Throwable t) {
       	warningReport();
           logger.error("THINK failed (with exception)", t);
           throw new Exception(t);
       }

       if (exitVal != 0) {
       	warningReport();
           // before we return barfing, we save the output in the failing list
           throw new IllegalStateException("THINK failed (error code "
                   + exitVal + ")");
       }

       exec.setProgress("Creating KNIME tables");
       BufferedDataTable outTable = readCSVFile(exec);

       return new BufferedDataTable[]{outTable};

   }

   private void warningReport( ) throws Exception {

   	StringBuilder warningMessage = new StringBuilder();
       String cwdString = m_working.getStringValue();
       File outputReport = new File(cwdString, "output.log"); 

       if ( outputReport.exists() && outputReport.isFile() ) {
      		if ( outputReport.length() > 0 ) {
      			BufferedReader in = new BufferedReader(new FileReader(outputReport));
      			String line;
      			while ((line = in.readLine()) != null) {
      				warningMessage.append( line + "\n");
      			}               
      			in.close();
      		}
      		else { 
               warningMessage.append( "Report from THINK is blank");      	     			
      		}
       }
      	else {
           warningMessage.append( "No report from THINK");      	
      	}	
       if (warningMessage.length() > 0) {
           setWarningMessage(warningMessage.toString());
       }
  }

   private BufferedDataTable readCSVFile(final ExecutionContext exec) throws Exception { 

       String cwdString =  m_working.getStringValue();
       File outFile = new File (cwdString, "focusedset.csv");
       if (!(outFile.exists() && outFile.isFile())) {
       	warningReport();
       	throw new IllegalStateException("THINK didn't produce any output"
                   + " at the specified location ('"
                   + outFile.getAbsolutePath() + "')");
       }
       HashSet<String> titles = new HashSet<String>();
       String title;
       String line;
       boolean headerRead = false;
       boolean columnsCreated = false;
       DataColumnSpec[] allColSpecs;
       BufferedReader in = new BufferedReader(new FileReader(outFile));
       DataTableSpec outputSpec;
       BufferedDataContainer cont =null;
       RowKey key =null;
       int count =0;
       int colcount =0;
       int rowcount =0;
       char chr='0';
       int keys = 0;
       int ilast = 0;
       String fieldNames[] = new String[100];
       int fieldTypes[] = new int[100];
       while ((line = in.readLine()) != null) {
           exec.checkCanceled();
//Extract column headers
           if ( !headerRead ) {
               headerRead = true;
               colcount =0;
               for ( int i=0; i<=line.length(); i++ ) {
                   if ( i < line.length() ) 
                       chr = line.charAt(i);
                   else
                       chr = ',';
                   if ( chr == ',' ) { 
                       fieldNames[colcount] = line.substring(ilast,i);
                       if ( line.substring(ilast,i).equalsIgnoreCase("SMILES"))
                       	fieldTypes[colcount]= 2;
                       else if ( line.substring(ilast,i).equalsIgnoreCase("NAME"))
                       	fieldTypes[colcount]= 1;
                       else if ( line.substring(ilast,i).equalsIgnoreCase("KEYS"))
                       	fieldTypes[colcount]= 3;
                       ilast = i+1;
                       colcount++;
                   }
               }
               colcount--;
           }
//Load data
           else {
               DataCell[] cells = new DataCell[colcount];
               rowcount++;
               count =0;                 
               double d_value;
                for ( int i=0; i<=line.length(); i++ ) {
                   if ( i < line.length() ) 
                       chr = line.charAt(i);
                   else
                       chr = ',';
                   if ( chr == ',' ) { 
                       if ( count == 0 ) {
                           title = line.substring(0,i);
                           if (title.length() == 0) 
                               title = "Mol " + rowcount;
                           if (titles.contains(title)) 
                               title = title + "-" + rowcount;
                           titles.add(title);
                           key = new RowKey(title);                           
                       }
                       else if  ( fieldTypes[count] == 2 ) 
                           cells[count-1] = new SmilesCell(line.substring(ilast,i));
                       else if  ( fieldTypes[count]== 3) 
                           cells[count-1] = new StringCell(line.substring(ilast,i));
                       else if ( count <= colcount ) {
                         try {
                           d_value = Double.valueOf(line.substring(ilast,i)).doubleValue();
                           cells[count-1] = new DoubleCell(d_value);
                           if ( fieldTypes[count] == 0 )
                           	fieldTypes[count] = 4;
                         } catch (Throwable t) {
                           if ( fieldTypes[count] == 0 ) {
                           	fieldTypes[count] = 3;
                               cells[count-1] = new StringCell(line.substring(ilast,i));
                           }
                           else
                               cells[count-1] = DataType.getMissingCell();
                         }
                       }
                       count ++;
                       ilast = i + 1;
                   }
                }
 // Create columns?
                if (!columnsCreated )
                {
                    allColSpecs = new DataColumnSpec[colcount];
                    columnsCreated = true;
 // count= 0 will be molecule names
                    for (count=1; count<=colcount; count++)
                    {                   	 
                      if  ( fieldTypes[count] == 2 ) 
                        allColSpecs[count-1] = new DataColumnSpecCreator(fieldNames[count],SmilesCell.TYPE).createSpec();
                      else if  ( fieldTypes[count] == 4 ) 
                        allColSpecs[count-1] = new DataColumnSpecCreator(fieldNames[count], DoubleCell.TYPE).createSpec();
                      else if ( fieldTypes[count] == 3 )
                        allColSpecs[count-1] = new DataColumnSpecCreator(fieldNames[count],StringCell.TYPE).createSpec();
                      else
                   	 allColSpecs[count-1] = new DataColumnSpecCreator("Unknown",StringCell.TYPE).createSpec();
                    }
                    outputSpec = new DataTableSpec(allColSpecs);
                    cont = exec.createDataContainer(outputSpec);               	 
                }
               DefaultRow row = new DefaultRow(key, cells);
               cont.addRowToTable(row);
           }
       }
       cont.close();
       in.close();

       return cont.getTable();
   }

   private BufferedDataTable readOutputFile(final ExecutionContext exec) throws Exception { 

       String cwdString =  m_working.getStringValue();
       File outFile = new File (cwdString, "focusedset.csv");
       if (!(outFile.exists() && outFile.isFile())) {
       	warningReport();
       	throw new IllegalStateException("THINK didn't produce any output"
                   + " at the specified location ('"
                   + outFile.getAbsolutePath() + "')");
       }
       HashSet<String> titles = new HashSet<String>();
       String title;
       String line;
       boolean headerRead = false;
       DataColumnSpec[] allColSpecs;
       BufferedReader in = new BufferedReader(new FileReader(outFile));
       DataTableSpec outputSpec;
       BufferedDataContainer cont =null;
       RowKey key =null;
       int count =0;
       int rowcount =0;
       char chr='0';
       int keys = 0;
       int ilast = 0;
       while ((line = in.readLine()) != null) {
           exec.checkCanceled();
//Extract column headers
           if ( !headerRead ) {
               headerRead = true;
               count =0;
               for ( int i=0; i<line.length(); i++ ) {
                   if ( line.charAt(i) == ',' ) {
                       if  ( !line.substring(ilast,i).equals("KEYS") ) {
                           count ++;
                   	}
                       else {
                           keys = count;
                       }
                       ilast = i+1;
                   }
               }
//Create column definitions
               allColSpecs = new DataColumnSpec[count];
               count =0;
               ilast =0;
               int keyskip = keys;
               for ( int i=0; i<=line.length(); i++ ) {
                   if ( i < line.length() ) {
                       chr = line.charAt(i);
                   }
                   else {
                       chr = ',';
                   }
                   if ( chr == ',' ) {
                       if  ( count == 1 ) {
                           allColSpecs[count-1] = 
                           new DataColumnSpecCreator(line.substring(ilast,i),SmilesCell.TYPE).createSpec();
                       }
                       else if  ( count == keyskip ) {
 /* 
                           allColSpecs[count-1] = 
                           new DataColumnSpecCreator(line.substring(ilast,i),StringCell.TYPE).createSpec();
*/
                       	keyskip =0;
                       	count --;
                       }
                       else if ( count > 1 ) {
                           allColSpecs[count-1] = 
                           new DataColumnSpecCreator(line.substring(ilast,i), DoubleCell.TYPE).createSpec();
                       }
                       count ++;
                       ilast = i + 1;
                   }
               }
               outputSpec = new DataTableSpec(allColSpecs);
               cont = exec.createDataContainer(outputSpec);
           }
//Load data
           else {
               DataCell[] cells = new DataCell[count-1];
               rowcount++;
               count =0;                  
               int keyskip = keys;
               double d_value;
                for ( int i=0; i<=line.length(); i++ ) {
                   if ( i < line.length() ) {
                       chr = line.charAt(i);
                   }
                   else
                       chr = ',';
                   if ( chr == ',' ) { 
                       if ( count == 0 ) {
                           title = line.substring(0,i);
                           if (title.length() == 0) {
                               title = "Mol " + rowcount;
                           }
                           if (titles.contains(title)) {
                               title = title + "-" + rowcount;
                           }
                           titles.add(title);
                           key = new RowKey(title);                           
                       }
                       else if  ( count == 1 ) {
                           cells[count-1] = new SmilesCell(line.substring(ilast,i));
                       }
                       else if  ( count == keyskip ) {
                       	keyskip = 0;
                       	count --;
//                         cells[count-1] = new StringCell(line.substring(ilast,i));
                       }
                       else {
                       try {
                           d_value = Double.valueOf(line.substring(ilast,i)).doubleValue();
                           cells[count-1] = new DoubleCell(d_value);
                       } catch (Throwable t) {
                           cells[count-1] = DataType.getMissingCell();
                       }
                       }
                       count ++;
                       ilast = i + 1;
                   }
               }
               DefaultRow row = new DefaultRow(key, cells);
               cont.addRowToTable(row);
           }
       }
       cont.close();
       in.close();

       return cont.getTable();
   }

    /**
     * @see org.knime.core.node.NodeModel#reset()
     */
    @Override
    protected void reset() {
        // TODO Code executed on reset.
        // Models build during execute are cleared here.
        // Also data handled in load/saveInternals will be erased here.
    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #configure(org.knime.core.data.DataTableSpec[])
     */


    /**
     * @see org.knime.core.node.NodeModel
     *      #saveSettingsTo(org.knime.core.node.NodeSettings)
     */
    @Override
    protected void saveSettingsTo(final NodeSettingsWO settings) {

        // TODO save user settings to the config object.
		
		m_count.saveSettingsTo(settings);
		m_centres.saveSettingsTo(settings);
		m_tolerance.saveSettingsTo(settings);
		m_smiles.saveSettingsTo(settings);
		m_working.saveSettingsTo(settings);
		m_single.saveSettingsTo(settings);
		m_conjugated.saveSettingsTo(settings);
		m_crowded.saveSettingsTo(settings);
		m_alpha.saveSettingsTo(settings);
		m_amide.saveSettingsTo(settings);
		m_ring.saveSettingsTo(settings);
		m_generation.saveSettingsTo(settings);
		m_samples.saveSettingsTo(settings);
		m_cpkratio.saveSettingsTo(settings);
		m_timelimit.saveSettingsTo(settings);
		m_syslimit.saveSettingsTo(settings);
		m_bondlimit.saveSettingsTo(settings);

    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #loadValidatedSettingsFrom(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void loadValidatedSettingsFrom(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO load (valid) settings from the config object.
        // It can be safely assumed that the settings are valided by the 
        // method below.
        
        m_count.loadSettingsFrom(settings);
        m_tolerance.loadSettingsFrom(settings);
        m_centres.loadSettingsFrom(settings);
        m_smiles.loadSettingsFrom(settings);
        m_working.loadSettingsFrom(settings);
        m_single.loadSettingsFrom(settings);
        m_conjugated.loadSettingsFrom(settings);
        m_crowded.loadSettingsFrom(settings);
        m_alpha.loadSettingsFrom(settings);
        m_amide.loadSettingsFrom(settings);
        m_ring.loadSettingsFrom(settings);
        m_generation.loadSettingsFrom(settings);
        m_samples.loadSettingsFrom(settings);
        m_cpkratio.loadSettingsFrom(settings);
        m_timelimit.loadSettingsFrom(settings);
        m_syslimit.loadSettingsFrom(settings);
        m_bondlimit.loadSettingsFrom(settings);

    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #validateSettings(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void validateSettings(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO check if the settings could be applied to our model
        // e.g. if the count is in a certain range (which is ensured by the
        // SettingsModel).
        // Do not actually set any values of any member variables.

        m_count.validateSettings(settings);
        m_tolerance.validateSettings(settings);
        m_centres.validateSettings(settings);
        m_smiles.validateSettings(settings);
        m_working.validateSettings(settings);
        m_single.validateSettings(settings);
        m_conjugated.validateSettings(settings);
        m_crowded.validateSettings(settings);
        m_alpha.validateSettings(settings);
        m_amide.validateSettings(settings);
        m_ring.validateSettings(settings);
        m_generation.validateSettings(settings);
        m_samples.validateSettings(settings);
        m_cpkratio.validateSettings(settings);
        m_timelimit.validateSettings(settings);
        m_syslimit.validateSettings(settings);
        m_bondlimit.validateSettings(settings);

    }
    
    /**
     * @see org.knime.core.node.NodeModel #loadInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void loadInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
        
		// TODO load internal data. 
		// Everything handed to output ports is loaded automatically (data
		// returned by the execute method, models loaded in loadModelContent,
		// and user settings set through loadSettingsFrom - is all taken care 
		// of). Load here only the other internals that need to be restored
		// (e.g. data used by the views).

    }
    
    /**
     * @see org.knime.core.node.NodeModel #saveInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void saveInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
       
       	// TODO save internal models. 
		// Everything written to output ports is saved automatically (data
		// returned by the execute method, models saved in the saveModelContent,
		// and user settings saved through saveSettingsTo - is all taken care 
		// of). Save here only the other internals that need to be preserved
		// (e.g. data used by the views).

    }

}
