package com.treweren.similarity;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;
import java.io.BufferedReader;
import java.io.FileReader;


import org.knime.base.node.io.filereader.FileAnalyzer;
import org.knime.base.node.io.filereader.FileReaderNodeSettings;
import org.knime.base.node.io.filereader.FileReaderSettings;
import org.knime.base.node.io.filereader.FileTable;
import org.knime.base.node.io.filereader.SettingsStatus;
import org.knime.chem.types.SdfValue;
import org.knime.chem.types.SmilesValue;
import org.knime.core.data.DataCell;
import org.knime.core.data.DataColumnSpec;
import org.knime.core.data.DataColumnSpecCreator;
import org.knime.core.data.DataRow;
import org.knime.core.data.DataTableSpec;
import org.knime.core.data.RowKey;
import org.knime.core.data.def.DefaultRow;
import org.knime.core.data.def.DoubleCell;
import org.knime.core.data.def.IntCell;
import org.knime.core.data.def.StringCell;
import org.knime.core.node.BufferedDataContainer;
import org.knime.core.node.BufferedDataTable;
import org.knime.core.node.CanceledExecutionException;
import org.knime.core.node.defaultnodesettings.SettingsModelString;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelDoubleBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;
import org.knime.core.node.port.PortType;
import org.knime.core.node.ExecutionContext;
import org.knime.core.node.ExecutionMonitor;
import org.knime.core.node.InvalidSettingsException;
import org.knime.core.node.NodeLogger;
import org.knime.core.node.NodeModel;
import org.knime.core.node.NodeSettingsRO;
import org.knime.core.node.NodeSettingsWO;

import org.knime.core.data.RowKey;
import org.knime.core.node.BufferedDataContainer;
import org.knime.core.util.MutableBoolean;
import org.knime.core.data.DataColumnSpecCreator;
import org.knime.chem.types.SmilesCell;
import org.knime.core.data.def.DoubleCell;
import org.knime.core.data.def.IntCell;
import org.knime.core.data.def.StringCell;
import org.knime.core.data.def.DefaultRow;
import org.knime.core.data.DataType;


/**
 * This is the model implementation of Similarity.
 * Performs similarity search
 *
 * @author Treweren Consultants
 */
public class SimilarityNodeModel extends NodeModel {
    
    // the logger instance
    private static final NodeLogger logger = NodeLogger
            .getLogger(SimilarityNodeModel.class);
    private static int n_Column = 0;
    private static int maxProcs = 1;
    private static String m_sdfColumn = null;
    private static String m_smilesColumn = null;
    private static String m_stringColumn = null;
    private static final int CANCEL_CHECK_INTERVAL = 1000;
       
    /** the settings key which is used to retrieve and 
        store the settings (from the dialog or from a settings file)    
       (package visibility to be usable from the dialog). */
	static final String CFGKEY_SMILES = "Smiles";
	static final String CFGKEY_CUTOFF = "Cutoff";
	static final String CFGKEY_SEARCH = "Similarity_Search";
	static final String CFGKEY_FILTER = "Filter";
	static final String CFGKEY_PARENT = "Parent";
	static final String CFGKEY_ROW1 = "Row1";
	static final String CFGKEY_RESTART = "Restart";
	static final String CFGKEY_PROCESS = "Process";
	static final String CFGKEY_WORKING = "Working";

	/** initial default count value. */
	static final Double DEFAULT_CUTOFF = 0.8;
	static final Boolean DEFAULT_SMILES = false;
	static final Boolean DEFAULT_ROW1 = true;
	static final Boolean DEFAULT_RESTART = true;
	static final int DEFAULT_PROCESS = 1;
	static final String DEFAULT_SEARCH = "Use input";
	static final boolean DEFAULT_FILTER = false;
	static final boolean DEFAULT_PARENT = false;
static final String DEFAULT_WORKING = System.getenv("THINK_WORKING");
	 
	// example value: the models count variable filled from the dialog 
	// and used in the models execution method. The default components of the
	// dialog work with "SettingsModels".
	
	private final SettingsModelString m_search =
		new SettingsModelString(SimilarityNodeModel.CFGKEY_SEARCH,
                    SimilarityNodeModel.DEFAULT_SEARCH );

	private final SettingsModelBoolean m_filter =
		new SettingsModelBoolean(SimilarityNodeModel.CFGKEY_FILTER,
                    SimilarityNodeModel.DEFAULT_FILTER );
	
	private final SettingsModelBoolean m_parent =
		new SettingsModelBoolean(SimilarityNodeModel.CFGKEY_PARENT,
                    SimilarityNodeModel.DEFAULT_PARENT );

	private final SettingsModelBoolean m_smiles =
		new SettingsModelBoolean(SimilarityNodeModel.CFGKEY_SMILES,
                    SimilarityNodeModel.DEFAULT_SMILES );

	private final SettingsModelDoubleBounded m_cutoff =
		new SettingsModelDoubleBounded(SimilarityNodeModel.CFGKEY_CUTOFF,
                    SimilarityNodeModel.DEFAULT_CUTOFF,
                    0., 1.0);
	private final SettingsModelBoolean m_row1 =
		new SettingsModelBoolean(SimilarityNodeModel.CFGKEY_ROW1,
                    SimilarityNodeModel.DEFAULT_ROW1 );
	
	private final SettingsModelBoolean m_restart =
			new SettingsModelBoolean(SimilarityNodeModel.CFGKEY_RESTART,
	                    SimilarityNodeModel.DEFAULT_RESTART );
	private final SettingsModelIntegerBounded m_process =
			new SettingsModelIntegerBounded(SimilarityNodeModel.CFGKEY_PROCESS,
	                    SimilarityNodeModel.DEFAULT_PROCESS,
	                    1, 36);
	private final SettingsModelString m_working =
		new SettingsModelString(SimilarityNodeModel.CFGKEY_WORKING, SimilarityNodeModel.DEFAULT_WORKING );
	

    /**
     * Constructor for the node model.
     */
	public static final PortType OPTIONAL_PORT_TYPE = new PortType(BufferedDataTable.class, true);
    protected SimilarityNodeModel() {
    
        // Two incoming ports (second optional) and two outgoing ports
   	    super(createOPOs(2, 2), createOPOs(2));
     }
   	private static PortType[] createOPOs(final int nrDataPorts, final int... optionalPortsIds)
   	{
   	    PortType[] portTypes = new PortType[nrDataPorts];
   	    Arrays.fill(portTypes, BufferedDataTable.TYPE);        
    	    if (optionalPortsIds.length > 0) {
   	        for (int portId : optionalPortsIds) {
   	            if ((portId - 1) < nrDataPorts) {
   	                portTypes[portId - 1] = OPTIONAL_PORT_TYPE;
   	            }
   	        }
   	    }
   	    return portTypes;
   	} 

    /**
     * @see org.knime.core.node.NodeModel #execute(BufferedDataTable[],
     *      ExecutionContext)
     */
    @Override
    protected DataTableSpec[] configure(final DataTableSpec[] inSpecs) throws InvalidSettingsException {

        if ( System.getenv("THINK_EXEC") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_EXEC is unset - It should be set to the folder containing the THINK software.");       	  		
    	}
        if ( System.getenv("THINK_WORKING") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_WORKING is not set to the folder in which working files will be created.");       	  		
    	}
        DataTableSpec inmol = inSpecs[1];
        if ( inmol != null ) {
            n_Column = columnSelect (inmol);       	
        }
        else {

        	if ( m_search.getStringValue().equals("") ) {
        		throw new InvalidSettingsException(
        		"File to be searched not specified");       	  		
        	}
           	if ( m_search.getStringValue().equals(DEFAULT_SEARCH) ) {
        		throw new InvalidSettingsException(
        		"Optional 'Molecules to be searched' Input not connected and file to be searched not specified");       	  		
        	}
        	File searchFile = new File ( m_search.getStringValue() ); 
        	if (!(searchFile.exists() && searchFile.isFile())) {
        		throw new IllegalStateException("Selected file does not exist ('"
                    + searchFile.getAbsolutePath() + "')");
        	}
        }
        DataTableSpec in = inSpecs[0];
        n_Column = columnSelect (in);
        return new DataTableSpec[] {null, null};
    }
    protected int columnSelect ( final DataTableSpec in ) throws  InvalidSettingsException {

        StringBuilder warningMessage = new StringBuilder();
        int n_column=0;
        int sdfColCount = 0;
        int n_sdf=0;
        int smilesColCount = 0;
        int n_smiles=0;
        int stringColCount = 0;
        int n_string=0;
        for (int i = 0; i < in.getNumColumns(); i++) {
            DataColumnSpec s = in.getColumnSpec(i);
            if (s.getType().isCompatible(SdfValue.class)) {
                if (m_sdfColumn == null) {
                    m_sdfColumn = in.getColumnSpec(i).getName();
                    n_sdf = i;
                }
                sdfColCount++;
            }
            else if ( s.getType().isCompatible(SmilesValue.class)) {
            	if (m_smilesColumn == null ) {
             		m_smilesColumn = in.getColumnSpec(i).getName();
             		n_smiles=i;
              	}
               	smilesColCount++;
            }
            else if ( in.getColumnSpec(i).getName().equalsIgnoreCase("smiles"))  {
            	if (m_stringColumn == null ) {
             		m_stringColumn = in.getColumnSpec(i).getName();
             		n_string=i;
              	}
               	stringColCount++;
            }
        }
        if ( smilesColCount == 0 & sdfColCount == 0 & stringColCount == 0) {
            throw new InvalidSettingsException(
                    "No column with Smiles or SD file compatible type");       	
        }
        else if ( ( m_smiles.getBooleanValue() | sdfColCount == 0 ) & smilesColCount > 0 ) {
        	m_sdfColumn = null;
        	m_stringColumn = null;
        	n_column = n_smiles;
        	if (smilesColCount > 1) {
                warningMessage.append("More than one Smiles compatible column in " 
                        + "input, using column \"" + m_smilesColumn + "\".");
        	}
        }
        else if ( sdfColCount > 0 ) {
           	m_smilesColumn = null;
        	m_stringColumn = null;
        	n_column = n_sdf;
        	if (sdfColCount > 1) {
                warningMessage.append("More than one SDF compatible column in " 
                        + "input, using column \"" + m_sdfColumn + "\".");
        	}
        }
        else if ( stringColCount > 0 ) {
        	m_smilesColumn = null;
        	m_sdfColumn = null;
        	n_column = n_string;
           	if (stringColCount > 1) {
                warningMessage.append("More than one Smiles string compatible column in " 
                        + "input, using column \"" + m_stringColumn + "\".");
           	}
        }
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
        return n_column;
    }

 
    /**
     * @see org.knime.core.node.NodeModel #execute(BufferedDataTable[],
     *      ExecutionContext)
     */
    @Override
    protected BufferedDataTable[] execute(final BufferedDataTable[] inData,
            final ExecutionContext exec) throws Exception {

        notifyViews(null);
        int i = 0;
        int[] exitVal = new int[36];
        final String cwdString = m_working.getStringValue();
        // clean up 
        String fileString = null; 
        BufferedDataTable in = inData[0];
        
        if ( n_Column == 0 ) {
        	DataTableSpec InSpec = in.getDataTableSpec();
        	n_Column = columnSelect(InSpec);
        }
      
        // write out input file
        if ( m_sdfColumn == null ) {
            fileString = "similarity.smiles"; 
        }
        else {
            fileString = "similarity.sdf"; 
        }
        File inFile = new File(cwdString, fileString); 
        BufferedWriter outWriter = new BufferedWriter(new FileWriter(inFile));
        int colIndex;
        if ( m_sdfColumn != null ) {
        	colIndex = in.getDataTableSpec().findColumnIndex(m_sdfColumn);
        }
        else if ( m_smilesColumn != null ) {
        	colIndex = in.getDataTableSpec().findColumnIndex(m_smilesColumn);        
        }
        else {
            colIndex = in.getDataTableSpec().findColumnIndex(m_stringColumn);        
        }
       	  
        final int count = in.getRowCount(); 
        if ( count > 1 & m_row1.getBooleanValue() ) {
            setWarningMessage("Using only first row/molecule");
         }
 
        int missingCount = 0;
        for (DataRow r : in) {
            exec.checkCanceled();
            DataCell c = r.getCell(colIndex);
            if (c.isMissing()) {
                missingCount++;
            } 
            else if ( m_sdfColumn != null ) {
                SdfValue v = (SdfValue)c;
                String toString = v.toString();
                outWriter.write(toString);
                if (!toString.trim().endsWith("$$$$")) {
                    outWriter.newLine();
                    outWriter.append("$$$$");
                    outWriter.newLine();
                }
            }
            else if ( m_smilesColumn != null ) {
                SmilesValue v = (SmilesValue)c;
                String toString = v.toString() + " " + r.getKey();
                outWriter.write(toString);        	
                outWriter.newLine();
            }
            else {
            	String toString = c.toString() + " " + r.getKey();
            	outWriter.write(toString);        	
            	outWriter.newLine();
            }
            i++;
            if ( m_row1.getBooleanValue() ) break;
        }
 
        outWriter.close();
        if (missingCount > 0) {
            setWarningMessage("Skipped " + missingCount 
                    + " row(s) because of missing values");
        }
     // Molecule input or file
        String moleString = null; 
        BufferedDataTable inmol = inData[1];
      	if ( inmol != null )
      	{
      		DataTableSpec MolSpec = inmol.getDataTableSpec();
        	n_Column = columnSelect(MolSpec);
        
        // write out molecule file
        	if ( m_sdfColumn == null ) 
        		moleString = "molecules.smiles"; 
        	else 
        		moleString = "molecules.sdf"; 
        	File moleFile = new File(cwdString, moleString); 
        	BufferedWriter moleWriter = new BufferedWriter(new FileWriter(moleFile));
        	int moleIndex;
        	if ( m_sdfColumn != null ) 
        		moleIndex = inmol.getDataTableSpec().findColumnIndex(m_sdfColumn);
        	else if ( m_smilesColumn != null ) 
        		moleIndex = inmol.getDataTableSpec().findColumnIndex(m_smilesColumn);        
        	else if ( m_stringColumn != null ) 
        		moleIndex = inmol.getDataTableSpec().findColumnIndex(m_stringColumn);       
        	else 
        		throw new IllegalStateException("No suitable reagent column - run configure (again)");   
        	i = 0; 
        	missingCount = 0;
        	for (DataRow r : inmol) {
        		exec.checkCanceled();
        		DataCell c = r.getCell(moleIndex);
        		if (c.isMissing()) 
        			missingCount++;
        		else if ( m_sdfColumn != null ) {
        			SdfValue v = (SdfValue)c;
        			String toString = v.toString();
        			moleWriter.write(toString);
        			if (!toString.trim().endsWith("$$$$")) {
        				moleWriter.newLine();
        				moleWriter.append("$$$$");
        				moleWriter.newLine();
        			}
        		}
        		else if ( m_smilesColumn != null) {
        			SmilesValue v = (SmilesValue)c;
        			String toString = v.toString() + " " + r.getKey();
        			moleWriter.write(toString);        	
        			moleWriter.newLine();
        		}
        		else {
        			String toString = c.toString() + " " + r.getKey();
        			moleWriter.write(toString);        	
        			moleWriter.newLine();
        		}
        		i++;
        	}
        	moleWriter.close();
        	if (missingCount > 0) 
        		setWarningMessage("Skipped " + missingCount + " row(s) because of missing values");
        	maxProcs = inmol.getRowCount() - missingCount;
            if ( maxProcs < 1 ) 
              	maxProcs = 1;
            else if ( m_row1.getBooleanValue())
              	maxProcs = 1;
            else if ( maxProcs > m_process.getIntValue() )
               	maxProcs = m_process.getIntValue();
      	}
        else {
             moleString = m_search.getStringValue();
             maxProcs = m_process.getIntValue();
        }
        int n_threads = m_process.getIntValue();
            
        final int n_process = maxProcs;
     	for ( i=0; i<n_process; i++)
        {
        	File searchFile = new File(cwdString, "search" + i + ".dat"); 
        	if ( !searchFile.exists() || m_restart.getBooleanValue())
        	{
        		searchFile.delete();
        		File outputReport = new File(cwdString, "output" + i + ".log"); 
        		outputReport.delete();
        		File outputFile = new File(cwdString, "search" + i + ".sdf"); 
        		outputFile.delete();
        		File progressFile = new File(cwdString, "progress"+i+".dat"); 
        		progressFile.delete();
        	}
            else
            	setWarningMessage("Continue THINK("+i+") from check point file");
 
        }

        // execute THINK
        File cwdFile = new File (cwdString);
        try {
            exec.setProgress("Starting THINK(" + n_process + ")");
            Runtime rt = Runtime.getRuntime();
            Double d_cutoff = m_cutoff.getDoubleValue();
            // prepare the command string
            String s_filter = "NONE";
            if ( m_filter.getBooleanValue() ) {
            	s_filter = "FILTER";
            }
            String s_parent = "NONE";
            if ( m_parent.getBooleanValue() ) {
            	s_parent = "PARENT";
            }
            final Process[] procs = new Process[36];
            for (i=0; i<n_process;i++)
            {
            	String cmdString = System.getenv("THINK_EXEC") + "think THINK_EXEC:similarity.log output" + i + ".log " + fileString + " " + d_cutoff + " \"" + moleString + "\""
            			+ " " + s_filter + " " + s_parent + " " + i + " " + n_process + " " + n_threads;
            	// Go!
                procs[i] = rt.exec(cmdString, null, cwdFile);	
            	logger.info("THINK command line: '" + cmdString + "'");
             }

            final MutableBoolean procDone = new MutableBoolean(false);
            new Thread(new Runnable() {
                public void run() {
                    synchronized (procDone) {
                        while (!procDone.booleanValue()) {
                            try {
                                exec.checkCanceled();
                            } catch (CanceledExecutionException cee) {
                                // blow away the running external process
                            	int j;
                            	for (j=0;j<n_process;j++) {
                            		procs[j].destroy();
                            	}                                
                                return;
                            }
                            try {
                               	procDone.wait(CANCEL_CHECK_INTERVAL);
                                // Read progress file and set progress value       
                               	double d= 0.;
                      			String progFile="progress.dat";
                                int j;
                               	int l=0;
                       			String line = ""; 
                       			String newline="";
                       			String prefix="";
                               	for (j=0;j<n_process;j++) {
                                	progFile = "progress" + j + ".dat";
                                	prefix = "\nTHINK(" + j + "): ";
                                  	double done=0;
                                  	File progressReport = new File(cwdString, progFile);
                                  	if ( progressReport.exists() && progressReport.isFile() ) {
                                  		if ( progressReport.length() > 0 ) {
                                  			BufferedReader in = new BufferedReader(new FileReader(progressReport));
                                  			if ( (newline = in.readLine()) != null) {
                                  				l++;
                                  				done = Double.valueOf(newline).doubleValue();
                                  				d = d + done;
                                  			}
                                  			if ( done == 1.0 ) {
                                  				line += prefix + "Tabulating output";
                                  			}
                                  			else if ( (newline = in.readLine()) == null) {
                                  				line += prefix + "Running";
                                  			}
                                  			else {
                                  				line += prefix + newline;
                                  			}
                                  			in.close();
                                  		}
                                  	}
                                }
                               	if ( l > 0 ) {
                               		d = d / l;
                               	}
                                exec.setProgress( d, line.toString());
                            } catch (InterruptedException e) {
                                // do nothing
                            } catch ( FileNotFoundException e) {
                                // do nothing
                            } catch ( IOException e) {
                                // do nothing
                            }
                        }
                    }
                }

            }).start();

            // wait until the external process finishes.
            for (i=0; i< n_process;i++)
            {
            	exitVal[i] = procs[i].waitFor();
            }

            synchronized (procDone) {
                // this should terminate the check cancel thread
                procDone.setValue(true);
            }

            exec.checkCanceled();

            exec.setProgress("Wrapping up");
            for (i=0; i< n_process;i++)
            {
            	logger.info("THINK("+i+") terminated with exit code: " + exitVal[i]);
            }
        } catch (Throwable t) {
           	warningReport();
            logger.error("THINK failed (with exception)", t);
            throw new Exception(t);
        }
        for (i=0;i<n_process;i++)
        {
        if (exitVal[i] != 0) {
        	warningReport();
            // before we return barfing, we save the output in the failing list
            throw new IllegalStateException("THINK("+i+") failed (error code "
                    + exitVal[i] + ")");
        }
        }
        // Read the output and echo it to the console
         exec.setProgress("Creating KNIME tables");
        BufferedDataTable sumTable = readSummaryFile(exec);
        BufferedDataTable outTable = readCSVFile(exec);

        return new BufferedDataTable[]{outTable, sumTable};

    }

    private void warningReport( ) throws Exception {

        final int n_process = maxProcs;
        StringBuilder warningMessage = new StringBuilder();
        String cwdString = m_working.getStringValue();
        int i=0;
        for (i=0;i<n_process;i++)
        {
        File outputReport = new File(cwdString, "output"+i+".log"); 

        if ( outputReport.exists() && outputReport.isFile() ) {
       		if ( outputReport.length() > 0 ) {
       			BufferedReader in = new BufferedReader(new FileReader(outputReport));
       			String line;
       			while ((line = in.readLine()) != null) {
       				warningMessage.append( line + "\n");
       			}               
       			in.close();
       		}
       		else { 
                warningMessage.append( "Report from THINK(" + i + ")is blank");      	     			
       		}
        }
       	else {
            warningMessage.append( "No report from THINK("+i+")");      	
       	}
        }
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
   }
    
    private BufferedDataTable readCSVFile(final ExecutionContext exec) throws Exception { 

        String cwdString =  m_working.getStringValue();
        HashSet<String> titles = new HashSet<String>();
        String title;
        String line;
        boolean headerRead = false;
        boolean columnsCreated = false;
        DataColumnSpec[] allColSpecs;
        DataTableSpec outputSpec;
        BufferedDataContainer cont =null;
        RowKey key =null;
        int count =0;
        int colcount =0;
        int rowcount =0;
        char chr='0';
        int keys = 0;
        int ilast = 0;
        String fieldNames[] = new String[100];
        int fieldTypes[] = new int[100];
        final int n_process = maxProcs;
        int j=0;
     	boolean headerSkip=false;
         for (j=0;j<n_process;j++)
         {
         File outFile = new File (cwdString, "similarity"+j+".csv");
     	if (!(outFile.exists() && outFile.isFile())) {
            	warningReport();
             throw new IllegalStateException("THINK("+j+") didn't produce any output"
                     + " at the specified location ('"
                     + outFile.getAbsolutePath() + "')");
         }
         BufferedReader in = new BufferedReader(new FileReader(outFile));

        while ((line = in.readLine()) != null) {
            exec.checkCanceled();
// Extract column headers
            if ( !headerRead ) {
                headerRead = true;
                colcount =0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) 
                        chr = line.charAt(i);
                    else
                        chr = ',';
                    if ( chr == ',' ) { 
                        fieldNames[colcount] = line.substring(ilast,i);
                        if ( line.substring(ilast,i).equalsIgnoreCase("SMILES"))
                        	fieldTypes[colcount]= 2;
                        else if ( line.substring(ilast,i).equalsIgnoreCase("NAME"))
                        	fieldTypes[colcount]= 1;
                        else if ( line.substring(ilast,i).equalsIgnoreCase("KEYS"))
                        	fieldTypes[colcount]= 3;
                        ilast = i+1;
                        colcount++;
                    }
                }
                colcount--;
            }
// Skip headers of second and subsequent output files
            else if (headerSkip) 
            	headerSkip=false;
// Load data
            else {
                DataCell[] cells = new DataCell[colcount];
                rowcount++;
                count =0;                 
                double d_value;
                 for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) 
                        chr = line.charAt(i);
                    else
                        chr = ',';
                    if ( chr == ',' ) { 
                        if ( count == 0 ) {
                            title = line.substring(0,i);
                            if (title.length() == 0) 
                                title = "Mol " + rowcount;
                            if (titles.contains(title)) 
                                title = title + "-" + rowcount;
                            titles.add(title);
                            key = new RowKey(title);                           
                        }
                        else if  ( fieldTypes[count] == 2 ) 
                            cells[count-1] = new SmilesCell(line.substring(ilast,i));
                        else if  ( fieldTypes[count]== 3) 
                            cells[count-1] = new StringCell(line.substring(ilast,i));
                        else if ( count <= colcount ) {
                          try {
                            d_value = Double.valueOf(line.substring(ilast,i)).doubleValue();
                            cells[count-1] = new DoubleCell(d_value);
                            if ( fieldTypes[count] == 0 )
                            	fieldTypes[count] = 4;
                          } catch (Throwable t) {
                            if ( fieldTypes[count] == 0 ) {
                            	fieldTypes[count] = 3;
                                cells[count-1] = new StringCell(line.substring(ilast,i));
                            }
                            else
                                cells[count-1] = DataType.getMissingCell();
                          }
                        }
                        count ++;
                        ilast = i + 1;
                    }
                 }
  // Create columns?
                 if (!columnsCreated )
                 {
                     allColSpecs = new DataColumnSpec[colcount];
                     columnsCreated = true;
  // count= 0 will be molecule names
                     for (count=1; count<=colcount; count++)
                     {                   	 
                       if  ( fieldTypes[count] == 2 ) 
                         allColSpecs[count-1] = new DataColumnSpecCreator(fieldNames[count],SmilesCell.TYPE).createSpec();
                       else if  ( fieldTypes[count] == 4 ) 
                         allColSpecs[count-1] = new DataColumnSpecCreator(fieldNames[count], DoubleCell.TYPE).createSpec();
                       else if ( fieldTypes[count] == 3 )
                         allColSpecs[count-1] = new DataColumnSpecCreator(fieldNames[count],StringCell.TYPE).createSpec();
                       else
                    	 allColSpecs[count-1] = new DataColumnSpecCreator("Unknown",StringCell.TYPE).createSpec();
                     }
                     outputSpec = new DataTableSpec(allColSpecs);
                     cont = exec.createDataContainer(outputSpec);               	 
                 }
                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
            }
        }
        in.close();
        headerSkip=true;
        }
        cont.close();


        return cont.getTable();
    }

    private BufferedDataTable readOutputFile(final ExecutionContext exec) throws Exception {

        final int n_process = maxProcs;
    	String cwdString = m_working.getStringValue();
    	
        HashSet<String> titles = new HashSet<String>();
        String title;
        String line;
        boolean headerRead = false;
        DataColumnSpec[] allColSpecs;
        DataTableSpec outputSpec;
        BufferedDataContainer cont =null;
        RowKey key =null;
        int count =0;
        int query =0;
        int rowcount =0;
        char chr='0';
        int similarity = 0;
        int nocols=2;
        if ( !m_row1.getBooleanValue() ) {
        	nocols = 3;
        }
        int j=0;
    	boolean headerSkip=false;
        for (j=0;j<n_process;j++)
        {
        File outFile = new File (cwdString, "similarity"+j+".csv");
    	if (!(outFile.exists() && outFile.isFile())) {
           	warningReport();
            throw new IllegalStateException("THINK("+j+") didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
        BufferedReader in = new BufferedReader(new FileReader(outFile));

        while ((line = in.readLine()) != null) {
            exec.checkCanceled();
// Extract column headers
            if ( !headerRead ) {
                headerRead = true;
                count =0;
                for ( int i=0; i<line.length(); i++ ) {
                    if ( line.charAt(i) == ',' ) count ++;
                }
// Create column definitions
                allColSpecs = new DataColumnSpec[nocols];
                count =0;
                int ilast=0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else {
                        chr = ',';
                    }
                    if ( chr == ',' ) {
                        if  ( count == 1 ) {
                            allColSpecs[0] = 
                            new DataColumnSpecCreator(line.substring(ilast,i),SmilesCell.TYPE).createSpec();
                        }
                        else if  ( line.substring(ilast,i).equals("Similarity") ) {
                        	similarity = count;
                            allColSpecs[1] = 
                            new DataColumnSpecCreator(line.substring(ilast,i), DoubleCell.TYPE).createSpec();
                        }
                        else if  ( nocols == 3 & line.substring(ilast,i).equals("QUERY") ) {
                        	query = count;
                            allColSpecs[2] = 
                            new DataColumnSpecCreator(line.substring(ilast,i), StringCell.TYPE).createSpec();
                        }
                        count ++;
                        ilast = i + 1;
                    }
                }
                outputSpec = new DataTableSpec(allColSpecs);
                cont = exec.createDataContainer(outputSpec);
            	}
// Skip headers of second and subsequent output files
            else if (headerSkip) {
            	headerSkip=false;
            }
// Load data
            else {
                DataCell[] cells = new DataCell[nocols];
                rowcount++;
                count =0;                  
                double d_value;
                int ilast=0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else
                        chr = ',';
                    if ( chr == ',' ) { 
                        if ( count == 0 ) {
                            title = line.substring(0,i);
                            if (title.length() == 0) {
                                title = "Mol " + rowcount;
                            }
                            if (titles.contains(title)) {
                                title = title + "-" + rowcount;
                            }
                            titles.add(title);
                            key = new RowKey(title);                           
                        }
                        else if  ( count == 1 ) {
                            cells[0] = new SmilesCell(line.substring(ilast,i));   	
                        }
                        else if  ( count == similarity ) {
                        try {
                            d_value = Double.valueOf(line.substring(ilast,i)).doubleValue();
                            cells[1] = new DoubleCell(d_value);
                        } catch (Throwable t) {
                            cells[1] = DataType.getMissingCell();
                        }
                        }
                        else if ( count == query ) {
                        	cells[2] = new StringCell(line.substring(ilast,i));
                        }
                        count ++;
                        ilast = i + 1;
                    }
                }
                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
            }
        }
        in.close();
        headerSkip=true;
        }
        cont.close();

        return cont.getTable();
    }
    private BufferedDataTable readSummaryFile(final ExecutionContext exec) throws Exception {

        final int n_process = maxProcs;
    	String cwdString = m_working.getStringValue();
        HashSet<String> titles = new HashSet<String>();
        String title;
        String line;
        boolean headerRead = false;
        DataColumnSpec[] allColSpecs;
        DataTableSpec outputSpec;
        BufferedDataContainer cont =null;
        RowKey key =null;
        int count =0;
        int rowcount =0;
        char chr='0';
        int j=0;
        boolean headerSkip=false;
        for (j=0;j<n_process;j++)
        {
        File outFile = new File (cwdString, "summary"+j+".csv");
    	if (!(outFile.exists() && outFile.isFile())) {
    		warningReport();
    		throw new IllegalStateException("THINK("+j+") didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();
// Extract column headers
            if ( !headerRead ) {
                headerRead = true;
 // Create column definitions
                allColSpecs = new DataColumnSpec[4];
                count =0;
                int ilast=0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else {
                        chr = ',';
                    }
                    if ( chr == ',' ) {
                        
                       if  ( count == 3 ) {
                            allColSpecs[count-1] = 
                            new DataColumnSpecCreator(line.substring(ilast,i),StringCell.TYPE).createSpec();
                        }
                        else if  ( count == 4 ) {
                            allColSpecs[count-1] = 
                            new DataColumnSpecCreator(line.substring(ilast,i), DoubleCell.TYPE).createSpec();
                        }
                        else if ( count > 0 ) {
                            allColSpecs[count-1] = 
                            new DataColumnSpecCreator(line.substring(ilast,i), IntCell.TYPE).createSpec();
                        }
                        count ++;
                        ilast = i + 1;
                    }
                }
                outputSpec = new DataTableSpec(allColSpecs);
                cont = exec.createDataContainer(outputSpec);
            	}
            else if (headerSkip) {
            	headerSkip=false;
            }
// Load data
            else {
                DataCell[] cells = new DataCell[4];
                rowcount++;
                count =0;                  
                double d_value;
                int n_value;
                int ilast=0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else
                        chr = ',';
                    if ( chr == ',' ) { 
                        if ( count == 0 ) {
                            title = line.substring(0,i);
                            if (title.length() == 0) {
                                title = "Mol " + rowcount;
                            }
                            if (titles.contains(title)) {
                                title = title + "-" + rowcount;
                            }
                            titles.add(title);
                            key = new RowKey(title);                           
                        }
                        else if  ( count == 3 ) {
                            cells[count-1] = new StringCell(line.substring(ilast,i));   	
                        }
                        else if  ( count == 4 ) {
                        try {
                            d_value = Double.valueOf(line.substring(ilast,i)).doubleValue();
                            cells[count-1] = new DoubleCell(d_value);
                        } catch (Throwable t) {
                            cells[count-1] = DataType.getMissingCell();
                        }
                        }
                        else {
                            try {
                                n_value = Integer.valueOf(line.substring(ilast,i)).intValue();
                                cells[count-1] = new IntCell(n_value);
                            } catch (Throwable t) {
                                cells[count-1] = DataType.getMissingCell();
                            }
                            }
                        count ++;
                        ilast = i + 1;
                    }
                }
                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
             }
        }

        in.close();
        headerSkip=true;
        }
        cont.close();

        return cont.getTable();
 
    }

    /**
     * @see org.knime.core.node.NodeModel#reset()
     */
    @Override
    protected void reset() {
        // TODO Code executed on reset.
        // Models build during execute are cleared here.
        // Also data handled in load/saveInternals will be erased here.
    }

   
    /**
     * @see org.knime.core.node.NodeModel
     *      #saveSettingsTo(org.knime.core.node.NodeSettings)
     */
    @Override
    protected void saveSettingsTo(final NodeSettingsWO settings) {

        // TODO save user settings to the config object.
		
		m_search.saveSettingsTo(settings);
	    m_filter.saveSettingsTo(settings);
	    m_parent.saveSettingsTo(settings);
		m_smiles.saveSettingsTo(settings);
		m_cutoff.saveSettingsTo(settings);
		m_row1.saveSettingsTo(settings);
		m_restart.saveSettingsTo(settings);
		m_process.saveSettingsTo(settings);
		m_working.saveSettingsTo(settings);

    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #loadValidatedSettingsFrom(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void loadValidatedSettingsFrom(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO load (valid) settings from the config object.
        // It can be safely assumed that the settings are valided by the 
        // method below.
        
        m_search.loadSettingsFrom(settings);
        m_filter.loadSettingsFrom(settings);
        m_parent.loadSettingsFrom(settings);
        m_smiles.loadSettingsFrom(settings);
        m_cutoff.loadSettingsFrom(settings);
        m_row1.loadSettingsFrom(settings);
        m_restart.loadSettingsFrom(settings);
        m_process.loadSettingsFrom(settings);
        m_working.loadSettingsFrom(settings);

    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #validateSettings(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void validateSettings(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO check if the settings could be applied to our model
        // e.g. if the count is in a certain range (which is ensured by the
        // SettingsModel).
        // Do not actually set any values of any member variables.

        m_search.validateSettings(settings);
        m_filter.validateSettings(settings);
        m_parent.validateSettings(settings);
        m_smiles.validateSettings(settings);
        m_cutoff.validateSettings(settings);
        m_row1.validateSettings(settings);
        m_restart.validateSettings(settings);
        m_process.validateSettings(settings);
        m_working.validateSettings(settings);

    }
    
    /**
     * @see org.knime.core.node.NodeModel #loadInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void loadInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
        
		// TODO load internal data. 
		// Everything handed to output ports is loaded automatically (data
		// returned by the execute method, models loaded in loadModelContent,
		// and user settings set through loadSettingsFrom - is all taken care 
		// of). Load here only the other internals that need to be restored
		// (e.g. data used by the views).

    }
    
    /**
     * @see org.knime.core.node.NodeModel #saveInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void saveInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
       
       	// TODO save internal models. 
		// Everything written to output ports is saved automatically (data
		// returned by the execute method, models saved in the saveModelContent,
		// and user settings saved through saveSettingsTo - is all taken care 
		// of). Save here only the other internals that need to be preserved
		// (e.g. data used by the views).

    }

}
