package com.treweren.similarity;

import javax.swing.JFileChooser;
import java.awt.FlowLayout;

import org.knime.core.node.defaultnodesettings.DefaultNodeSettingsPane;
import org.knime.core.node.defaultnodesettings.DialogComponentBoolean;
import org.knime.core.node.defaultnodesettings.DialogComponentFileChooser;
import org.knime.core.node.defaultnodesettings.DialogComponentNumber;
import org.knime.core.node.defaultnodesettings.DialogComponentString;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelString;
import org.knime.core.node.defaultnodesettings.SettingsModelDoubleBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;


/**
 * <code>NodeDialog</code> for the "Similarity" Node.
 * Performs similarity search
 *
 * This node dialog derives from {@link DefaultNodeSettingsPane} which allows
 * creation of a simple dialog with standard components. If you need a more 
 * complex dialog please derive directly from 
 * {@link org.knime.core.node.NodeDialogPane}.
 * 
 * @author Treweren Consultants
 */
public class SimilarityNodeDialog extends DefaultNodeSettingsPane {

    /**
     * New pane for configuring Similarity node dialog.
     * This is just a suggestion to demonstrate possible default dialog
     * components.
     */
    protected SimilarityNodeDialog() {
        super();
        DialogComponentFileChooser c_search = new DialogComponentFileChooser(
                new SettingsModelString (
                        SimilarityNodeModel.CFGKEY_SEARCH,
                        SimilarityNodeModel.DEFAULT_SEARCH ),
                        "Similarity search", JFileChooser.OPEN_DIALOG, "smi", "csv", "smiles", "sdf" );
        addDialogComponent(c_search);
        c_search.setToolTipText("The molecules being searched are in this file");
        c_search.setBorderTitle("Search molecules in file");
        
        DialogComponentBoolean c_filter = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        SimilarityNodeModel.CFGKEY_FILTER,
                        SimilarityNodeModel.DEFAULT_FILTER),
                        "Apply drug-like filters" );
        addDialogComponent(c_filter);
        c_filter.setToolTipText("Eliminates molecules which have properties which are not drug-like");
        c_filter.getComponentPanel().setLayout( new FlowLayout(0));
        
        final DialogComponentBoolean c_parent = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        SimilarityNodeModel.CFGKEY_PARENT,
                        SimilarityNodeModel.DEFAULT_PARENT),
                        "Convert salts to parent form" );
        addDialogComponent(c_parent);
        c_parent.setToolTipText("Converts molecules to parent(neutral) form and eliminates counter-ions(salts)");
        c_parent.getComponentPanel().setLayout( new FlowLayout(0));
           
        DialogComponentNumber c_cutoff = new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                        SimilarityNodeModel.CFGKEY_CUTOFF,
                        SimilarityNodeModel.DEFAULT_CUTOFF, 0., 1. ),
                        "Similarity cutoff for hits", .05, 5 ) ;
      
        addDialogComponent(c_cutoff);
        c_cutoff.setToolTipText("Molecules which are more similar than this cutoff are considered hits and retained in the output table");
        c_cutoff.getComponentPanel().setLayout( new FlowLayout(2));
        
        createNewTab("THINK setup");         

        DialogComponentBoolean c_column = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        SimilarityNodeModel.CFGKEY_SMILES,
                        SimilarityNodeModel.DEFAULT_SMILES),
                        "Use SMILES in preference to SD column" );
        addDialogComponent(c_column);
        c_column.getComponentPanel().setLayout( new FlowLayout(0));
        c_column.setToolTipText("The molecule data is transferred to THINK using SD or SMILES format depending on this setting");
        
        DialogComponentBoolean c_row = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        SimilarityNodeModel.CFGKEY_ROW1,
                        SimilarityNodeModel.DEFAULT_ROW1),
                        "Use first row/molecule (not all)" );
        
        addDialogComponent(c_row);
        c_row.getComponentPanel().setLayout( new FlowLayout(0));
        c_row.setToolTipText("The data transferred may be limited to the first row using this control");

        DialogComponentBoolean c_restart = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        SimilarityNodeModel.CFGKEY_RESTART,
                        SimilarityNodeModel.DEFAULT_RESTART),
                        "Restart search (instead of continuing previous search)" );
        addDialogComponent(c_restart);
        c_restart.setToolTipText("Check point files (searchn.dat) are deleted before THINK restarts");
        c_restart.getComponentPanel().setLayout( new FlowLayout(0));

         DialogComponentNumber c_process = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        SimilarityNodeModel.CFGKEY_PROCESS,
                        SimilarityNodeModel.DEFAULT_PROCESS,
                        1, 36 ),
                        "Maximum number of THINK processes", /*step*/ 1, /*componentwidth*/ 2);
        
        addDialogComponent(c_process);
        c_process.getComponentPanel().setLayout( new FlowLayout(2));
        c_process.setToolTipText("The search is divided between the THINK processes");
  
        DialogComponentString c_working = new DialogComponentString(
                new SettingsModelString (
                        SimilarityNodeModel.CFGKEY_WORKING,
                        SimilarityNodeModel.DEFAULT_WORKING),
                        "Working folder" );
        addDialogComponent(c_working);
        c_working.getComponentPanel().setLayout( new FlowLayout(2));
        c_working.setToolTipText("The temporary files used to transfer the data to and from THINK are placed in this folder");
                     
    }
}
