package com.treweren.conformers;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.BufferedReader;
import java.io.FileReader;
import java.util.HashSet;


import org.knime.base.node.io.filereader.FileAnalyzer;
import org.knime.base.node.io.filereader.FileReaderNodeSettings;
import org.knime.base.node.io.filereader.FileReaderSettings;
import org.knime.base.node.io.filereader.FileTable;
import org.knime.base.node.io.filereader.SettingsStatus;
import org.knime.chem.types.SdfValue;
import org.knime.chem.types.SmilesValue;
import org.knime.chem.types.SdfCell;
import org.knime.core.data.DataCell;
import org.knime.core.data.DataColumnSpec;
import org.knime.core.data.DataColumnSpecCreator;
import org.knime.core.data.DataRow;
import org.knime.core.data.DataTableSpec;
import org.knime.core.data.RowKey;
import org.knime.core.data.def.DefaultRow;
import org.knime.core.data.def.DoubleCell;
import org.knime.core.data.def.IntCell;
import org.knime.core.data.def.StringCell;
import org.knime.core.node.BufferedDataContainer;
import org.knime.core.node.BufferedDataTable;
import org.knime.core.node.CanceledExecutionException;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelDoubleBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelString;
import org.knime.core.node.ExecutionContext;
import org.knime.core.node.ExecutionMonitor;
import org.knime.core.node.InvalidSettingsException;
import org.knime.core.node.NodeLogger;
import org.knime.core.node.NodeModel;
import org.knime.core.node.NodeSettingsRO;
import org.knime.core.node.NodeSettingsWO;
import org.knime.core.util.MutableBoolean;




/**
 * This is the model implementation of Conformers.
 * Generate 3D conformers for a molecule
 *
 * @author Treweren Consultants
 */
public class ConformersNodeModel extends NodeModel {
    
    // the logger instance
    private static final NodeLogger logger = NodeLogger
            .getLogger(ConformersNodeModel.class);
    private static int n_Column = 0;
    private static String m_sdfColumn = null;
    private static String m_smilesColumn = null;
    private static String m_stringColumn = null;
    private static final int CANCEL_CHECK_INTERVAL = 1000;

    
    private static final DataColumnSpecCreator CREATOR =
        new DataColumnSpecCreator("Molecule", SdfCell.TYPE);
    private static final DataTableSpec SPEC =
        new DataTableSpec(CREATOR.createSpec());
       
    /** the settings key which is used to retrieve and 
        store the settings (from the dialog or from a settings file)    
       (package visibility to be usable from the dialog). */
	static final String CFGKEY_SMILES = "Smiles";
	static final String CFGKEY_ROW1 = "Row1";
	static final String CFGKEY_WORKING = "Working";
	static final String CFGKEY_SINGLE = "Single";
	static final String CFGKEY_CONJUGATED = "Conjugated";
	static final String CFGKEY_CROWDED = "Crowded";
	static final String CFGKEY_ALPHA = "Alpha";
	static final String CFGKEY_AMIDE = "Amide";
	static final String CFGKEY_RING = "Ring";
	static final String CFGKEY_GENERATION = "Generation";
	static final String CFGKEY_SAMPLES = "Samples";
	static final String CFGKEY_CPKRATIO = "CPKratio";
	static final String CFGKEY_TIMELIMIT = "TimeLimit";
	static final String CFGKEY_SYSLIMIT = "SysLimit";
	static final String CFGKEY_BONDLIMIT = "BondLimit";
	
	/** initial default count value. */
	static final boolean DEFAULT_SMILES = false;
	static final boolean DEFAULT_ROW1 = true;
	static final String DEFAULT_WORKING = System.getenv("THINK_WORKING");
	static final int DEFAULT_SINGLE = 3;
	static final int DEFAULT_CONJUGATED = 2;
	static final int DEFAULT_CROWDED = 2;
	static final int DEFAULT_ALPHA = 6;
	static final int DEFAULT_AMIDE = 0;
	static final int DEFAULT_RING = 0;
	static final String DEFAULT_GENERATION = "Systematic";
	static final int DEFAULT_SAMPLES = 1000;
	static final double DEFAULT_CPKRATIO = 0.6;
	static final double DEFAULT_TIMELIMIT = 60.;
	static final int DEFAULT_SYSLIMIT = 0;
	static final int DEFAULT_BONDLIMIT = 10;
	
	// example value: the models count variable filled from the dialog 
	// and used in the models execution method. The default components of the
	// dialog work with "SettingsModels".

	private final SettingsModelBoolean m_smiles =
		new SettingsModelBoolean(ConformersNodeModel.CFGKEY_SMILES,
                    ConformersNodeModel.DEFAULT_SMILES );

	private final SettingsModelBoolean m_row1 =
		new SettingsModelBoolean(ConformersNodeModel.CFGKEY_ROW1,
                    ConformersNodeModel.DEFAULT_ROW1 );

	private final SettingsModelString m_working =
		new SettingsModelString(ConformersNodeModel.CFGKEY_WORKING, ConformersNodeModel.DEFAULT_WORKING );

	private final SettingsModelIntegerBounded m_single =
		new SettingsModelIntegerBounded(ConformersNodeModel.CFGKEY_SINGLE,
                    ConformersNodeModel.DEFAULT_SINGLE, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_conjugated =
		new SettingsModelIntegerBounded(ConformersNodeModel.CFGKEY_CONJUGATED,
                    ConformersNodeModel.DEFAULT_CONJUGATED, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_crowded =
		new SettingsModelIntegerBounded(ConformersNodeModel.CFGKEY_CROWDED,
                    ConformersNodeModel.DEFAULT_CROWDED, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_alpha =
		new SettingsModelIntegerBounded(ConformersNodeModel.CFGKEY_ALPHA,
                    ConformersNodeModel.DEFAULT_ALPHA, 0, Integer.MAX_VALUE);


	private final SettingsModelIntegerBounded m_amide =
		new SettingsModelIntegerBounded(ConformersNodeModel.CFGKEY_AMIDE,
                    ConformersNodeModel.DEFAULT_AMIDE, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_ring =
		new SettingsModelIntegerBounded(ConformersNodeModel.CFGKEY_RING,
                    ConformersNodeModel.DEFAULT_RING, 0, Integer.MAX_VALUE);
	
	private final SettingsModelString m_generation =
		new SettingsModelString(ConformersNodeModel.CFGKEY_GENERATION,
                    ConformersNodeModel.DEFAULT_GENERATION);

	private final SettingsModelIntegerBounded m_samples =
		new SettingsModelIntegerBounded(ConformersNodeModel.CFGKEY_SAMPLES,
                    ConformersNodeModel.DEFAULT_SAMPLES, 0, Integer.MAX_VALUE);

	private final SettingsModelDoubleBounded m_cpkratio =
		new SettingsModelDoubleBounded(ConformersNodeModel.CFGKEY_CPKRATIO,
                    ConformersNodeModel.DEFAULT_CPKRATIO,
                    0., 5.0);
	
	private final SettingsModelDoubleBounded m_timelimit =
		new SettingsModelDoubleBounded(ConformersNodeModel.CFGKEY_TIMELIMIT,
                    ConformersNodeModel.DEFAULT_TIMELIMIT,
                    0., 1440.);

	private final SettingsModelIntegerBounded m_syslimit =
		new SettingsModelIntegerBounded(ConformersNodeModel.CFGKEY_SYSLIMIT,
                    ConformersNodeModel.DEFAULT_SYSLIMIT, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_bondlimit =
		new SettingsModelIntegerBounded(ConformersNodeModel.CFGKEY_BONDLIMIT,
                    ConformersNodeModel.DEFAULT_BONDLIMIT, 0, Integer.MAX_VALUE);
	
    /**
     * Constructor for the node model.
     */
    protected ConformersNodeModel() {
    
        // TODO one incoming port and one outgoing port is assumed
        super(1, 1);
    }

    /**
     * @see org.knime.core.node.NodeModel #execute(BufferedDataTable[],
     *      ExecutionContext)
     */
    @Override
    protected DataTableSpec[] configure(final DataTableSpec[] inSpecs) throws InvalidSettingsException {

        if ( System.getenv("THINK_EXEC") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_EXEC is unset - It should be set to the folder containing the THINK software.");       	  		
    	}
        if ( System.getenv("THINK_WORKING") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_WORKING is not set to the folder in which working files will be created.");       	  		
    	}

        DataTableSpec in = inSpecs[0];
        n_Column = columnSelect (in);
        return new DataTableSpec[] {null};
    }
    protected int columnSelect ( final DataTableSpec in ) throws  InvalidSettingsException {

        StringBuilder warningMessage = new StringBuilder();
        int n_column=0;
        int sdfColCount = 0;
        int n_sdf=0;
        int smilesColCount = 0;
        int n_smiles=0;
        int stringColCount = 0;
        int n_string=0;
        for (int i = 0; i < in.getNumColumns(); i++) {
            DataColumnSpec s = in.getColumnSpec(i);
            if (s.getType().isCompatible(SdfValue.class)) {
                if (m_sdfColumn == null) {
                    m_sdfColumn = in.getColumnSpec(i).getName();
                    n_sdf = i;
                }
                sdfColCount++;
            }
            else if ( s.getType().isCompatible(SmilesValue.class)) {
            	if (m_smilesColumn == null ) {
             		m_smilesColumn = in.getColumnSpec(i).getName();
             		n_smiles=i;
              	}
               	smilesColCount++;
            }
            else if ( in.getColumnSpec(i).getName().equalsIgnoreCase("smiles"))  {
            	if (m_stringColumn == null ) {
             		m_stringColumn = in.getColumnSpec(i).getName();
             		n_string=i;
              	}
               	stringColCount++;
            }
        }
        if ( smilesColCount == 0 & sdfColCount == 0 & stringColCount == 0) {
            throw new InvalidSettingsException(
                    "No column with Smiles or SD file compatible type");       	
        }
        else if ( ( m_smiles.getBooleanValue() | sdfColCount == 0 ) & smilesColCount > 0 ) {
        	m_sdfColumn = null;
        	m_stringColumn = null;
        	n_column = n_smiles;
        	if (smilesColCount > 1) {
                warningMessage.append("More than one Smiles compatible column in " 
                        + "input, using column \"" + m_smilesColumn + "\".");
        	}
        }
        else if ( sdfColCount > 0 ) {
           	m_smilesColumn = null;
        	m_stringColumn = null;
        	n_column = n_sdf;
        	if (sdfColCount > 1) {
                warningMessage.append("More than one SDF compatible column in " 
                        + "input, using column \"" + m_sdfColumn + "\".");
        	}
        }
        else if ( stringColCount > 0 ) {
        	m_smilesColumn = null;
        	m_sdfColumn = null;
        	n_column = n_string;
           	if (stringColCount > 1) {
                warningMessage.append("More than one Smiles string compatible column in " 
                        + "input, using column \"" + m_stringColumn + "\".");
           	}
        }
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
        return n_column;
    }

 
    /**
     * @see org.knime.core.node.NodeModel #execute(BufferedDataTable[],
     *      ExecutionContext)
     */
    @Override
    protected BufferedDataTable[] execute(final BufferedDataTable[] inData,
            final ExecutionContext exec) throws Exception {

        notifyViews(null);

        final String cwdString = m_working.getStringValue();
        // clean up 
        File outputReport = new File(cwdString, "output.log"); 
        outputReport.delete();
        File outputFile = new File(cwdString, "conformers.sdf"); 
        outputFile.delete();
        File progressFile = new File(cwdString, "progress0.dat"); 
        progressFile.delete();

        BufferedDataTable in = inData[0];

        if ( n_Column == 0 ) {
        	DataTableSpec InSpec = in.getDataTableSpec();
        	n_Column = columnSelect(InSpec);
        }    

        // write out input file
        String fileString = null; 
        if ( m_sdfColumn == null ) {
            fileString = "query.smiles"; 
        }
        else {
            fileString = "query.sdf"; 
        }
        File inFile = new File(cwdString, fileString); 
        BufferedWriter outWriter = new BufferedWriter(new FileWriter(inFile));
        int colIndex;
        if ( m_sdfColumn != null ) {
        	colIndex = in.getDataTableSpec().findColumnIndex(m_sdfColumn);
        }
        else if ( m_smilesColumn != null ) {
        	colIndex = in.getDataTableSpec().findColumnIndex(m_smilesColumn);        
        }
        else {
            colIndex = in.getDataTableSpec().findColumnIndex(m_stringColumn);        
        }
        	  
        final double count = in.getRowCount(); // floating point operations
        if ( count > 1 & m_row1.getBooleanValue() ) {
            setWarningMessage("Using only first row/molecule");
         }
        int i = 0; 
        int missingCount = 0;
        for (DataRow r : in) {
            exec.checkCanceled();
            DataCell c = r.getCell(colIndex);
            if (c.isMissing()) {
                missingCount++;
            } 
            else if ( m_sdfColumn != null ) {
                SdfValue v = (SdfValue)c;
                String toString = v.toString();
                outWriter.write(toString);
                if (!toString.trim().endsWith("$$$$")) {
                    outWriter.newLine();
                    outWriter.append("$$$$");
                    outWriter.newLine();
                }
            }
            else if ( m_smilesColumn != null ){
                SmilesValue v = (SmilesValue)c;
                String toString = v.toString() + " " + r.getKey();
                outWriter.write(toString);        	
                outWriter.newLine();
            }
            else {
            	String toString = c.toString() + " " + r.getKey();
            	outWriter.write(toString);        	
            	outWriter.newLine();
            }
             i++;
             if ( m_row1.getBooleanValue() ) break;
       }

        outWriter.close();
        if (missingCount > 0) {
            setWarningMessage("Skipped " + missingCount 
                    + " row(s) because of missing values");
        }

        // execute THINK
        File cwdFile = new File (cwdString);
        int exitVal;
        try {
            exec.setProgress("Starting THINK");
            Runtime rt = Runtime.getRuntime();
            int n_single = m_single.getIntValue();
            int n_conjugated = m_conjugated.getIntValue();
            int n_crowded = m_crowded.getIntValue();
            int n_alpha = m_alpha.getIntValue();
            int n_amide = m_amide.getIntValue();
            int n_ring = m_ring.getIntValue();
            String s_generation = m_generation.getStringValue();
            int n_samples = m_samples.getIntValue();
            double d_cpkratio = m_cpkratio.getDoubleValue();
            double d_timelimit = m_timelimit.getDoubleValue();
            int n_syslimit = m_syslimit.getIntValue();
            int n_bondlimit = m_bondlimit.getIntValue();

      
            // prepare the command string
            String cmdString = System.getenv("THINK_EXEC") + "think THINK_EXEC:conformers.log output.log " + fileString + " " + n_samples + " " + s_generation + " " 
            + n_single + " " + n_conjugated + " " + n_crowded + " " + n_alpha  + " " + n_amide + " " + n_ring
            + " " + d_cpkratio + " " + d_timelimit + " " + n_syslimit + " " + n_bondlimit;

            // Go!
            logger.info("THINK command line: '" + cmdString + "'");
            final Process proc = rt.exec(cmdString, null, cwdFile);
            final MutableBoolean procDone = new MutableBoolean(false);
            new Thread(new Runnable() {
                public void run() {
                    synchronized (procDone) {
                        while (!procDone.booleanValue()) {
                            try {
                                exec.checkCanceled();
                            } catch (CanceledExecutionException cee) {
                                // blow away the running external process
                                proc.destroy();
                                return;
                            }
                            try {
                               	procDone.wait(CANCEL_CHECK_INTERVAL);
                                // Read progress file and set progress value       
                               	double d= 0.;
                                File progressReport = new File(cwdString, "progress0.dat");
                                if ( progressReport.exists() && progressReport.isFile() ) {
                               		if ( progressReport.length() > 0 ) {
                               			BufferedReader in = new BufferedReader(new FileReader(progressReport));
                               			String line;
                               			if ( (line = in.readLine()) != null) {
                               			  d = Double.valueOf(line).doubleValue();
                               			}               
                              			if ( (line = in.readLine()) == null) {
                              				line = "Running THINK";
                              			}
                               			in.close();
                                        exec.setProgress( d, line.toString());
                               		}
                                }
                            } catch (InterruptedException e) {
                                // do nothing
                            } catch ( FileNotFoundException e) {
                                // do nothing
                            } catch ( IOException e) {
                                // do nothing
                            }
                        }
                    }
                }

            }).start();

            // wait until the external process finishes.
            exitVal = proc.waitFor();

            synchronized (procDone) {
                // this should terminate the check cancel thread
                procDone.setValue(true);
            }

            exec.checkCanceled();

            exec.setProgress("Wrapping up");
            logger.info("THINK terminated with exit code: " + exitVal);
        } catch (Throwable t) {
        	warningReport();
            logger.error("THINK failed (with exception)", t);
            throw new Exception(t);
        }

        if (exitVal != 0) {
            // before we return barfing, we save the output in the failing list
        	warningReport();
            throw new IllegalStateException("THINK failed (error code "
                    + exitVal + ")");
        }

        exec.setProgress("Creating KNIME tables");
        BufferedDataTable outTable = readOutputFile(exec);

        return new BufferedDataTable[]{outTable};

    }

    private void warningReport( ) throws Exception {

    	StringBuilder warningMessage = new StringBuilder();
        String cwdString = m_working.getStringValue();
        File outputReport = new File(cwdString, "output.log"); 

        if ( outputReport.exists() && outputReport.isFile() ) {
       		if ( outputReport.length() > 0 ) {
       			BufferedReader in = new BufferedReader(new FileReader(outputReport));
       			String line;
       			while ((line = in.readLine()) != null) {
       				warningMessage.append( line + "\n");
       			}               
       			in.close();
       		}
       		else { 
                warningMessage.append( "Report from THINK is blank");      	     			
       		}
        }
       	else {
            warningMessage.append( "No report from THINK");      	
       	}	
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
   }

    private BufferedDataTable readOutputFile(final ExecutionContext exec)
            throws Exception {

    	String cwdString = m_working.getStringValue();
        File outFile = new File (cwdString, "conformers.sdf");

    	if (!(outFile.exists() && outFile.isFile())) {
    		warningReport();
    		throw new IllegalStateException("THINK didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
        BufferedDataContainer cont = exec.createDataContainer(SPEC);
        
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        
        HashSet<String> titles = new HashSet<String>();

        String line;
        StringBuilder buf = new StringBuilder(4096);
        boolean titleRead = false;
        String title = "";
        int count = 1;
        exec.setMessage("Reading molecule #" + count);
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();

            if (!titleRead) {
                title = line;
                if (title.length() == 0) {
                    title = "Mol " + count;
                }
                if (titles.contains(title)) {
                    title = title + "-" + count;
                }
                titles.add(title);
                titleRead = true;
            }
            buf.append(line).append('\n');

            if (line.startsWith("$$$$")) {
                SdfCell sdf = new SdfCell(buf.toString());
                RowKey key = new RowKey(title);

                DefaultRow row = new DefaultRow(key, sdf);
                cont.addRowToTable(row);
                buf.delete(0, buf.length());
                titleRead = false;
                count++;
                exec.setMessage("Reading molecule #" + count);
            }
        }
        cont.close();
        in.close();

        return cont.getTable();
 
  
    }

    /**
     * @see org.knime.core.node.NodeModel#reset()
     */
    @Override
    protected void reset() {
        // TODO Code executed on reset.
        // Models build during execute are cleared here.
        // Also data handled in load/saveInternals will be erased here.
    }

     /**
     * @see org.knime.core.node.NodeModel
     *      #saveSettingsTo(org.knime.core.node.NodeSettings)
     */
    @Override
    protected void saveSettingsTo(final NodeSettingsWO settings) {

        // TODO save user settings to the config object.
		
		m_smiles.saveSettingsTo(settings);
		m_row1.saveSettingsTo(settings);
		m_working.saveSettingsTo(settings);
		m_single.saveSettingsTo(settings);
		m_conjugated.saveSettingsTo(settings);
		m_crowded.saveSettingsTo(settings);
		m_alpha.saveSettingsTo(settings);
		m_amide.saveSettingsTo(settings);
		m_ring.saveSettingsTo(settings);
		m_generation.saveSettingsTo(settings);
		m_samples.saveSettingsTo(settings);
		m_cpkratio.saveSettingsTo(settings);
		m_timelimit.saveSettingsTo(settings);
		m_syslimit.saveSettingsTo(settings);
		m_bondlimit.saveSettingsTo(settings);

    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #loadValidatedSettingsFrom(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void loadValidatedSettingsFrom(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO load (valid) settings from the config object.
        // It can be safely assumed that the settings are valided by the 
        // method below.
        
        m_smiles.loadSettingsFrom(settings);
        m_row1.loadSettingsFrom(settings);
        m_working.loadSettingsFrom(settings);
        m_single.loadSettingsFrom(settings);
        m_conjugated.loadSettingsFrom(settings);
        m_crowded.loadSettingsFrom(settings);
        m_alpha.loadSettingsFrom(settings);
        m_amide.loadSettingsFrom(settings);
        m_ring.loadSettingsFrom(settings);
        m_generation.loadSettingsFrom(settings);
        m_samples.loadSettingsFrom(settings);
        m_cpkratio.loadSettingsFrom(settings);
        m_timelimit.loadSettingsFrom(settings);
        m_syslimit.loadSettingsFrom(settings);
        m_bondlimit.loadSettingsFrom(settings);

    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #validateSettings(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void validateSettings(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO check if the settings could be applied to our model
        // e.g. if the count is in a certain range (which is ensured by the
        // SettingsModel).
        // Do not actually set any values of any member variables.

         m_smiles.validateSettings(settings);
        m_row1.validateSettings(settings);
        m_working.validateSettings(settings);
        m_single.validateSettings(settings);
        m_conjugated.validateSettings(settings);
        m_crowded.validateSettings(settings);
        m_alpha.validateSettings(settings);
        m_amide.validateSettings(settings);
        m_ring.validateSettings(settings);
        m_generation.validateSettings(settings);
        m_samples.validateSettings(settings);
        m_cpkratio.validateSettings(settings);
        m_timelimit.validateSettings(settings);
        m_syslimit.validateSettings(settings);
        m_bondlimit.validateSettings(settings);

    }
    
    /**
     * @see org.knime.core.node.NodeModel #loadInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void loadInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
        
		// TODO load internal data. 
		// Everything handed to output ports is loaded automatically (data
		// returned by the execute method, models loaded in loadModelContent,
		// and user settings set through loadSettingsFrom - is all taken care 
		// of). Load here only the other internals that need to be restored
		// (e.g. data used by the views).

    }
    
    /**
     * @see org.knime.core.node.NodeModel #saveInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void saveInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
       
       	// TODO save internal models. 
		// Everything written to output ports is saved automatically (data
		// returned by the execute method, models saved in the saveModelContent,
		// and user settings saved through saveSettingsTo - is all taken care 
		// of). Save here only the other internals that need to be preserved
		// (e.g. data used by the views).

    }

}
