package com.treweren.conformers;

import java.awt.FlowLayout;

import org.knime.core.node.defaultnodesettings.DefaultNodeSettingsPane;
import org.knime.core.node.defaultnodesettings.DialogComponentBoolean;
import org.knime.core.node.defaultnodesettings.DialogComponentNumber;
import org.knime.core.node.defaultnodesettings.DialogComponentString;
import org.knime.core.node.defaultnodesettings.DialogComponentStringSelection;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelDoubleBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelString;


/**
 * <code>NodeDialog</code> for the "Conformers" Node.
 * Generate 3D conformers for a molecule
 *
 * This node dialog derives from {@link DefaultNodeSettingsPane} which allows
 * creation of a simple dialog with standard components. If you need a more 
 * complex dialog please derive directly from 
 * {@link org.knime.core.node.NodeDialogPane}.
 * 
 * @author Treweren Consultants
 */
public class ConformersNodeDialog extends DefaultNodeSettingsPane {

    /**
     * New pane for configuring Conformers node dialog.
     * This is just a suggestion to demonstrate possible default dialog
     * components.
     */
    protected ConformersNodeDialog() {
        super();
        DialogComponentStringSelection c_generation = new DialogComponentStringSelection(
                new SettingsModelString(
                        ConformersNodeModel.CFGKEY_GENERATION,
                        ConformersNodeModel.DEFAULT_GENERATION),
                        "Conformer Mode", "Systematic", "Sample", "Random");
        addDialogComponent(c_generation);
        c_generation.setToolTipText("For the most flexible molecules sample or random modes take less time");
        c_generation.getComponentPanel().setLayout( new FlowLayout(2));

        DialogComponentNumber c_samples = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        ConformersNodeModel.CFGKEY_SAMPLES,
                        ConformersNodeModel.DEFAULT_SAMPLES,
                        1, Integer.MAX_VALUE),
                        "Number of sample/random conformations", /*step*/ 100, /*componentwidth*/ 4);
        addDialogComponent(c_samples);
        c_samples.setToolTipText("The number of conformations should be sufficiently large to be representative");
        c_samples.getComponentPanel().setLayout( new FlowLayout(2));
        
        DialogComponentNumber c_cpkratio = new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                        ConformersNodeModel.CFGKEY_CPKRATIO,
                        ConformersNodeModel.DEFAULT_CPKRATIO, 0., 5. ),
                        "Contact to VdW radius ratio", .1, 3 );
        
        addDialogComponent(c_cpkratio);
        c_cpkratio.setToolTipText("Conformers with separation distances between distances less than the sum of the contact radii are skipped");
        c_cpkratio.getComponentPanel().setLayout( new FlowLayout(2));
        
        DialogComponentNumber c_timelimit = new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                        ConformersNodeModel.CFGKEY_TIMELIMIT,
                        ConformersNodeModel.DEFAULT_TIMELIMIT, 0., 1440. ),
                        "Molecule time limit (m)", 15., 4 );
        addDialogComponent(c_timelimit );
        c_timelimit.setToolTipText("In order to progress other molecules, the search for a molecule which takes longer than the time limit is abandoned");
        c_timelimit.getComponentPanel().setLayout( new FlowLayout(2));

        createNewGroup("Systematic limits");
        DialogComponentNumber c_syslimit = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        ConformersNodeModel.CFGKEY_SYSLIMIT,
                        ConformersNodeModel.DEFAULT_SYSLIMIT,
                        0, Integer.MAX_VALUE),
                        "Number of conformations", /*step*/ 100, /*componentwidth*/ 4);
        addDialogComponent(c_syslimit);
        c_syslimit.setToolTipText("If the estimated number of conformers to be generated exceeds the systematic limit, sampling is used instead");
        c_syslimit.getComponentPanel().setLayout( new FlowLayout(2));
        
        DialogComponentNumber c_bondlimit = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        ConformersNodeModel.CFGKEY_BONDLIMIT,
                        ConformersNodeModel.DEFAULT_BONDLIMIT,
                        0, Integer.MAX_VALUE),
                        "Number of rotating bonds", /*step*/ 1, /*componentwidth*/ 2);
        addDialogComponent(c_bondlimit);
        c_bondlimit.setToolTipText("If the estimated number of rotatable bonds exceeds the systematic limit, sampling is used instead");
        c_bondlimit.getComponentPanel().setLayout( new FlowLayout(2));

        closeCurrentGroup();
       
         
        createNewTab("Bond rotations");
        
        DialogComponentNumber c_single = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        ConformersNodeModel.CFGKEY_SINGLE,
                        ConformersNodeModel.DEFAULT_SINGLE,
                        0, Integer.MAX_VALUE),
                        "Single bond increments", /*step*/ 1, /*componentwidth*/ 2);
        addDialogComponent(c_single);
        c_single.setToolTipText("Single bonds are incremented by 360/n degrees where n is the number of steps (0=no rotation)");
        c_single.getComponentPanel().setLayout( new FlowLayout(2));
        
        DialogComponentNumber c_conjugated = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        ConformersNodeModel.CFGKEY_CONJUGATED,
                        ConformersNodeModel.DEFAULT_CONJUGATED,
                        0, Integer.MAX_VALUE),
                        "Conjugated bond increments", /*step*/ 1, /*componentwidth*/ 2);
        addDialogComponent(c_conjugated);
        c_conjugated.setToolTipText("Conjugated (csp2-csp2) bonds are incremented by 360/n degrees where n is the number of steps");
        c_conjugated.getComponentPanel().setLayout( new FlowLayout(2));

        DialogComponentNumber c_crowded = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        ConformersNodeModel.CFGKEY_CROWDED,
                        ConformersNodeModel.DEFAULT_CROWDED,
                        0, Integer.MAX_VALUE),
                        "Crowded bond increments", /*step*/ 1, /*componentwidth*/ 2);
        addDialogComponent(c_crowded);
        c_crowded.setToolTipText("Crowded (csp2-csp2) bonds are incremented by 360/n degrees where n is the number of steps");
        c_crowded.getComponentPanel().setLayout( new FlowLayout(2));
 
        DialogComponentNumber c_alpha = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                    ConformersNodeModel.CFGKEY_ALPHA,
                    ConformersNodeModel.DEFAULT_ALPHA,
                    0, Integer.MAX_VALUE),
                    "Alpha bond increments", /*step*/ 1, /*componentwidth*/ 2);
        addDialogComponent(c_alpha);
        c_alpha.setToolTipText("Alpha (csp2-csp3) bonds are incremented by 360/n degrees where n is the number of steps");
        c_alpha.getComponentPanel().setLayout( new FlowLayout(2));

        DialogComponentNumber c_amide = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        ConformersNodeModel.CFGKEY_AMIDE,
                        ConformersNodeModel.DEFAULT_AMIDE,
                        0, Integer.MAX_VALUE),
                        "Amide bond increments", /*step*/ 1, /*componentwidth*/ 2);
        
        addDialogComponent(c_amide);
        c_amide.setToolTipText("Amide bonds are incremented by 360/n degrees where n is the number of steps");
        c_amide.getComponentPanel().setLayout( new FlowLayout(2));

        DialogComponentNumber c_ring = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        ConformersNodeModel.CFGKEY_RING,
                        ConformersNodeModel.DEFAULT_RING,
                        0, Integer.MAX_VALUE),
                        "Ring bond increments", /*step*/ 1, /*componentwidth*/ 2); 
        addDialogComponent(c_ring);
        c_ring.setToolTipText("Ring bonds are incremented by 360/n degrees where n is the number of steps");
        c_ring.getComponentPanel().setLayout( new FlowLayout(2));
       
        createNewTab("THINK setup");         

        DialogComponentBoolean c_column = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        ConformersNodeModel.CFGKEY_SMILES,
                        ConformersNodeModel.DEFAULT_SMILES),
                        "Use SMILES in preference to SD column" );
        addDialogComponent(c_column);
        c_column.getComponentPanel().setLayout( new FlowLayout(0));
        c_column.setToolTipText("The molecule data is transferred to THINK using SD or SMILES format depending on this setting");
        
        DialogComponentBoolean c_row = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        ConformersNodeModel.CFGKEY_ROW1,
                        ConformersNodeModel.DEFAULT_ROW1),
                        "Use first row/molecule (not all)" );
        
        addDialogComponent(c_row);
        c_row.getComponentPanel().setLayout( new FlowLayout(0));
        c_row.setToolTipText("The data transferred may be limited to the first row using this control");
  
        DialogComponentString c_working = new DialogComponentString(
                new SettingsModelString (
                        ConformersNodeModel.CFGKEY_WORKING,
                        ConformersNodeModel.DEFAULT_WORKING),
                        "Working folder" );
        addDialogComponent(c_working);
        c_working.getComponentPanel().setLayout( new FlowLayout(2));
        c_working.setToolTipText("The temporary files used to transfer the data to and from THINK are placed in this folder");

                     
    }
}
