package com.treweren.docking;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;

import org.knime.chem.types.SdfCell;
import org.knime.chem.types.SdfValue;
import org.knime.chem.types.SmilesCell;
import org.knime.chem.types.SmilesValue;
import org.knime.core.data.DataCell;
import org.knime.core.data.DataColumnSpec;
import org.knime.core.data.DataColumnSpecCreator;
import org.knime.core.data.DataRow;
import org.knime.core.data.DataTableSpec;
import org.knime.core.data.DataType;
import org.knime.core.data.RowKey;
import org.knime.core.data.def.DefaultRow;
import org.knime.core.data.def.DoubleCell;
import org.knime.core.data.def.IntCell;
import org.knime.core.data.def.StringCell;
import org.knime.core.node.BufferedDataContainer;
import org.knime.core.node.BufferedDataTable;
import org.knime.core.node.CanceledExecutionException;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelDoubleBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelString;
import org.knime.core.node.port.PortType;
import org.knime.core.node.ExecutionContext;
import org.knime.core.node.ExecutionMonitor;
import org.knime.core.node.InvalidSettingsException;
import org.knime.core.node.NodeLogger;
import org.knime.core.node.NodeModel;
import org.knime.core.node.NodeSettingsRO;
import org.knime.core.node.NodeSettingsWO;
import org.knime.core.util.MutableBoolean;



/**
 * This is the model implementation of Docking.
 * Docks molecules into binding site
 *
 * @author Treweren Consultants
 */
public class DockingNodeModel extends NodeModel {
    
    // the logger instance
    private static final NodeLogger logger = NodeLogger
            .getLogger(DockingNodeModel.class);
    /**
     * time (millisec) between two calls to checkCancel.
     */
    private static int n_Column = 0;
    private static int maxProcs=1;
    private static String m_sdfColumn = null;
    private static String m_smilesColumn = null;
    private static String m_stringColumn = null;
    private static final int CANCEL_CHECK_INTERVAL = 1000;
      
    /** the settings key which is used to retrieve and 
        store the settings (from the dialog or from a settings file)    
       (package visibility to be usable from the dialog). */
	static final String CFGKEY_SEARCH = "Search";
	static final String CFGKEY_TOLERANCE = "Tolerance";
	static final String CFGKEY_FILTER = "Filter";
	static final String CFGKEY_PARENT = "Parent";
	static final String CFGKEY_SPEED = "Speed";
	static final String CFGKEY_CENTRES = "Interactions";
	static final String CFGKEY_RADIUS = "Radius";
	static final String CFGKEY_REUSE = "Reuse";
	static final String CFGKEY_SIDECHAINS = "Sidechains";
	static final String CFGKEY_CONFORMERS = "Conformers";
	static final String CFGKEY_REFINE = "Refine";
	static final String CFGKEY_SMILES = "Smiles";
	static final String CFGKEY_ROW = "Row";
	static final String CFGKEY_RESTART = "Restart";
	static final String CFGKEY_PROCESS = "Process";
	static final String CFGKEY_WORKING = "Working";
	static final String CFGKEY_SINGLE = "Single";
	static final String CFGKEY_CONJUGATED = "Conjugated";
	static final String CFGKEY_CROWDED = "Crowded";
	static final String CFGKEY_ALPHA = "Alpha";
	static final String CFGKEY_AMIDE = "Amide";
	static final String CFGKEY_RING = "Ring";
	static final String CFGKEY_GENERATION = "Generation";
	static final String CFGKEY_SAMPLES = "Samples";
	static final String CFGKEY_CPKRATIO = "CPKratio";
	static final String CFGKEY_FRACTION = "Fraction";
	static final String CFGKEY_TIMELIMIT = "TimeLimit";
	static final String CFGKEY_SYSLIMIT = "SysLimit";
	static final String CFGKEY_BONDLIMIT = "BondLimit";

	/** initial default count value. */
	static final String DEFAULT_SEARCH = "Use input";
	static final boolean DEFAULT_FILTER = false;
	static final boolean DEFAULT_PARENT = false;
	static final double DEFAULT_TOLERANCE = 0.5;
	static final double DEFAULT_SPEED = 10.;
	static final int DEFAULT_CENTRES = 4;
	static final boolean DEFAULT_RADIUS = false;
	static final boolean DEFAULT_REUSE = true;
	static final boolean DEFAULT_SIDECHAINS = false;
	static final String DEFAULT_REFINE = "None";
	static final boolean DEFAULT_CONFORMERS = false;
	static final boolean DEFAULT_SMILES = false;
	static final int DEFAULT_ROW = 1;
	static final boolean DEFAULT_RESTART = true;
	static final int DEFAULT_PROCESS = 1;
	static final String DEFAULT_WORKING = System.getenv("THINK_WORKING");
	static final int DEFAULT_SINGLE = 3;
	static final int DEFAULT_CONJUGATED = 2;
	static final int DEFAULT_CROWDED = 2;
	static final int DEFAULT_ALPHA = 6;
	static final int DEFAULT_AMIDE = 0;
	static final int DEFAULT_RING = 0;
	static final String DEFAULT_GENERATION = "Systematic";
	static final int DEFAULT_SAMPLES = 1000;
	static final double DEFAULT_CPKRATIO = 0.6;
	static final double DEFAULT_TIMELIMIT = 60.;
	static final double DEFAULT_FRACTION = 0.25;
	static final int DEFAULT_SYSLIMIT = 0;
	static final int DEFAULT_BONDLIMIT = 10;

	// example value: the models count variable filled from the dialog 
	// and used in the models execution method. The default components of the
	// dialog work with "SettingsModels".
	private final SettingsModelString m_search =
		new SettingsModelString(DockingNodeModel.CFGKEY_SEARCH,
                    DockingNodeModel.DEFAULT_SEARCH);
	
	private final SettingsModelBoolean m_filter =
		new SettingsModelBoolean(DockingNodeModel.CFGKEY_FILTER,
                    DockingNodeModel.DEFAULT_FILTER );
	
	private final SettingsModelBoolean m_parent =
		new SettingsModelBoolean(DockingNodeModel.CFGKEY_PARENT,
                    DockingNodeModel.DEFAULT_PARENT );

	private final SettingsModelDoubleBounded m_tolerance =
		new SettingsModelDoubleBounded(DockingNodeModel.CFGKEY_TOLERANCE,
                    DockingNodeModel.DEFAULT_TOLERANCE,
                    0., 5.0);
	private final SettingsModelDoubleBounded m_speed =
		new SettingsModelDoubleBounded(DockingNodeModel.CFGKEY_SPEED,
                    DockingNodeModel.DEFAULT_SPEED,
                    1., 100.0);
		
	private final SettingsModelBoolean m_radius =
		new SettingsModelBoolean(DockingNodeModel.CFGKEY_RADIUS,
                    DockingNodeModel.DEFAULT_RADIUS );
	
	private final SettingsModelBoolean m_reuse =
		new SettingsModelBoolean(DockingNodeModel.CFGKEY_REUSE,
                    DockingNodeModel.DEFAULT_REUSE );

	private final SettingsModelBoolean m_smiles =
		new SettingsModelBoolean(DockingNodeModel.CFGKEY_SMILES,
                    DockingNodeModel.DEFAULT_SMILES );

	private final SettingsModelIntegerBounded m_centres =
		new SettingsModelIntegerBounded(DockingNodeModel.CFGKEY_CENTRES,
                    DockingNodeModel.DEFAULT_CENTRES, 2, 4 );
/*
	private final SettingsModelBoolean m_sidechains =
		new SettingsModelBoolean(DockingNodeModel.CFGKEY_SIDECHAINS,
                    DockingNodeModel.DEFAULT_SIDECHAINS );
	*/
	private final SettingsModelString m_refine =
		new SettingsModelString(DockingNodeModel.CFGKEY_REFINE,
                    DockingNodeModel.DEFAULT_REFINE);

	private final SettingsModelBoolean m_conformers =
		new SettingsModelBoolean(DockingNodeModel.CFGKEY_CONFORMERS,
                    DockingNodeModel.DEFAULT_CONFORMERS );

	private final SettingsModelIntegerBounded m_row =
			new SettingsModelIntegerBounded(DockingNodeModel.CFGKEY_ROW,
	                    DockingNodeModel.DEFAULT_ROW, 0, Integer.MAX_VALUE );
		
	private final SettingsModelBoolean m_restart =
			new SettingsModelBoolean(DockingNodeModel.CFGKEY_RESTART,
	                    DockingNodeModel.DEFAULT_RESTART );
		
	private final SettingsModelIntegerBounded m_process =
			new SettingsModelIntegerBounded(DockingNodeModel.CFGKEY_PROCESS,
	                    DockingNodeModel.DEFAULT_PROCESS, 1 ,36 );
		
	private final SettingsModelString m_working =
		new SettingsModelString(DockingNodeModel.CFGKEY_WORKING,
                    DockingNodeModel.DEFAULT_WORKING);

	private final SettingsModelIntegerBounded m_single =
		new SettingsModelIntegerBounded(DockingNodeModel.CFGKEY_SINGLE,
                    DockingNodeModel.DEFAULT_SINGLE, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_conjugated =
		new SettingsModelIntegerBounded(DockingNodeModel.CFGKEY_CONJUGATED,
                    DockingNodeModel.DEFAULT_CONJUGATED, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_crowded =
		new SettingsModelIntegerBounded(DockingNodeModel.CFGKEY_CROWDED,
                    DockingNodeModel.DEFAULT_CROWDED, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_alpha =
		new SettingsModelIntegerBounded(DockingNodeModel.CFGKEY_ALPHA,
                    DockingNodeModel.DEFAULT_ALPHA, 0, Integer.MAX_VALUE);


	private final SettingsModelIntegerBounded m_amide =
		new SettingsModelIntegerBounded(DockingNodeModel.CFGKEY_AMIDE,
                    DockingNodeModel.DEFAULT_AMIDE, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_ring =
		new SettingsModelIntegerBounded(DockingNodeModel.CFGKEY_RING,
                    DockingNodeModel.DEFAULT_RING, 0, Integer.MAX_VALUE);
	
	private final SettingsModelString m_generation =
		new SettingsModelString(DockingNodeModel.CFGKEY_GENERATION,
                    DockingNodeModel.DEFAULT_GENERATION);

	private final SettingsModelIntegerBounded m_samples =
		new SettingsModelIntegerBounded(DockingNodeModel.CFGKEY_SAMPLES,
                    DockingNodeModel.DEFAULT_SAMPLES, 0, Integer.MAX_VALUE);

	private final SettingsModelDoubleBounded m_cpkratio =
		new SettingsModelDoubleBounded(DockingNodeModel.CFGKEY_CPKRATIO,
                    DockingNodeModel.DEFAULT_CPKRATIO,
                    0., 5.0);
	
	private final SettingsModelDoubleBounded m_timelimit =
		new SettingsModelDoubleBounded(DockingNodeModel.CFGKEY_TIMELIMIT,
                    DockingNodeModel.DEFAULT_TIMELIMIT,
                    0., 1440.);
	
	private final SettingsModelDoubleBounded m_fraction =
		new SettingsModelDoubleBounded(DockingNodeModel.CFGKEY_FRACTION,
                    DockingNodeModel.DEFAULT_FRACTION,
                    0., 1.);

	private final SettingsModelIntegerBounded m_syslimit =
		new SettingsModelIntegerBounded(DockingNodeModel.CFGKEY_SYSLIMIT,
                    DockingNodeModel.DEFAULT_SYSLIMIT, 0, Integer.MAX_VALUE);

	private final SettingsModelIntegerBounded m_bondlimit =
		new SettingsModelIntegerBounded(DockingNodeModel.CFGKEY_BONDLIMIT,
                    DockingNodeModel.DEFAULT_BONDLIMIT, 0, Integer.MAX_VALUE);


    /**
     * Constructor for the node model.
     */
    public static final PortType OPTIONAL_PORT_TYPE = new PortType(BufferedDataTable.class, true);
	
    protected DockingNodeModel() {
    
        // Two incoming ports (second optional) and three outgoing port
   	    super(createOPOs(2, 2), createOPOs(3));
    }
   	private static PortType[] createOPOs(final int nrDataPorts, final int... optionalPortsIds)
   	{
   	    PortType[] portTypes = new PortType[nrDataPorts];
   	    Arrays.fill(portTypes, BufferedDataTable.TYPE);        
    	    if (optionalPortsIds.length > 0) {
   	        for (int portId : optionalPortsIds) {
   	            if ((portId - 1) < nrDataPorts) {
   	                portTypes[portId - 1] = OPTIONAL_PORT_TYPE;
   	            }
   	        }
   	    }
   	    return portTypes;
   	} 

    /**
     * @see org.knime.core.node.NodeModel #execute(BufferedDataTable[],
     *      ExecutionContext)
     */
    @Override
    protected BufferedDataTable[] execute(final BufferedDataTable[] inData,
            final ExecutionContext exec) throws Exception {

        notifyViews(null);

        final String cwdString = m_working.getStringValue();

        // get out query file name
        String fileString = "sites.lis"; 
        BufferedDataTable in = inData[0];
        final int count = in.getRowCount(); 
        if ( m_row.getIntValue()> 0 && count == m_row.getIntValue() ) {
            setWarningMessage("Using only " + count + " row/query");
         }
        int colIndex=2;
         	
        int i = 0; 
        int missingCount = 0;
        File inFile = new File(cwdString, fileString); 
        BufferedWriter outWriter = new BufferedWriter(new FileWriter(inFile));

        int rowCount=0;
        for (DataRow r : in) {
            exec.checkCanceled();
             DataCell c = r.getCell(colIndex);
             rowCount++;
            if (c.isMissing()) {
                missingCount++;
            } 
            else if ( rowCount == m_row.getIntValue() || m_row.getIntValue() == 0 ){
            	String toString = c.toString();
            	outWriter.write(toString);        	
            	outWriter.newLine();
            }
            i++;
            if ( m_row.getIntValue() == rowCount ) break;
       }
        outWriter.close();
        if (missingCount > 0) {
            setWarningMessage("Skipped " + missingCount 
                    + " row(s) because of missing values");
        }
        // Molecule input or file
        String moleString = null; 
        BufferedDataTable inmol = inData[1];
      	if ( inmol != null )
      	{
      		DataTableSpec MolSpec = inmol.getDataTableSpec();
        	n_Column = columnSelect(MolSpec);
        
        // write out molecule file
        	if ( m_sdfColumn == null ) {
        		moleString = "molecules.smiles"; 
        	}
        	else {
        		moleString = "molecules.sdf"; 
        	}
        	File moleFile = new File(cwdString, moleString); 
        	BufferedWriter moleWriter = new BufferedWriter(new FileWriter(moleFile));
        	int moleIndex;
        	if ( m_sdfColumn != null ) {
        		moleIndex = inmol.getDataTableSpec().findColumnIndex(m_sdfColumn);
        	}
        	else if ( m_smilesColumn != null ) {
        		moleIndex = inmol.getDataTableSpec().findColumnIndex(m_smilesColumn);        
        	}
        	else if ( m_stringColumn != null ) {
        		moleIndex = inmol.getDataTableSpec().findColumnIndex(m_stringColumn);        
        	}
        	else {
        		throw new IllegalStateException("No suitable reagent column - run configure (again)");
        	}       
        	i = 0; 
        	missingCount = 0;
        	for (DataRow r : inmol) {
        		exec.checkCanceled();
        		DataCell c = r.getCell(moleIndex);
        		if (c.isMissing()) {
        			missingCount++;
        		} 
        		else if ( m_sdfColumn != null ) {
        			SdfValue v = (SdfValue)c;
        			String toString = v.toString();
        			moleWriter.write(toString);
        			if (!toString.trim().endsWith("$$$$")) {
        				moleWriter.newLine();
        				moleWriter.append("$$$$");
        				moleWriter.newLine();
        			}
        		}
        		else if ( m_smilesColumn != null) {
        			SmilesValue v = (SmilesValue)c;
        			String toString = v.toString() + " " + r.getKey();
        			moleWriter.write(toString);        	
        			moleWriter.newLine();
        		}
        		else {
        			String toString = c.toString() + " " + r.getKey();
        			moleWriter.write(toString);        	
        			moleWriter.newLine();
        		}
        		i++;
        	}
        	moleWriter.close();
        	if (missingCount > 0) {
        		setWarningMessage("Skipped " + missingCount 
        				+ " row(s) because of missing values");
        	}
            maxProcs = inmol.getRowCount() - missingCount;
            if ( maxProcs < 1 ) 
            	maxProcs = 1;
            else if ( maxProcs > m_process.getIntValue() )
            	maxProcs = m_process.getIntValue();
      	}
      	else {
            moleString = m_search.getStringValue();
        	maxProcs = m_process.getIntValue();
      	}

        
        int n_threads = m_process.getIntValue();
        final int n_process = maxProcs;

        
        for ( i=0; i<n_process; i++)
        {
            File searchRestart = new File(cwdString, "search" + i + ".dat");
            if (!searchRestart.exists() || m_restart.getBooleanValue()){
              searchRestart.delete();
              File outputHits = new File(cwdString, "docked" + i + ".sdf"); 
              outputHits.delete();
        	  File outputReport = new File(cwdString, "output" + i + ".log"); 
        	  outputReport.delete();
        	  File outputFile = new File(cwdString, "failures" + i + ".csv"); 
        	  outputFile.delete();
        	  File summaryFile = new File(cwdString, "summary" + i + ".csv"); 
        	  summaryFile.delete();
              File progressFile = new File(cwdString, "progress"+i+".dat"); 
              progressFile.delete();
            }
            else
            	setWarningMessage("Continue THINK("+i+") from check point file");
        }

        
        // execute THINK
        File cwdFile = new File (cwdString);
/*      File inFile = new File (cwdString, fileString);
      	if (!(inFile.exists() && inFile.isFile())) {
            throw new IllegalStateException("THINK input file not valid "
                    + " at the specified location ('"
                    + inFile.getAbsolutePath() + "')");
        }
 */       
        int[] exitVal = new int[36];

       try {
            exec.setProgress(0., "Starting THINK");
            Runtime rt = Runtime.getRuntime();
            double d_tolerance = m_tolerance.getDoubleValue();
            int n_centres = m_centres.getIntValue();
            String s_refine = "None";
            if (m_refine.getStringValue().equals("Ligand") ) {
            	s_refine = "Ligand";
            }
            else if (m_refine.getStringValue().equals("Side-chain and Ligand") ) {
            	s_refine = "Side-chains";
            }
            int n_single = m_single.getIntValue();
            int n_conjugated = m_conjugated.getIntValue();
            int n_crowded = m_crowded.getIntValue();
            int n_alpha = m_alpha.getIntValue();
            int n_amide = m_amide.getIntValue();
            int n_ring = m_ring.getIntValue();
            String s_generation = m_generation.getStringValue();
            int n_samples = m_samples.getIntValue();
            double d_cpkratio = m_cpkratio.getDoubleValue();
            double d_timelimit = m_timelimit.getDoubleValue();
            int n_syslimit = m_syslimit.getIntValue();
            int n_bondlimit = m_bondlimit.getIntValue();

            String s_reuse = "DIFFERENT";
            if ( m_reuse.getBooleanValue() ) {
            	s_reuse = "ANY";
             }
            String s_radius = "NOTOLERANCE";
            if ( m_radius.getBooleanValue() ) {
              	s_radius = "TOLERANCE";
            }
/*            String s_sidechains = "NONE";
            if ( m_sidechains.getBooleanValue() ) {
            	s_sidechains = "SIDE";
            }
            */
            String s_conformers = "NONE";
            if ( m_conformers.getBooleanValue() ) {
            	s_conformers = "ALL";
            }
            String s_filter = "NONE";
            if ( m_filter.getBooleanValue() ) {
            	s_filter = "FILTER";
            }
            String s_parent = "NONE";
            if ( m_parent.getBooleanValue() ) {
            	s_parent = "PARENT";
            }

            double d_speed = m_speed.getDoubleValue();
            double d_fraction = m_fraction.getDoubleValue();
            final Process[] procs = new Process[36];
            for (i=0; i<n_process;i++)
            {
          
            // prepare the command string
            String cmdString = System.getenv("THINK_EXEC") + "think THINK_EXEC:docking.log output"+ i + ".log " + fileString + " \"" + moleString + "\" " + d_tolerance + " " + n_centres + " " + s_refine + " " + s_conformers + " " + d_speed + " " + s_reuse + " " + s_radius 
            + " " + n_single + " " + n_conjugated + " " + n_crowded + " " + n_alpha  + " " + n_amide + " " + n_ring
            + " " + s_generation + " " + n_samples + " " + d_cpkratio + " " + d_timelimit + " " + n_syslimit + " " + n_bondlimit
            + " " + s_filter + " " + s_parent + " " + d_fraction + " " + i + " " + n_process + " " + n_threads;
                     // Go!
            logger.info("THINK command line: '" + cmdString + "'");
            procs[i] = rt.exec(cmdString, null, cwdFile);
            }

            final MutableBoolean procDone = new MutableBoolean(false);
            new Thread(new Runnable() {
                public void run() {
                    synchronized (procDone) {
                        while (!procDone.booleanValue()) {
                            try {
                                exec.checkCanceled();
                            } catch (CanceledExecutionException cee) {
                                // blow away the running external process
                              	int j;
                            	for (j=0;j<n_process;j++) {
                            		procs[j].destroy();
                            	}                                

                                return;
                            }
                            try {
                               	procDone.wait(CANCEL_CHECK_INTERVAL);
                                // Read progress file and set progress value       
                               	double d= 0.;
                       			String line = ""; 
                       			String newline="";
                       			String prefix="";
                       			String progFile="progress.dat";
                                int j;
                               	int l=0;
                               	for (j=0;j<n_process;j++) {
                                	progFile = "progress" + j + ".dat";
                                	prefix = "\nTHINK(" + j + "): ";
                                  	double done=0;
                                  	File progressReport = new File(cwdString, progFile);
                                  	if ( progressReport.exists() && progressReport.isFile() ) {
                                  		if ( progressReport.length() > 0 ) {
                                  			BufferedReader in = new BufferedReader(new FileReader(progressReport));
                                  			if ( (newline = in.readLine()) != null) {
                                  				l++;
                                  				done = Double.valueOf(newline).doubleValue();
                                  				d = d + done;
                                  			}
                                  			if ( done == 1.0 ) {
                                  				line += prefix + "Tabulating output";
                                  			}
                                  			else if ( (newline = in.readLine()) == null) {
                                  				line += prefix + "Running";
                                  			}
                                  			else {
                                  				line += prefix + newline;
                                  			}
                                  			in.close();
                                  		}
                                  	}
                                }
                               	if ( l > 0 ) {
                               		d = d / l;
                               	}
                                exec.setProgress( d, line.toString());
                            } catch (InterruptedException e) {
                                // do nothing
                            } catch ( FileNotFoundException e) {
                                // do nothing
                            } catch ( IOException e) {
                                // do nothing
                            }
                        }
                    }
                }

            }).start();
           
            // wait until the external process finishes.
            for (i=0; i< n_process;i++)
            {
            	exitVal[i] = procs[i].waitFor();
            }
          
            synchronized (procDone) {
                // this should terminate the check cancel thread
                procDone.setValue(true);
            }
            exec.checkCanceled();

            exec.setProgress("Wrapping up");
            for (i=0; i< n_process;i++)
            {
            	logger.info("THINK("+i+") terminated with exit code: " + exitVal[i]);
            }
         } catch (Throwable t) {
            logger.error("THINK failed (with exception)", t);
            warningReport();
            throw new Exception(t);
        }

       for (i=0;i<n_process;i++)
       {
       if (exitVal[i] != 0) {
       	warningReport();
           // before we return barfing, we save the output in the failing list
           throw new IllegalStateException("THINK("+i+") failed (error code "
                   + exitVal[i] + ")");
       }
       }

        exec.setProgress("Creating KNIME tables");
        BufferedDataTable outTable = readOutputFile(exec);
        BufferedDataTable sumTable = readSummaryFile(exec);
        BufferedDataTable failTable = readFailureFile(exec);

        return new BufferedDataTable[]{outTable, sumTable, failTable};

    }
    protected int columnSelect ( final DataTableSpec in ) throws  InvalidSettingsException {

        StringBuilder warningMessage = new StringBuilder();
        int n_column=0;
        int sdfColCount = 0;
        int n_sdf=0;
        int smilesColCount = 0;
        int n_smiles=0;
        int stringColCount = 0;
        int n_string=0;
        for (int i = 0; i < in.getNumColumns(); i++) {
            DataColumnSpec s = in.getColumnSpec(i);
            if (s.getType().isCompatible(SdfValue.class)) {
                if (m_sdfColumn == null) {
                    m_sdfColumn = in.getColumnSpec(i).getName();
                    n_sdf = i;
                }
                sdfColCount++;
            }
            else if ( s.getType().isCompatible(SmilesValue.class)) {
            	if (m_smilesColumn == null ) {
             		m_smilesColumn = in.getColumnSpec(i).getName();
             		n_smiles=i;
              	}
               	smilesColCount++;
            }
            else if ( in.getColumnSpec(i).getName().equalsIgnoreCase("smiles"))  {
            	if (m_stringColumn == null ) {
             		m_stringColumn = in.getColumnSpec(i).getName();
             		n_string=i;
              	}
               	stringColCount++;
            }
        }
        if ( smilesColCount == 0 & sdfColCount == 0 & stringColCount == 0) {
            throw new InvalidSettingsException(
                    "No column with Smiles or SD file compatible type");       	
        }
        else if ( ( m_smiles.getBooleanValue() | sdfColCount == 0 ) & smilesColCount > 0 ) {
        	m_sdfColumn = null;
        	m_stringColumn = null;
        	n_column = n_smiles;
        	if (smilesColCount > 1) {
                warningMessage.append("More than one Smiles compatible column in " 
                        + "input, using column \"" + m_smilesColumn + "\".");
        	}
        }
        else if ( sdfColCount > 0 ) {
           	m_smilesColumn = null;
        	m_stringColumn = null;
        	n_column = n_sdf;
        	if (sdfColCount > 1) {
                warningMessage.append("More than one SDF compatible column in " 
                        + "input, using column \"" + m_sdfColumn + "\".");
        	}
        }
        else if ( stringColCount > 0 ) {
        	m_smilesColumn = null;
        	m_sdfColumn = null;
        	n_column = n_string;
           	if (stringColCount > 1) {
                warningMessage.append("More than one Smiles string compatible column in " 
                        + "input, using column \"" + m_stringColumn + "\".");
           	}
        }
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
        return n_column;
    }
   

    private void warningReport( ) throws Exception {

        final int n_process = maxProcs;
        StringBuilder warningMessage = new StringBuilder();
        String cwdString = m_working.getStringValue();
        int i=0;
        for (i=0;i<n_process;i++)
        {
        File outputReport = new File(cwdString, "output"+i+".log"); 

        if ( outputReport.exists() && outputReport.isFile() ) {
       		if ( outputReport.length() > 0 ) {
       			BufferedReader in = new BufferedReader(new FileReader(outputReport));
       			String line;
       			while ((line = in.readLine()) != null) {
       				warningMessage.append( line + "\n");
       			}               
       			in.close();
       		}
       		else { 
                warningMessage.append( "Report from THINK(" + i + ")is blank");      	     			
       		}
        }
       	else {
            warningMessage.append( "No report from THINK("+i+")");      	
       	}
        }
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
    }
    private BufferedDataTable readOutputFile(final ExecutionContext exec) throws Exception {

    	String cwdString = m_working.getStringValue();
        String line;
        String fieldName;
        String doubleFields[] = new String[100];
        String stringFields[]= new String[100];
        int smileCount=0;
        int keyCount=0;
        int doubleCount=0;
        int stringCount=0;
        int molcount=0;
        int keyStart=0;
        int doubleStart=0;
        int stringStart=0;
        int count =0;
        int j=0;
        int n_process = maxProcs;
        boolean tableCreated = false;

        
        for (j=0;j<n_process;j++)
        {
        File outFile = new File (cwdString, "docked"+j+".sdf");

    	if (!(outFile.exists() && outFile.isFile())) {
    		warningReport();
    		throw new IllegalStateException("THINK("+j+") didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
        }
    	// Scan file for fields and data types
       
        File outSee = new File (cwdString, "docked0.sdf");
        BufferedReader looksee = new BufferedReader(new FileReader(outSee));
        while ((line = looksee.readLine()) != null) {
            if (line.startsWith("$$$$")) 
            	break;
            else if (line.startsWith("> <")) {
            	int l = line.length() - 1;
            	fieldName = line.substring(3,l);
            	line = looksee.readLine();
            	if ( fieldName.equals("SMILES")) {
            		smileCount = 1;
            	}
            	else if ( fieldName.equals("KEYS")) {
            		keyCount = 1;
            	}
            	else {
            		try {
            			double d_value = Double.valueOf(line).doubleValue();
            			doubleFields[doubleCount] = fieldName;
            			doubleCount ++;
            		}
            		catch (Throwable t){
            			stringFields[stringCount] = fieldName;
            			stringCount++;
            		}
            	}
            }	
        }	        
        looksee.close();
        
        // Create the table
        DataColumnSpec[] allColSpecs;
        allColSpecs = new DataColumnSpec[stringCount+doubleCount+smileCount+keyCount+1];
         allColSpecs[0] = 
            new DataColumnSpecCreator("Molecule",SdfCell.TYPE).createSpec();
        if ( smileCount == 1 ) {
        	count ++;
            allColSpecs[count] = 
            	new DataColumnSpecCreator("SMILES",SmilesCell.TYPE).createSpec();
        }
        if ( keyCount == 1 ) {
        	keyCount = count + 1;
        	count ++;
            allColSpecs[count] = 
            	new DataColumnSpecCreator("KEYS",StringCell.TYPE).createSpec();
        }
        if ( stringCount > 0 ) {
        	stringStart=count + 1;
        	for (int i=0; i<stringCount; i++ ) {
        		count ++;
                allColSpecs[count] = 
        		new DataColumnSpecCreator(stringFields[i],StringCell.TYPE).createSpec();
        	}
        }
        if ( doubleCount > 0 ) {
           	doubleStart=count + 1;      	
        	for (int i=0; i<doubleCount; i++ ) {
        		count ++;
                allColSpecs[count] = 
        		new DataColumnSpecCreator(doubleFields[i],DoubleCell.TYPE).createSpec();
        	}
        }
        DataTableSpec outputSpec = new DataTableSpec(allColSpecs);
        BufferedDataContainer cont = exec.createDataContainer(outputSpec);
              
        for (j=0;j<n_process;j++)
        {
            File outFile = new File (cwdString, "docked"+j+".sdf");
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        HashSet<String> titles = new HashSet<String>();
        StringBuilder buf = new StringBuilder(4096);
        boolean titleRead = false;
        String title = "";    		
        DataCell[] cells = new DataCell[count+1];
        
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();

            if (!titleRead) {
                title = line;
                if (title.length() == 0) {
                    title = "Mol " + molcount;
                }
                if (titles.contains(title)) {
                    title = title + "-" + molcount;
                }
                titles.add(title);
                titleRead = true;
            }
            buf.append(line).append('\n');

            if (line.startsWith("$$$$")) {
            	molcount ++;
                cells[0] = new SdfCell(buf.toString());
                RowKey key = new RowKey(title);

                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
                buf.delete(0, buf.length());
                titleRead = false;
            }
            else if ( line.startsWith("> <")) {
            	int l = line.length() - 1;
            	fieldName = line.substring(3,l);
            	line = in.readLine();
                buf.append(line).append('\n');
            	if ( smileCount == 1 & fieldName.equals("SMILES")) {
            		cells[1] = new SmilesCell(line);
            	}
            	else if ( keyCount == 1 & fieldName.equals("KEYS")) {
                    cells[keyStart]= new StringCell(line);
            	}
            	else {
            		for (int i=0; i<doubleCount; i++ ) {
            			if ( doubleFields[i].equals(fieldName)) {
            				try {
            				double d_value = Double.valueOf(line).doubleValue();
            				cells[doubleStart+i]= new DoubleCell(d_value);
            				} catch (Throwable t) {
            					cells[doubleStart+i] = DataType.getMissingCell();
            				}
            				break;
            			}
            		}
            		for (int i=0; i<stringCount; i++ ) {
            			if ( stringFields[i].equals(fieldName)) {
            				cells[stringStart+i]= new StringCell(line);
            				break;
            			}
            		}
            	}
            }
        }
        in.close();
        }
        cont.close();
 
        return cont.getTable();
 
  
    }

    private BufferedDataTable readOutputFile1(final ExecutionContext exec) throws Exception {

    	String cwdString = m_working.getStringValue();
        String line;
        String fieldName;
        String doubleFields[] = new String[100];
        String stringFields[]= new String[100];
        int smileCount=0;
        int keyCount=0;
        int doubleCount=0;
        int stringCount=0;
        File outFile = new File (cwdString, "docked.sdf");

    	if (!(outFile.exists() && outFile.isFile())) {
    		warningReport();
    		throw new IllegalStateException("THINK didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
    	// Scan file for fields and data types
    	
        BufferedReader looksee = new BufferedReader(new FileReader(outFile));
        while ((line = looksee.readLine()) != null) {
            if (line.startsWith("$$$$")) 
            	break;
            else if (line.startsWith("> <")) {
            	int l = line.length() - 1;
            	fieldName = line.substring(3,l);
            	line = looksee.readLine();
            	if ( fieldName.equals("SMILES")) {
            		smileCount = 1;
            	}
            	else if ( fieldName.equals("KEYS")) {
            		keyCount = 1;
            	}
            	else {
            		try {
            			double d_value = Double.valueOf(line).doubleValue();
            			doubleFields[doubleCount] = fieldName;
            			doubleCount ++;
            		}
            		catch (Throwable t){
            			stringFields[stringCount] = fieldName;
            			stringCount++;
            		}
            	}
            }	
        }	        
        looksee.close();
        
        // Create the table
        DataColumnSpec[] allColSpecs;
        allColSpecs = new DataColumnSpec[stringCount+doubleCount+smileCount+keyCount+1];
        int count =0;
        int keyStart=0;
        int doubleStart=0;
        int stringStart=0;
        allColSpecs[0] = 
            new DataColumnSpecCreator("Molecule",SdfCell.TYPE).createSpec();
        if ( smileCount == 1 ) {
        	count ++;
            allColSpecs[count] = 
            	new DataColumnSpecCreator("SMILES",SmilesCell.TYPE).createSpec();
        }
        if ( keyCount == 1 ) {
        	keyCount = count + 1;
        	count ++;
            allColSpecs[count] = 
            	new DataColumnSpecCreator("KEYS",StringCell.TYPE).createSpec();
        }
        if ( stringCount > 0 ) {
        	stringStart=count + 1;
        	for (int i=0; i<stringCount; i++ ) {
        		count ++;
                allColSpecs[count] = 
        		new DataColumnSpecCreator(stringFields[i],StringCell.TYPE).createSpec();
        	}
        }
        if ( doubleCount > 0 ) {
           	doubleStart=count + 1;      	
        	for (int i=0; i<doubleCount; i++ ) {
        		count ++;
                allColSpecs[count] = 
        		new DataColumnSpecCreator(doubleFields[i],DoubleCell.TYPE).createSpec();
        	}
        }
        DataTableSpec outputSpec = new DataTableSpec(allColSpecs);
        BufferedDataContainer cont = exec.createDataContainer(outputSpec);
                
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        
        HashSet<String> titles = new HashSet<String>();

        StringBuilder buf = new StringBuilder(4096);
        boolean titleRead = false;
        String title = "";
    		
        DataCell[] cells = new DataCell[count+1];
        int molcount=0;
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();

            if (!titleRead) {
                title = line;
                if (title.length() == 0) {
                    title = "Mol " + molcount;
                }
                if (titles.contains(title)) {
                    title = title + "-" + molcount;
                }
                titles.add(title);
                titleRead = true;
            }
            buf.append(line).append('\n');

            if (line.startsWith("$$$$")) {
            	molcount ++;
                cells[0] = new SdfCell(buf.toString());
                RowKey key = new RowKey(title);

                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
                buf.delete(0, buf.length());
                titleRead = false;
            }
            else if ( line.startsWith("> <")) {
            	int l = line.length() - 1;
            	fieldName = line.substring(3,l);
            	line = in.readLine();
                buf.append(line).append('\n');
            	if ( smileCount == 1 & fieldName.equals("SMILES")) {
            		cells[1] = new SmilesCell(line);
            	}
            	else if ( keyCount == 1 & fieldName.equals("KEYS")) {
                    cells[keyStart]= new StringCell(line);
            	}
            	else {
            		for (int i=0; i<doubleCount; i++ ) {
            			if ( doubleFields[i].equals(fieldName)) {
            				try {
            				double d_value = Double.valueOf(line).doubleValue();
            				cells[doubleStart+i]= new DoubleCell(d_value);
            				} catch (Throwable t) {
            					cells[doubleStart+i] = DataType.getMissingCell();
            				}
            				break;
            			}
            		}
            		for (int i=0; i<stringCount; i++ ) {
            			if ( stringFields[i].equals(fieldName)) {
            				cells[stringStart+i]= new StringCell(line);
            				break;
            			}
            		}
            	}
            }
        }
        cont.close();
        in.close();
 
        return cont.getTable();
 
  
    }

    private BufferedDataTable readFailureFile(final ExecutionContext exec) throws Exception {

    	String cwdString = m_working.getStringValue();
    	int n_process=maxProcs;
        HashSet<String> titles = new HashSet<String>();
        String title;
        String line;
        DataColumnSpec[] allColSpecs;
        DataTableSpec outputSpec;
        BufferedDataContainer cont =null;
        RowKey key =null;
        Boolean headerCreated=false;
        int count =0;
        int ilast=0;
        int rowcount =0;
        char chr='0';
    	int j=0;
    	for (j=0;j<n_process;j++)
    	{
        File outFile = new File (cwdString, "failures"+j+".csv");

    	if (!(outFile.exists() && outFile.isFile())) {
    		warningReport();
    		throw new IllegalStateException("THINK("+j+") didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
        boolean headerRead = false;
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();
// Extract column headers
            if ( !headerCreated ) {
                headerCreated = true;
 // Create column definitions
                allColSpecs = new DataColumnSpec[2];
                allColSpecs[0] = 
                new DataColumnSpecCreator("Reason",StringCell.TYPE).createSpec();
                allColSpecs[1] = 
                    new DataColumnSpecCreator("Query",StringCell.TYPE).createSpec();
                outputSpec = new DataTableSpec(allColSpecs);
                cont = exec.createDataContainer(outputSpec);
            	headerRead = true;
            }
            else if ( !headerRead) {
            	headerRead = true;
            }
            // Load data
            else if ( !line.equalsIgnoreCase("Molecule,Reason,Query")) {
                DataCell[] cells = new DataCell[2];
                rowcount++;
                count =0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else
                        chr = ',';
                    if ( chr == ',' ) { 
                        if ( count == 0 ) {
                            title = line.substring(0,i);
                            if (title.length() == 0) {
                                title = "Mol " + rowcount;
                            }
                            if (titles.contains(title)) {
                                title = title + "-" + rowcount;
                            }
                            titles.add(title);
                            key = new RowKey(title);                           
                        }
                        else if ( count == 1 ){
                            cells[0] = new StringCell(line.substring(ilast,i));   	
                        }
                        else if ( count == 2 ){
                            cells[1] = new StringCell(line.substring(ilast,i));   	
                        }
                        count ++;
                        ilast = i + 1;
                    }
                }
                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
             }
        }

        in.close();
    	}
        cont.close();

        return cont.getTable();
 
    }

    private BufferedDataTable readSummaryFile(final ExecutionContext exec) throws Exception {

    	String cwdString = m_working.getStringValue();
    	int n_process = maxProcs;
        DataTableSpec outputSpec;
        BufferedDataContainer cont =null;
        RowKey key =null;
        int count =0;
        int rowcount =0;
        int ncols = 4;
        char chr='0';
        boolean headerCreated=false;
        HashSet<String> titles = new HashSet<String>();
        String title;
        String line;
      	int j=0;
    	for (j=0;j<n_process;j++)
    	{
        File outFile = new File (cwdString, "summary"+j+".csv");

    	if (!(outFile.exists() && outFile.isFile())) {
    		warningReport();
    		throw new IllegalStateException("THINK("+j+") didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
        boolean headerRead = false;
        DataColumnSpec[] allColSpecs;
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();
// Extract column headers
            if ( !headerCreated ) {
                headerCreated = true;
// Pharmacophore map output included?
                if ( line.indexOf("Map_")> 0 )
                	ncols=14;
 // Create column definitions
                allColSpecs = new DataColumnSpec[ncols];
                count =0;
                int ilast=0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else {
                        chr = ',';
                    }
                    if ( chr == ',' ) {
                        
                       if  ( count == 3 ) {
                            allColSpecs[count-1] = 
                            new DataColumnSpecCreator(line.substring(ilast,i),StringCell.TYPE).createSpec();
                        }
                        else if  ( count == 4 ) {
                            allColSpecs[count-1] = 
                            new DataColumnSpecCreator(line.substring(ilast,i), DoubleCell.TYPE).createSpec();
                        }
                        else if ( count > 0 ) {
                            allColSpecs[count-1] = 
                            new DataColumnSpecCreator(line.substring(ilast,i), IntCell.TYPE).createSpec();
                        }
                        count ++;
                        ilast = i + 1;
                    }
                }
                outputSpec = new DataTableSpec(allColSpecs);
                cont = exec.createDataContainer(outputSpec);
            	headerRead = true;
            }
            else if ( !headerRead ) {
            	headerRead=true;
            }
// Load data
            else {
                DataCell[] cells = new DataCell[ncols];
                rowcount++;
                count =0;                  
                double d_value;
                int n_value;
                int ilast=0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else
                        chr = ',';
                    if ( chr == ',' ) { 
                        if ( count == 0 ) {
                            title = line.substring(0,i);
                            if (title.length() == 0) {
                                title = "Mol " + rowcount;
                            }
                            if (titles.contains(title)) {
                                title = title + "-" + rowcount;
                            }
                            titles.add(title);
                            key = new RowKey(title);                           
                        }
                        else if  ( count == 3 ) {
                            cells[count-1] = new StringCell(line.substring(ilast,i));   	
                        }
                        else if  ( count == 4 ) {
                        try {
                            d_value = Double.valueOf(line.substring(ilast,i)).doubleValue();
                            cells[count-1] = new DoubleCell(d_value);
                        } catch (Throwable t) {
                            cells[count-1] = DataType.getMissingCell();
                        }
                        }
                        else {
                            try {
                                n_value = Integer.valueOf(line.substring(ilast,i)).intValue();
                                cells[count-1] = new IntCell(n_value);
                            } catch (Throwable t) {
                                cells[count-1] = DataType.getMissingCell();
                            }
                            }
                        count ++;
                        ilast = i + 1;
                    }
                }
                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
             }
        }
        in.close();
        }
        cont.close();


        return cont.getTable();
 
    }

    private BufferedDataTable readSummaryFile1(final ExecutionContext exec) throws Exception {

    	String cwdString = m_working.getStringValue();
        File outFile = new File (cwdString, "summary.csv");

    	if (!(outFile.exists() && outFile.isFile())) {
    		warningReport();
    		throw new IllegalStateException("THINK didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
        HashSet<String> titles = new HashSet<String>();
        String title;
        String line;
        boolean headerRead = false;
        DataColumnSpec[] allColSpecs;
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        DataTableSpec outputSpec;
        BufferedDataContainer cont =null;
        RowKey key =null;
        int count =0;
        int rowcount =0;
        int hits =0;
        char chr='0';
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();
// Extract column headers
            if ( !headerRead ) {
                headerRead = true;
 // Create column definitions
                allColSpecs = new DataColumnSpec[4];
                count =0;
                int ilast=0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else {
                        chr = ',';
                    }
                    if ( chr == ',' ) {
                        
                       if  ( count == 3 ) {
                            allColSpecs[count-1] = 
                            new DataColumnSpecCreator(line.substring(ilast,i),StringCell.TYPE).createSpec();
                        }
                        else if  ( count == 4 ) {
                            allColSpecs[count-1] = 
                            new DataColumnSpecCreator(line.substring(ilast,i), DoubleCell.TYPE).createSpec();
                        }
                        else if ( count > 0 ) {
                            allColSpecs[count-1] = 
                            new DataColumnSpecCreator(line.substring(ilast,i), IntCell.TYPE).createSpec();
                        }
                        count ++;
                        ilast = i + 1;
                    }
                }
                outputSpec = new DataTableSpec(allColSpecs);
                cont = exec.createDataContainer(outputSpec);
            }
// Load data
            else {
                DataCell[] cells = new DataCell[4];
                rowcount++;
                count =0;                  
                double d_value;
                int n_value;
                int ilast=0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else
                        chr = ',';
                    if ( chr == ',' ) { 
                        if ( count == 0 ) {
                            title = line.substring(0,i);
                            if (title.length() == 0) {
                                title = "Mol " + rowcount;
                            }
                            if (titles.contains(title)) {
                                title = title + "-" + rowcount;
                            }
                            titles.add(title);
                            key = new RowKey(title);                           
                        }
                        else if  ( count == 3 ) {
                            cells[count-1] = new StringCell(line.substring(ilast,i));   	
                        }
                        else if  ( count == 4 ) {
                        try {
                            d_value = Double.valueOf(line.substring(ilast,i)).doubleValue();
                            cells[count-1] = new DoubleCell(d_value);
                        } catch (Throwable t) {
                            cells[count-1] = DataType.getMissingCell();
                        }
                        }
                        else {
                            try {
                                n_value = Integer.valueOf(line.substring(ilast,i)).intValue();
                                cells[count-1] = new IntCell(n_value);
                                if ( count == 1 ) {
                                	hits = hits + n_value;
                                }
                            } catch (Throwable t) {
                                cells[count-1] = DataType.getMissingCell();
                            }
                            }
                        count ++;
                        ilast = i + 1;
                    }
                }
                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
             }
        }
        
        cont.close();
        in.close();
        if ( hits == 0 ){
        	setWarningMessage("No hits found");
        }
        return cont.getTable();
 
    }
    private BufferedDataTable readFailureFile1(final ExecutionContext exec) throws Exception {

    	String cwdString = m_working.getStringValue();
        File outFile = new File (cwdString, "failures.csv");

    	if (!(outFile.exists() && outFile.isFile())) {
    		warningReport();
    		throw new IllegalStateException("THINK didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
        HashSet<String> titles = new HashSet<String>();
        String title;
        String line;
        boolean headerRead = false;
        DataColumnSpec[] allColSpecs;
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        DataTableSpec outputSpec;
        BufferedDataContainer cont =null;
        RowKey key =null;
        int count =0;
        int ilast=0;
        int rowcount =0;
        char chr='0';
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();
// Extract column headers
            if ( !headerRead ) {
                headerRead = true;
 // Create column definitions
                allColSpecs = new DataColumnSpec[2];
                allColSpecs[0] = 
                new DataColumnSpecCreator("Reason",StringCell.TYPE).createSpec();
                allColSpecs[1] = 
                    new DataColumnSpecCreator("Query",StringCell.TYPE).createSpec();
                outputSpec = new DataTableSpec(allColSpecs);
                cont = exec.createDataContainer(outputSpec);
            }
// Load data
            else if ( !line.equalsIgnoreCase("Molecule,Reason,Query")) {
                DataCell[] cells = new DataCell[2];
                rowcount++;
                count =0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else
                        chr = ',';
                    if ( chr == ',' ) { 
                        if ( count == 0 ) {
                            title = line.substring(0,i);
                            if (title.length() == 0) {
                                title = "Mol " + rowcount;
                            }
                            if (titles.contains(title)) {
                                title = title + "-" + rowcount;
                            }
                            titles.add(title);
                            key = new RowKey(title);                           
                        }
                        else if ( count == 1 ){
                            cells[0] = new StringCell(line.substring(ilast,i));   	
                        }
                        else if ( count == 2 ){
                            cells[1] = new StringCell(line.substring(ilast,i));   	
                        }
                        count ++;
                        ilast = i + 1;
                    }
                }
                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
             }
        }

        cont.close();
        in.close();

        return cont.getTable();
 
    }

    /**
     * @see org.knime.core.node.NodeModel#reset()
     */
    @Override
    protected void reset() {
        // TODO Code executed on reset.
        // Models build during execute are cleared here.
        // Also data handled in load/saveInternals will be erased here.
    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #configure(org.knime.core.data.DataTableSpec[])
     */
    @Override
    protected DataTableSpec[] configure(final DataTableSpec[] inSpecs)
            throws InvalidSettingsException {
        
        // Check if user settings are available, fit to the incoming
        // table structure, and the incoming types are feasible for the node
        // to execute. If the node can execute in its current state return
        // the spec of its output data table(s) (if you can, otherwise an array
        // with null elements), or throw an exception with a useful user message
        DataTableSpec inmol = inSpecs[1];
        if ( inmol != null ) {
            n_Column = columnSelect (inmol);       	
        }
        else {

        	if ( m_search.getStringValue().equals("") ) {
        		throw new InvalidSettingsException(
        		"File of molecules to be docked not specified");       	  		
        	}
          	if ( m_search.getStringValue().equals(DEFAULT_SEARCH) ) {
        		throw new InvalidSettingsException(
        		"Optional 'Molecules to be searched' Input not connected and file to be searched not specified");       	  		
        	}

        	File searchFile = new File ( m_search.getStringValue() ); 
        	if (!(searchFile.exists() && searchFile.isFile())) {
        		throw new IllegalStateException("Selected file does not exist ('"
                    + searchFile.getAbsolutePath() + "')");
        	}
        }
       	if ( System.getenv("THINK_EXEC") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_EXEC is unset - It should be set to the folder containing the THINK software.");       	  		
    	}
        if ( System.getenv("THINK_WORKING") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_WORKING is not set to the folder in which working files will be created.");       	  		
    	}

    	int colIndex=2;
        DataTableSpec in = inSpecs[0];
    	if ( !in.getColumnSpec(colIndex).getName().equals("File")) { 
            throw new InvalidSettingsException(
            "Input table does not have column named 'File' where expected");       	
    	}
        return new DataTableSpec[]{null, null, null};
    }


    /**
     * @see org.knime.core.node.NodeModel
     *      #saveSettingsTo(org.knime.core.node.NodeSettings)
     */
    @Override
    protected void saveSettingsTo(final NodeSettingsWO settings) {

        // TODO save user settings to the config object.
		
		m_search.saveSettingsTo(settings);
	    m_filter.saveSettingsTo(settings);
	    m_parent.saveSettingsTo(settings);
	    m_tolerance.saveSettingsTo(settings);
	    m_speed.saveSettingsTo(settings);
	  	m_centres.saveSettingsTo(settings);
	  	m_reuse.saveSettingsTo(settings);
	  	m_radius.saveSettingsTo(settings);
		m_refine.saveSettingsTo(settings);
		m_conformers.saveSettingsTo(settings);
		m_row.saveSettingsTo(settings);
		m_restart.saveSettingsTo(settings);
		m_process.saveSettingsTo(settings);
		m_fraction.saveSettingsTo(settings);
		m_working.saveSettingsTo(settings);
		m_smiles.saveSettingsTo(settings);
		m_single.saveSettingsTo(settings);
		m_conjugated.saveSettingsTo(settings);
		m_crowded.saveSettingsTo(settings);
		m_alpha.saveSettingsTo(settings);
		m_amide.saveSettingsTo(settings);
		m_ring.saveSettingsTo(settings);
		m_generation.saveSettingsTo(settings);
		m_samples.saveSettingsTo(settings);
		m_cpkratio.saveSettingsTo(settings);
		m_timelimit.saveSettingsTo(settings);
		m_syslimit.saveSettingsTo(settings);
		m_bondlimit.saveSettingsTo(settings);

    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #loadValidatedSettingsFrom(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void loadValidatedSettingsFrom(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO load (valid) settings from the config object.
        // It can be safely assumed that the settings are valided by the 
        // method below.
        
        m_search.loadSettingsFrom(settings);
        m_filter.loadSettingsFrom(settings);
        m_parent.loadSettingsFrom(settings);
        m_tolerance.loadSettingsFrom(settings);
        m_speed.loadSettingsFrom(settings);
        m_centres.loadSettingsFrom(settings);
        m_radius.loadSettingsFrom(settings);
        m_reuse.loadSettingsFrom(settings);
        m_conformers.loadSettingsFrom(settings);
        m_refine.loadSettingsFrom(settings);
        m_row.loadSettingsFrom(settings);
        m_restart.loadSettingsFrom(settings);
        m_process.loadSettingsFrom(settings);
        m_smiles.loadSettingsFrom(settings);
        m_fraction.loadSettingsFrom(settings);
        m_working.loadSettingsFrom(settings);
        m_single.loadSettingsFrom(settings);
        m_conjugated.loadSettingsFrom(settings);
        m_crowded.loadSettingsFrom(settings);
        m_alpha.loadSettingsFrom(settings);
        m_amide.loadSettingsFrom(settings);
        m_ring.loadSettingsFrom(settings);
        m_generation.loadSettingsFrom(settings);
        m_samples.loadSettingsFrom(settings);
        m_cpkratio.loadSettingsFrom(settings);
        m_timelimit.loadSettingsFrom(settings);
        m_syslimit.loadSettingsFrom(settings);
        m_bondlimit.loadSettingsFrom(settings);
 
    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #validateSettings(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void validateSettings(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO check if the settings could be applied to our model
        // e.g. if the count is in a certain range (which is ensured by the
        // SettingsModel).
        // Do not actually set any values of any member variables.

        m_search.validateSettings(settings);
        m_tolerance.validateSettings(settings);
        m_filter.validateSettings(settings);
        m_parent.validateSettings(settings);
        m_speed.validateSettings(settings);
        m_centres.validateSettings(settings);
        m_reuse.validateSettings(settings);
        m_radius.validateSettings(settings);
        m_refine.validateSettings(settings);
        m_conformers.validateSettings(settings);
        m_row.validateSettings(settings);
        m_restart.validateSettings(settings);
        m_process.validateSettings(settings);
        m_fraction.validateSettings(settings);
        m_working.validateSettings(settings);
        m_smiles.validateSettings(settings);
        m_single.validateSettings(settings);
        m_conjugated.validateSettings(settings);
        m_crowded.validateSettings(settings);
        m_alpha.validateSettings(settings);
        m_amide.validateSettings(settings);
        m_ring.validateSettings(settings);
        m_generation.validateSettings(settings);
        m_samples.validateSettings(settings);
        m_cpkratio.validateSettings(settings);
        m_timelimit.validateSettings(settings);
        m_syslimit.validateSettings(settings);
        m_bondlimit.validateSettings(settings);

    }
    
    /**
     * @see org.knime.core.node.NodeModel #loadInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void loadInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
        
		// TODO load internal data. 
		// Everything handed to output ports is loaded automatically (data
		// returned by the execute method, models loaded in loadModelContent,
		// and user settings set through loadSettingsFrom - is all taken care 
		// of). Load here only the other internals that need to be restored
		// (e.g. data used by the views).

    }
    
    /**
     * @see org.knime.core.node.NodeModel #saveInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void saveInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
       
       	// TODO save internal models. 
		// Everything written to output ports is saved automatically (data
		// returned by the execute method, models saved in the saveModelContent,
		// and user settings saved through saveSettingsTo - is all taken care 
		// of). Save here only the other internals that need to be preserved
		// (e.g. data used by the views).

    }

}
