package com.treweren.enumeration;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;

import org.knime.chem.types.SdfCell;
import org.knime.chem.types.SdfValue;
import org.knime.chem.types.SmilesCell;
import org.knime.chem.types.SmilesValue;
import org.knime.core.data.DataCell;
import org.knime.core.data.DataColumnSpec;
import org.knime.core.data.DataColumnSpecCreator;
import org.knime.core.data.DataRow;
import org.knime.core.data.DataTableSpec;
import org.knime.core.data.DataType;
import org.knime.core.data.RowKey;
import org.knime.core.data.def.DefaultRow;
import org.knime.core.data.def.DoubleCell;
import org.knime.core.data.def.IntCell;
import org.knime.core.data.def.StringCell;
import org.knime.core.node.BufferedDataContainer;
import org.knime.core.node.BufferedDataTable;
import org.knime.core.node.CanceledExecutionException;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelString;
import org.knime.core.node.port.PortType;
import org.knime.core.node.ExecutionContext;
import org.knime.core.node.ExecutionMonitor;
import org.knime.core.node.InvalidSettingsException;
import org.knime.core.node.NodeLogger;
import org.knime.core.node.NodeModel;
import org.knime.core.node.NodeSettingsRO;
import org.knime.core.node.NodeSettingsWO;
import org.knime.core.util.MutableBoolean;

import org.knime.core.node.BufferedDataTable;





/**
 * This is the model implementation of Enumeration.
 * Enumerate a combinatorial library
 *
 * @author Treweren Consultants
 */
public class EnumerationNodeModel extends NodeModel {
    
    // the logger instance
    private static final NodeLogger logger = NodeLogger
            .getLogger(EnumerationNodeModel.class);
    private static int[] n_Column =  {0,0,0,0,0,0,0,0,0};
    private static int n_Ligand = 0;
    private static String m_sdfColumn = null;
    private static String m_smilesColumn = null;
    private static String m_stringColumn = null;
    private static final int CANCEL_CHECK_INTERVAL = 1000;
    
    /** the settings key which is used to retrieve and 
        store the settings (from the dialog or from a settings file)    
       (package visibility to be usable from the dialog). */
	static final String CFGKEY_RGROUP = "R-group";
	static final String CFGKEY_FILTER = "Filter";
	static final String CFGKEY_SMILES = "Smiles";
	static final String CFGKEY_WORKING = "Working";
	static final String CFGKEY_ROW = "Row";
	static final String CFGKEY_GROUP1 = "Group 1";
	static final String CFGKEY_GROUP2 = "Group 2";
	static final String CFGKEY_GROUP3 = "Group 3";
	static final String CFGKEY_GROUP4 = "Group 4";
	
	/** initial default count value. */
	static final int DEFAULT_RGROUP = 2;
	static final boolean DEFAULT_FILTER = false;
	static final boolean DEFAULT_SMILES = false;
	static final String DEFAULT_WORKING = System.getenv("THINK_WORKING");
	static final int DEFAULT_ROW = 1;
	static final String DEFAULT_GROUP1 = "";
	static final String DEFAULT_GROUP2 = "";
	static final String DEFAULT_GROUP3 = "";
	static final String DEFAULT_GROUP4 = "";

	// example value: the models count variable filled from the dialog 
	// and used in the models execution method. The default components of the
	// dialog work with "SettingsModels".
	
	private final SettingsModelIntegerBounded m_rgroup =
		new SettingsModelIntegerBounded(EnumerationNodeModel.CFGKEY_RGROUP,
                    EnumerationNodeModel.DEFAULT_RGROUP,
                    1, 4);
	
	private final SettingsModelBoolean m_filter =
		new SettingsModelBoolean(EnumerationNodeModel.CFGKEY_FILTER,
                    EnumerationNodeModel.DEFAULT_FILTER );
	
	private final SettingsModelBoolean m_smiles =
		new SettingsModelBoolean(EnumerationNodeModel.CFGKEY_SMILES,
                    EnumerationNodeModel.DEFAULT_SMILES );

	private final SettingsModelString m_working =
		new SettingsModelString(EnumerationNodeModel.CFGKEY_WORKING,
                    EnumerationNodeModel.DEFAULT_WORKING);

	private final SettingsModelIntegerBounded m_row =
			new SettingsModelIntegerBounded(EnumerationNodeModel.CFGKEY_ROW,
	                    EnumerationNodeModel.DEFAULT_ROW,
	                    1, Integer.MAX_VALUE);
	
	private final SettingsModelString m_group1 =
			new SettingsModelString(EnumerationNodeModel.CFGKEY_GROUP1,
	                    EnumerationNodeModel.DEFAULT_GROUP1);
	private final SettingsModelString m_group2 =
			new SettingsModelString(EnumerationNodeModel.CFGKEY_GROUP2,
	                    EnumerationNodeModel.DEFAULT_GROUP2);
	private final SettingsModelString m_group3 =
			new SettingsModelString(EnumerationNodeModel.CFGKEY_GROUP3,
	                    EnumerationNodeModel.DEFAULT_GROUP3);
	private final SettingsModelString m_group4 =
			new SettingsModelString(EnumerationNodeModel.CFGKEY_GROUP4,
	                    EnumerationNodeModel.DEFAULT_GROUP4);

    /**
     * Constructor for the node model.
     */
	public static final PortType OPTIONAL_PORT_TYPE = new PortType(BufferedDataTable.class, true);
    protected EnumerationNodeModel() {
    
        // 5 incoming ports with 3-5 optional and one outgoing port
//        super(5, 1);
        super(createOPOs(7, 3, 4, 5, 6, 7), createOPOs(1));
    }
    private static PortType[] createOPOs(final int nrDataPorts, final int... optionalPortsIds)
    {
        PortType[] portTypes = new PortType[nrDataPorts];
        Arrays.fill(portTypes, BufferedDataTable.TYPE);        

        if (optionalPortsIds.length > 0) {
            for (int portId : optionalPortsIds) {
                if ((portId - 1) < nrDataPorts) {
                    portTypes[portId - 1] = OPTIONAL_PORT_TYPE;
                }
            }
        }
        return portTypes;
    } 
    /**
     * @see org.knime.core.node.NodeModel
     *      #configure(org.knime.core.data.DataTableSpec[])
     */

    @Override
    protected DataTableSpec[] configure(final DataTableSpec[] inSpecs) throws InvalidSettingsException {

        if ( System.getenv("THINK_EXEC") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_EXEC is unset - It should be set to the folder containing the THINK software.");       	  		
    	}
        if ( System.getenv("THINK_WORKING") == null ) {
            throw new InvalidSettingsException(
            "Environment variable THINK_WORKING is not set to the folder in which working files will be created.");       	  		
    	}
        for ( int i=0;i<=m_rgroup.getIntValue();i++)
        {
        	DataTableSpec in = inSpecs[i];
        	if ( in != null ) {
        		n_Column[i] = columnSelect (i,in);
        	}
        	else if ( i == 0 )
              throw new InvalidSettingsException(
                    "No reaction connected");       	
        	else
                throw new InvalidSettingsException(
                      "No R-group " + i + " connected");       	
       }
        // Optional sites table
        DataTableSpec inpdb = inSpecs[5];
        if ( inpdb != null ) {
            if ( inpdb.getNumColumns() < 3 ) {
                throw new InvalidSettingsException(
                "Site input table does not have 3 columns");       	           	
            }
        	if ( !inpdb.getColumnSpec(2).getName().equals("File")) { 
                throw new InvalidSettingsException(
                "Site input table does not have third column named 'File' ");       	
        	}
        }
        // Optional ligand query
    	DataTableSpec inLigand = inSpecs[6];
    	if ( inLigand != null ) {
    		n_Ligand = columnSelect (6,inLigand);
    		if ( inpdb == null )
                throw new InvalidSettingsException(
                "You must specify a protein site as well as a ligand ");       	
    	}

        return new DataTableSpec[] {null};
    }
    protected int columnSelect ( int rgroup, final DataTableSpec in ) throws  InvalidSettingsException {

        StringBuilder warningMessage = new StringBuilder();
        int n_column=0;
        int sdfColCount = 0;
        int n_sdf=0;
        int smilesColCount = 0;
        int n_smiles=0;
        int stringColCount = 0;
        int n_string=0;
        String rgroupText = "Core ";
        if ( rgroup > 0 )
        	rgroupText = "R-group ";
        
        for (int i = 0; i < in.getNumColumns(); i++) {
            DataColumnSpec s = in.getColumnSpec(i);
            if (s.getType().isCompatible(SdfValue.class)) {
                if (m_sdfColumn == null) {
                    m_sdfColumn = in.getColumnSpec(i).getName();
                    n_sdf = i;
                }
                sdfColCount++;
            }
            else if ( s.getType().isCompatible(SmilesValue.class)) {
            	if (m_smilesColumn == null ) {
             		m_smilesColumn = in.getColumnSpec(i).getName();
             		n_smiles=i;
              	}
               	smilesColCount++;
            }
            else if ( in.getColumnSpec(i).getName().equalsIgnoreCase("smiles"))  {
            	if (m_stringColumn == null ) {
             		m_stringColumn = in.getColumnSpec(i).getName();
             		n_string=i;
              	}
               	stringColCount++;
            }
        }
        if ( smilesColCount == 0 & sdfColCount == 0 & stringColCount == 0) {
            throw new InvalidSettingsException(
                    "No column for " + rgroupText + rgroup + " with Smiles or SD file compatible type");       	
        }
        else if ( ( m_smiles.getBooleanValue() | sdfColCount == 0 ) & smilesColCount > 0 ) {
        	m_sdfColumn = null;
        	m_stringColumn = null;
        	n_column = n_smiles;
        	if (smilesColCount > 1) {
                warningMessage.append("More than one Smiles compatible column for " + rgroupText + rgroup  
                        + ", using column \"" + m_smilesColumn + "\".");
        	}
        }
        else if ( sdfColCount > 0 ) {
           	m_smilesColumn = null;
        	m_stringColumn = null;
        	n_column = n_sdf;
        	if (sdfColCount > 1) {
                warningMessage.append("More than one SDF compatible column for " + rgroupText + rgroup 
                        + ", using column \"" + m_sdfColumn + "\".");
        	}
        }
        else if ( stringColCount > 0 ) {
        	m_smilesColumn = null;
        	m_sdfColumn = null;
        	n_column = n_string;
           	if (stringColCount > 1) {
                warningMessage.append("More than one Smiles string compatible column for R-group " + rgroup 
                        + ", using column \"" + m_stringColumn + "\".");
           	}
        }
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
        return n_column;
    }

    /**
     * @see org.knime.core.node.NodeModel #execute(BufferedDataTable[],
     *      ExecutionContext)
     */
    @Override
    protected BufferedDataTable[] execute(final BufferedDataTable[] inData,
            final ExecutionContext exec) throws Exception {

        notifyViews(null);

        final String cwdString =  m_working.getStringValue();
        // clean up 
        File outputReport = new File(cwdString, "output.log"); 
        outputReport.delete();
        File outputFile = new File(cwdString, "enumerate.csv"); 
        outputFile.delete();
        File SDFile = new File(cwdString, "enumerate.sdf"); 
        SDFile.delete();
        File progressFile = new File(cwdString, "progress0.dat"); 
        progressFile.delete();
     	String fileString[] = {"NONE", "NONE", "NONE", "NONE", "NONE", "NONE", "NONE", "NONE", "NONE"}; 

        // Check we have some valid inputs
        for ( int rgp=0; rgp<=m_rgroup.getIntValue();rgp++)
        {
   		    BufferedDataTable in = inData[rgp];
			DataTableSpec inSpec = in.getDataTableSpec();
      		if ( n_Column[rgp] == 0 ) {
         			n_Column[rgp] = columnSelect(rgp, inSpec);
        		}
/*       		int col = n_Column[rgp];
       		DataColumnSpec s = inSpec.getColumnSpec(col);
            if (s.getType().isCompatible(SdfValue.class)) {
                if (m_sdfColumn == null) {
                    m_sdfColumn = inSpec.getColumnSpec(col).getName();
                }
            }
            else if ( s.getType().isCompatible(SmilesValue.class)) {
            	if (m_smilesColumn == null ) {
             		m_smilesColumn = inSpec.getColumnSpec(col).getName();
              	}
            }
            else if ( s.getName().equals("SMILES"))  {
            	if (m_stringColumn == null ) {
             		m_stringColumn = inSpec.getColumnSpec(col).getName();
              	}
            }
  */
      		// write out input files
     		int col = n_Column[rgp];
            m_sdfColumn=null;
            m_smilesColumn=null;
            m_stringColumn=null;
     		if ( inSpec.getColumnSpec(col).getType().isCompatible(SdfValue.class)) {
            		fileString[rgp] = "rgroup" + rgp + ".sdf"; 
                    m_sdfColumn = inSpec.getColumnSpec(col).getName();
            	}
            	else if ( inSpec.getColumnSpec(col).getType().isCompatible(SmilesValue.class)) {
            		fileString[rgp] = "rgroup" + rgp + ".smiles"; 
             		m_smilesColumn = inSpec.getColumnSpec(col).getName();
            	}
            	else {
            		fileString[rgp] = "query" + rgp + ".smiles"; 
             		m_stringColumn = inSpec.getColumnSpec(col).getName();
            	}
            	File inFile = new File(cwdString, fileString[rgp]); 
            	BufferedWriter outWriter = new BufferedWriter(new FileWriter(inFile));
            	/*
            	if ( m_sdfColumn != null ) {
            		colIndex = in.getDataTableSpec().findColumnIndex(m_sdfColumn);
            	}
            	else if ( m_smilesColumn != null ) {
            		colIndex = in.getDataTableSpec().findColumnIndex(m_smilesColumn);        
            	}
            	else if ( m_stringColumn != null ) {
            		colIndex = in.getDataTableSpec().findColumnIndex(m_stringColumn);        
            	}
            	else {
            		throw new IllegalStateException("No suitable input column - run configure (again)");
            	}
        
            	final double count = in.getRowCount(); // floating point operations
            	if ( count > 1 & rgp == 0 ) {
            		setWarningMessage("Using only first row/molecule from reaction");
            	}
            	*/
            	int i = 0;
            	int colIndex = n_Column[rgp];
            	int missingCount = 0;
            	for (DataRow r : in) {
            		exec.checkCanceled();
            		DataCell c = r.getCell(colIndex);
            		if (c.isMissing()) {
            			missingCount++;
            		} 
            		else if ( m_sdfColumn != null ) {
            			SdfValue v = (SdfValue)c;
            			String toString = v.toString();
            			outWriter.write(toString);
            			if (!toString.trim().endsWith("$$$$")) {
            				outWriter.newLine();
            				outWriter.append("$$$$");
            				outWriter.newLine();
            			}
            		}
            		else if ( m_smilesColumn != null) {
            			SmilesValue v = (SmilesValue)c;
            			String toString = v.toString() + " " + r.getKey();
            			outWriter.write(toString);        	
            			outWriter.newLine();
            		}
            		else {
            			String toString = c.toString() + " " + r.getKey();
            			outWriter.write(toString);        	
            			outWriter.newLine();
            		}
            		i++;
            		if ( rgp == 0 ) break;
            	}

            	outWriter.close();
            	if (missingCount > 0) {
            		setWarningMessage("Skipped " + missingCount 
            				+ " row(s) because of missing values");
            	}
        }
        String pdbString=null;
        BufferedDataTable inpdb = inData[5];
        if ( inpdb != null ){
          final double count = inpdb.getRowCount(); // floating point operations
          if ( count > 1 ) {
              setWarningMessage("Using only using site " + m_row.getIntValue());
          }
          int rowCount=0;
          for (DataRow r : inpdb ) {
              exec.checkCanceled();
              rowCount++;
              DataCell c = r.getCell(2);
              if (!c.isMissing() && rowCount == m_row.getIntValue()){
              	pdbString = c.toString();
                break;
             }
           }
        }

        // output ligand (optional)
        String fragmentString ="null";
        if ( inData[6] != null ) {
        	
        	BufferedDataTable in = inData[6];
      		DataTableSpec inSpec = in.getDataTableSpec();
        	if ( n_Ligand == 0 ) {
         		n_Ligand = columnSelect(6, inSpec);
        	}
            m_sdfColumn = inSpec.getColumnSpec(n_Ligand).getName();
        	File inFile = new File(cwdString, "ligand.sdf");
        	fragmentString = "ligand.sdf";
        	BufferedWriter outWriter = new BufferedWriter(new FileWriter(inFile));
        	int colIndex = n_Ligand;
        	final double count = in.getRowCount(); // floating point operations
        	if ( count == 1 && m_row.getIntValue() > 1 ) {
        		setWarningMessage("Using first ligand");
        	}
        	int rowCount = 0; 
        	int missingCount = 0;
        	for (DataRow r : in) {
        		exec.checkCanceled();
        		rowCount++;
        		if ( rowCount == m_row.getIntValue() || count == 1 ) {    		
        			DataCell c = r.getCell(colIndex);
        			if (c.isMissing()) {
        				missingCount++;
        			} 
        			else if ( m_sdfColumn != null ) {
        				SdfValue v = (SdfValue)c;
        				String toString = v.toString();
        				outWriter.write(toString);
        				if (!toString.trim().endsWith("$$$$")) {
        					outWriter.newLine();
        					outWriter.append("$$$$");
        					outWriter.newLine();
        				}
        			}
        			break;
        		}
        	}
         	outWriter.close();
        	if (missingCount > 0) {
        		setWarningMessage("Skipped " + missingCount 
    				+ " row(s) because of missing values");
        	}
        }
        // execute THINK
        File cwdFile = new File (cwdString);
        int exitVal;
        try {
            exec.setProgress(0.,"Starting THINK");
            Runtime rt = Runtime.getRuntime();
            String s_filter = "NONE";
            if ( m_filter.getBooleanValue() ) {
            	s_filter = "FILTER";
            }
            String s_group1 = m_group1.getStringValue();
            String s_group2 = m_group2.getStringValue();
            String s_group3 = m_group3.getStringValue();
            String s_group4 = m_group4.getStringValue();
  
            // prepare the command string
            String cmdString = System.getenv("THINK_EXEC") + "think THINK_EXEC:enumerate.log output.log " + " " + fileString[0] + " " + fileString[1] + " " + fileString[2] + " " + fileString[3] + " " + fileString[4] + " " + fileString[5] + " " + fileString[6] + " " + fileString[7] + " " + fileString[8] + " " + s_filter + " \"" + pdbString + "\"  \"" + fragmentString + "\"  \""+ s_group1 + " \""+ " \"" + s_group2 + " \""+ " \"" + s_group3 + " \""+ " \"" + s_group4 + " \"" ;
            // Go!
            logger.info("THINK command line: '" + cmdString + "'");
            final Process proc = rt.exec(cmdString, null, cwdFile);
            final MutableBoolean procDone = new MutableBoolean(false);
            new Thread(new Runnable() {
                public void run() {
                    synchronized (procDone) {
                        while (!procDone.booleanValue()) {
                            try {
                                exec.checkCanceled();
                            } catch (CanceledExecutionException cee) {
                                // blow away the running external process
                                proc.destroy();
                                return;
                            }
                            try {
                               	procDone.wait(CANCEL_CHECK_INTERVAL);
                                // Read progress file and set progress value       
                               	double d= 0.;
                                File progressReport = new File(cwdString, "progress0.dat");
                                if ( progressReport.exists() && progressReport.isFile() ) {
                               		if ( progressReport.length() > 0 ) {
                               			BufferedReader in = new BufferedReader(new FileReader(progressReport));
                               			String line;
                               			if ( (line = in.readLine()) != null) {
                               			  d = Double.valueOf(line).doubleValue();
                               			}               
                              			if ( (line = in.readLine()) == null) {
                              				line = "Running THINK";
                              			}
                               			in.close();
                                        exec.setProgress( d, line.toString());
                               		}
                                }
                            } catch (InterruptedException e) {
                                // do nothing
                            } catch ( FileNotFoundException e) {
                                // do nothing
                            } catch ( IOException e) {
                                // do nothing
                            }
                        }
                    }
                }

            }).start();
           

            // wait until the external process finishes.
            exitVal = proc.waitFor();

            synchronized (procDone) {
                // this should terminate the check cancel thread
                procDone.setValue(true);
            }

            exec.checkCanceled();

            exec.setProgress("Wrapping up");
            logger.info("THINK terminated with exit code: " + exitVal);
        } catch (Throwable t) {
        	warningReport();
            logger.error("THINK failed (with exception)", t);
            throw new Exception(t);
        }

        if (exitVal != 0) {
            // before we return barfing, we save the output in the failing list
        	warningReport();
            throw new IllegalStateException("THINK failed (error code "
                    + exitVal + ")");
        }
         
        exec.setProgress("Creating KNIME tables");
        BufferedDataTable outTable = null;
        if ( pdbString == null )
           outTable = readCSVFile(exec);
        else
     	   outTable = readSDFile(exec);
        return new BufferedDataTable[]{outTable };

    }

    private BufferedDataTable readSDFile(final ExecutionContext exec) throws Exception {

    	String cwdString = m_working.getStringValue();
        String line;
        String fieldName;
        String doubleFields[] = new String[100];
        String stringFields[]= new String[100];
        int smileCount=0;
        int keyCount=0;
        int doubleCount=0;
        int stringCount=0;
        File outFile = new File (cwdString, "enumerate.sdf");

    	if (!(outFile.exists() && outFile.isFile())) {
    		warningReport();
    		throw new IllegalStateException("THINK didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
    	// Scan file for fields and data types
    	
        BufferedReader looksee = new BufferedReader(new FileReader(outFile));
        while ((line = looksee.readLine()) != null) {
            if (line.startsWith("$$$$")) 
            	break;
            else if (line.startsWith("> <")) {
            	int l = line.length() - 1;
            	fieldName = line.substring(3,l);
            	line = looksee.readLine();
            	if ( fieldName.equals("SMILES")) {
            		smileCount = 1;
            	}
            	else if ( fieldName.equals("KEYS")) {
            		keyCount = 1;
            	}
            	else {
            		try {
            			double d_value = Double.valueOf(line).doubleValue();
            			doubleFields[doubleCount] = fieldName;
            			doubleCount ++;
            		}
            		catch (Throwable t){
            			stringFields[stringCount] = fieldName;
            			stringCount++;
            		}
            	}
            }	
        }	        
        looksee.close();
        
        // Create the table
        DataColumnSpec[] allColSpecs;
        allColSpecs = new DataColumnSpec[stringCount+doubleCount+smileCount+keyCount+1];
        int count =0;
        int keyStart=0;
        int doubleStart=0;
        int stringStart=0;
        allColSpecs[0] = 
            new DataColumnSpecCreator("Molecule",SdfCell.TYPE).createSpec();
        if ( smileCount == 1 ) {
        	count ++;
            allColSpecs[count] = 
            	new DataColumnSpecCreator("SMILES",SmilesCell.TYPE).createSpec();
        }
        if ( keyCount == 1 ) {
        	keyCount = count + 1;
        	count ++;
            allColSpecs[count] = 
            	new DataColumnSpecCreator("KEYS",StringCell.TYPE).createSpec();
        }
        if ( stringCount > 0 ) {
        	stringStart=count + 1;
        	for (int i=0; i<stringCount; i++ ) {
        		count ++;
                allColSpecs[count] = 
        		new DataColumnSpecCreator(stringFields[i],StringCell.TYPE).createSpec();
        	}
        }
        if ( doubleCount > 0 ) {
           	doubleStart=count + 1;      	
        	for (int i=0; i<doubleCount; i++ ) {
        		count ++;
                allColSpecs[count] = 
        		new DataColumnSpecCreator(doubleFields[i],DoubleCell.TYPE).createSpec();
        	}
        }
        DataTableSpec outputSpec = new DataTableSpec(allColSpecs);
        BufferedDataContainer cont = exec.createDataContainer(outputSpec);
                
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        
        HashSet<String> titles = new HashSet<String>();

        StringBuilder buf = new StringBuilder(4096);
        boolean titleRead = false;
        String title = "";
    		
        DataCell[] cells = new DataCell[count+1];
        int molcount=0;
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();

            if (!titleRead) {
                title = line;
                if (title.length() == 0) {
                    title = "Mol " + molcount;
                }
                if (titles.contains(title)) {
                    title = title + "-" + molcount;
                }
                titles.add(title);
                titleRead = true;
            }
            buf.append(line).append('\n');

            if (line.startsWith("$$$$")) {
            	molcount ++;
                cells[0] = new SdfCell(buf.toString());
                RowKey key = new RowKey(title);

                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
                buf.delete(0, buf.length());
                titleRead = false;
            }
            else if ( line.startsWith("> <")) {
            	int l = line.length() - 1;
            	fieldName = line.substring(3,l);
            	line = in.readLine();
                buf.append(line).append('\n');
            	if ( smileCount == 1 & fieldName.equals("SMILES")) {
            		cells[1] = new SmilesCell(line);
            	}
            	else if ( keyCount == 1 & fieldName.equals("KEYS")) {
                    cells[keyStart]= new StringCell(line);
            	}
            	else {
            		for (int i=0; i<doubleCount; i++ ) {
            			if ( doubleFields[i].equals(fieldName)) {
            				try {
            				double d_value = Double.valueOf(line).doubleValue();
            				cells[doubleStart+i]= new DoubleCell(d_value);
            				} catch (Throwable t) {
            					cells[doubleStart+i] = DataType.getMissingCell();
            				}
            				break;
            			}
            		}
            		for (int i=0; i<stringCount; i++ ) {
            			if ( stringFields[i].equals(fieldName)) {
            				cells[stringStart+i]= new StringCell(line);
            				break;
            			}
            		}
            	}
            }
        }
        cont.close();
        in.close();
 
        return cont.getTable();
 
  
    }
  
    private BufferedDataTable readCSVFile(final ExecutionContext exec) throws Exception { 

        String cwdString =  m_working.getStringValue();
        File outFile = new File (cwdString, "enumerate.csv");
        if (!(outFile.exists() && outFile.isFile())) {
        	warningReport();
        	throw new IllegalStateException("THINK didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
        HashSet<String> titles = new HashSet<String>();
        String title;
        String line;
        boolean headerRead = false;
        boolean columnsCreated = false;
        DataColumnSpec[] allColSpecs;
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        DataTableSpec outputSpec;
        BufferedDataContainer cont =null;
        RowKey key =null;
        int count =0;
        int colcount =0;
        int rowcount =0;
        char chr='0';
        int keys = 0;
        int ilast = 0;
        String fieldNames[] = new String[100];
        int fieldTypes[] = new int[100];
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();
// Extract column headers
            if ( !headerRead ) {
                headerRead = true;
                colcount =0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) 
                        chr = line.charAt(i);
                    else
                        chr = ',';
                    if ( chr == ',' ) { 
                        fieldNames[colcount] = line.substring(ilast,i);
                        if ( line.substring(ilast,i).equalsIgnoreCase("SMILES"))
                        	fieldTypes[colcount]= 2;
                        else if ( line.substring(ilast,i).equalsIgnoreCase("NAME"))
                        	fieldTypes[colcount]= 1;
                        else if ( line.substring(ilast,i).equalsIgnoreCase("KEYS"))
                        	fieldTypes[colcount]= 3;
                        ilast = i+1;
                        colcount++;
                    }
                }
                colcount--;
            }
// Load data
            else {
                DataCell[] cells = new DataCell[colcount];
                rowcount++;
                count =0;                 
                double d_value;
                 for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) 
                        chr = line.charAt(i);
                    else
                        chr = ',';
                    if ( chr == ',' ) { 
                        if ( count == 0 ) {
                            title = line.substring(0,i);
                            if (title.length() == 0) 
                                title = "Mol " + rowcount;
                            if (titles.contains(title)) 
                                title = title + "-" + rowcount;
                            titles.add(title);
                            key = new RowKey(title);                           
                        }
                        else if  ( fieldTypes[count] == 2 ) 
                            cells[count-1] = new SmilesCell(line.substring(ilast,i));
                        else if  ( fieldTypes[count]== 3) 
                            cells[count-1] = new StringCell(line.substring(ilast,i));
                        else if ( count <= colcount ) {
                          try {
                            d_value = Double.valueOf(line.substring(ilast,i)).doubleValue();
                            cells[count-1] = new DoubleCell(d_value);
                            if ( fieldTypes[count] == 0 )
                            	fieldTypes[count] = 4;
                          } catch (Throwable t) {
                            if ( fieldTypes[count] == 0 ) {
                            	fieldTypes[count] = 3;
                                cells[count-1] = new StringCell(line.substring(ilast,i));
                            }
                            else
                                cells[count-1] = DataType.getMissingCell();
                          }
                        }
                        count ++;
                        ilast = i + 1;
                    }
                 }
  // Create columns?
                 if (!columnsCreated )
                 {
                     allColSpecs = new DataColumnSpec[colcount];
                     columnsCreated = true;
  // count= 0 will be molecule names
                     for (count=1; count<=colcount; count++)
                     {                   	 
                       if  ( fieldTypes[count] == 2 ) 
                         allColSpecs[count-1] = new DataColumnSpecCreator(fieldNames[count],SmilesCell.TYPE).createSpec();
                       else if  ( fieldTypes[count] == 4 ) 
                         allColSpecs[count-1] = new DataColumnSpecCreator(fieldNames[count], DoubleCell.TYPE).createSpec();
                       else if ( fieldTypes[count] == 3 )
                         allColSpecs[count-1] = new DataColumnSpecCreator(fieldNames[count],StringCell.TYPE).createSpec();
                       else
                    	 allColSpecs[count-1] = new DataColumnSpecCreator("Unknown",StringCell.TYPE).createSpec();
                     }
                     outputSpec = new DataTableSpec(allColSpecs);
                     cont = exec.createDataContainer(outputSpec);               	 
                 }
                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
            }
        }
        cont.close();
        in.close();

        return cont.getTable();
    }

    private BufferedDataTable readOutputFile(final ExecutionContext exec) throws Exception {

    	String cwdString = m_working.getStringValue();
        File outFile = new File (cwdString, "enumerate.csv");

    	if (!(outFile.exists() && outFile.isFile())) {
           	warningReport();
            throw new IllegalStateException("THINK didn't produce any output"
                    + " at the specified location ('"
                    + outFile.getAbsolutePath() + "')");
        }
    	
        HashSet<String> titles = new HashSet<String>();
        String title;
        String line;
        boolean headerRead = false;
        DataColumnSpec[] allColSpecs;
        BufferedReader in = new BufferedReader(new FileReader(outFile));
        DataTableSpec outputSpec;
        BufferedDataContainer cont =null;
        RowKey key =null;
        int count =0;
        int query =0;
        int rowcount =0;
        char chr='0';
        int nocols=1;
        while ((line = in.readLine()) != null) {
            exec.checkCanceled();
// Extract column headers
            if ( !headerRead ) {
                headerRead = true;
                count =0;
                for ( int i=0; i<line.length(); i++ ) {
                    if ( line.charAt(i) == ',' ) count ++;
                }
// Create column definitions (see Similarity this template allows for selected columns to be loaded)
                allColSpecs = new DataColumnSpec[nocols];
                count =0;
                int ilast=0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else {
                        chr = ',';
                    }
                    if ( chr == ',' ) {
                        if  ( count == 1 ) {
                            allColSpecs[0] = 
                            new DataColumnSpecCreator(line.substring(ilast,i),SmilesCell.TYPE).createSpec();
                        }
                        count ++;
                        ilast = i + 1;
                    }
                }
                outputSpec = new DataTableSpec(allColSpecs);
                cont = exec.createDataContainer(outputSpec);
            }
// Load data
            else {
                DataCell[] cells = new DataCell[nocols];
                rowcount++;
                count =0;                  
                int ilast=0;
                for ( int i=0; i<=line.length(); i++ ) {
                    if ( i < line.length() ) {
                        chr = line.charAt(i);
                    }
                    else
                        chr = ',';
                    if ( chr == ',' ) { 
                        if ( count == 0 ) {
                            title = line.substring(0,i);
                            if (title.length() == 0) {
                                title = "Mol " + rowcount;
                            }
                            if (titles.contains(title)) {
                                title = title + "-" + rowcount;
                            }
                            titles.add(title);
                            key = new RowKey(title);                           
                        }
                        else if  ( count == 1 ) {
                            cells[0] = new SmilesCell(line.substring(ilast,i));   	
                        }
                        count ++;
                        ilast = i + 1;
                    }
                }
                DefaultRow row = new DefaultRow(key, cells);
                cont.addRowToTable(row);
            }
        }
        cont.close();
        in.close();

        return cont.getTable();
    }

    private void warningReport( ) throws Exception {

    	StringBuilder warningMessage = new StringBuilder();
        String cwdString = m_working.getStringValue();
        File outputReport = new File(cwdString, "output.log"); 

        if ( outputReport.exists() && outputReport.isFile() ) {
       		if ( outputReport.length() > 0 ) {
       			BufferedReader in = new BufferedReader(new FileReader(outputReport));
       			String line;
       			while ((line = in.readLine()) != null) {
       				warningMessage.append( line + "\n");
       			}               
       			in.close();
       		}
       		else { 
                warningMessage.append( "Report from THINK is blank");      	     			
       		}
        }
       	else {
            warningMessage.append( "No report from THINK");      	
       	}	
        if (warningMessage.length() > 0) {
            setWarningMessage(warningMessage.toString());
        }
   }

    /**
     * @see org.knime.core.node.NodeModel#reset()
     */
    @Override
    protected void reset() {
        // TODO Code executed on reset.
        // Models build during execute are cleared here.
        // Also data handled in load/saveInternals will be erased here.
    }

 
    /**
     * @see org.knime.core.node.NodeModel
     *      #saveSettingsTo(org.knime.core.node.NodeSettings)
     */
    @Override
    protected void saveSettingsTo(final NodeSettingsWO settings) {

        // TODO save user settings to the config object.
		
		m_rgroup.saveSettingsTo(settings);
		m_filter.saveSettingsTo(settings);
		m_smiles.saveSettingsTo(settings);
		m_working.saveSettingsTo(settings);
		m_row.saveSettingsTo(settings);
		m_group1.saveSettingsTo(settings);
		m_group2.saveSettingsTo(settings);
		m_group3.saveSettingsTo(settings);
		m_group4.saveSettingsTo(settings);
    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #loadValidatedSettingsFrom(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void loadValidatedSettingsFrom(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO load (valid) settings from the config object.
        // It can be safely assumed that the settings are valided by the 
        // method below.
        
        m_rgroup.loadSettingsFrom(settings);
        m_filter.loadSettingsFrom(settings);
        m_smiles.loadSettingsFrom(settings);
        m_working.loadSettingsFrom(settings);
        m_row.loadSettingsFrom(settings);
        m_group1.loadSettingsFrom(settings);
        m_group2.loadSettingsFrom(settings);
        m_group3.loadSettingsFrom(settings);
        m_group4.loadSettingsFrom(settings);
    }

    /**
     * @see org.knime.core.node.NodeModel
     *      #validateSettings(org.knime.core.node.NodeSettingsRO)
     */
    @Override
    protected void validateSettings(final NodeSettingsRO settings)
            throws InvalidSettingsException {
            
        // TODO check if the settings could be applied to our model
        // e.g. if the count is in a certain range (which is ensured by the
        // SettingsModel).
        // Do not actually set any values of any member variables.

        m_rgroup.validateSettings(settings);
        m_filter.validateSettings(settings);
        m_smiles.validateSettings(settings);
        m_working.validateSettings(settings);
        m_row.validateSettings(settings);
        m_group1.validateSettings(settings);
        m_group2.validateSettings(settings);
        m_group3.validateSettings(settings);
        m_group4.validateSettings(settings);
   }
    
    /**
     * @see org.knime.core.node.NodeModel #loadInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void loadInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
        
		// TODO load internal data. 
		// Everything handed to output ports is loaded automatically (data
		// returned by the execute method, models loaded in loadModelContent,
		// and user settings set through loadSettingsFrom - is all taken care 
		// of). Load here only the other internals that need to be restored
		// (e.g. data used by the views).

    }
    
    /**
     * @see org.knime.core.node.NodeModel #saveInternals(java.io.File,
     *      org.knime.core.node.ExecutionMonitor)
     */
    @Override
    protected void saveInternals(final File internDir,
            final ExecutionMonitor exec) throws IOException,
            CanceledExecutionException {
       
       	// TODO save internal models. 
		// Everything written to output ports is saved automatically (data
		// returned by the execute method, models saved in the saveModelContent,
		// and user settings saved through saveSettingsTo - is all taken care 
		// of). Save here only the other internals that need to be preserved
		// (e.g. data used by the views).

    }

}
