package com.treweren.docking;

import java.awt.FlowLayout;
import javax.swing.JFileChooser;

import org.knime.core.node.defaultnodesettings.DefaultNodeSettingsPane;
import org.knime.core.node.defaultnodesettings.DialogComponentBoolean;
import org.knime.core.node.defaultnodesettings.DialogComponentFileChooser;
import org.knime.core.node.defaultnodesettings.DialogComponentNumber;
import org.knime.core.node.defaultnodesettings.DialogComponentString;
import org.knime.core.node.defaultnodesettings.DialogComponentStringSelection;
import org.knime.core.node.defaultnodesettings.SettingsModelBoolean;
import org.knime.core.node.defaultnodesettings.SettingsModelDoubleBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelIntegerBounded;
import org.knime.core.node.defaultnodesettings.SettingsModelString;






/**
 * <code>NodeDialog</code> for the "Docking" Node.
 * Docks molecules into binding site
 *
 * This node dialog derives from {@link DefaultNodeSettingsPane} which allows
 * creation of a simple dialog with standard components. If you need a more 
 * complex dialog please derive directly from 
 * {@link org.knime.core.node.NodeDialogPane}.
 * 
 * @author Treweren Consultants
 */
public class DockingNodeDialog extends DefaultNodeSettingsPane {

    /**
     * New pane for configuring Docking node dialog.
     * This is just a suggestion to demonstrate possible default dialog
     * components.
     */
    protected DockingNodeDialog() {
        super();
        
        DialogComponentFileChooser c_search = new DialogComponentFileChooser(
                new SettingsModelString (
                        DockingNodeModel.CFGKEY_SEARCH,
                        DockingNodeModel.DEFAULT_SEARCH ),
                        "Search file", JFileChooser.OPEN_DIALOG, "smi", "csv", "smiles","sdf" );
        addDialogComponent(c_search );
        c_search.setToolTipText("The molecules being searched are in this file");
        c_search.setBorderTitle("Search molecules in file");
        
        DialogComponentBoolean c_filter = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        DockingNodeModel.CFGKEY_FILTER,
                        DockingNodeModel.DEFAULT_FILTER),
                        "Apply drug-like filters" );
        addDialogComponent(c_filter);
        c_filter.setToolTipText("Eliminates molecules which have properties which are not drug-like");
        c_filter.getComponentPanel().setLayout( new FlowLayout(0));
        
        final DialogComponentBoolean c_parent = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        DockingNodeModel.CFGKEY_PARENT,
                        DockingNodeModel.DEFAULT_PARENT),
                        "Convert salts to parent form" );
        addDialogComponent(c_parent);
        c_parent.setToolTipText("Converts molecules to parent(neutral) form and eliminates counter-ions(salts)");
        c_parent.getComponentPanel().setLayout( new FlowLayout(0));

        DialogComponentNumber c_warp=new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                        DockingNodeModel.CFGKEY_SPEED,
                        DockingNodeModel.DEFAULT_SPEED, 1., 100. ),
                        "Warp speed", 5., 4 ) ;
        addDialogComponent(c_warp);
        c_warp.setToolTipText("Higher warp speed reduces search time by skipping some pharmacophore (when the product of centres weights * warp speed > 100 )");
        c_warp.getComponentPanel().setLayout( new FlowLayout(2));
 
        DialogComponentNumber c_tolerance = new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                        DockingNodeModel.CFGKEY_TOLERANCE,
                        DockingNodeModel.DEFAULT_TOLERANCE, 0., 2. ),
                        "Distance tolerance", .05, 5 );
        
        addDialogComponent(c_tolerance);
        c_tolerance.setToolTipText("Set the maximum separation of the interaction centre from the matched atom");
        c_tolerance.getComponentPanel().setLayout( new FlowLayout(2));
 
        DialogComponentNumber c_centres = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                    DockingNodeModel.CFGKEY_CENTRES,
                    DockingNodeModel.DEFAULT_CENTRES,
                    2, 4),
                    "Number of pharmacophore centres", /*step*/ 1, /*componentwidth*/ 1);
        addDialogComponent(c_centres);
        c_centres.setToolTipText("The number of interaction centres in the pharmacophore");
        c_centres.getComponentPanel().setLayout( new FlowLayout(2));
        DialogComponentStringSelection c_refine = new DialogComponentStringSelection(
                new SettingsModelString(
                        DockingNodeModel.CFGKEY_REFINE,
                        DockingNodeModel.DEFAULT_REFINE),
                        "Refinement", "None", "Ligand", "Side-chain and Ligand");
        addDialogComponent(c_refine);
        c_refine.setToolTipText("Control which (if any) torsions are refined");
        c_refine.getComponentPanel().setLayout( new FlowLayout(2));
/*
        DialogComponentBoolean c_sidechains = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        DockingNodeModel.CFGKEY_SIDECHAINS,
                        DockingNodeModel.DEFAULT_SIDECHAINS),
                        "Relax side-chains positions for docking" );
        addDialogComponent(c_sidechains);
        c_sidechains.setToolTipText("Moving side-chains can allow more molecules to dock but takes longer");
        c_sidechains.getComponentPanel().setLayout( new FlowLayout(0));
  */      

        DialogComponentBoolean c_reuse = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        DockingNodeModel.CFGKEY_REUSE,
                        DockingNodeModel.DEFAULT_REUSE),
                        "Permit more than one interaction per residue" );
        addDialogComponent(c_reuse);
        c_reuse.setToolTipText("Better inhibitors often interact with different residues");
        c_reuse.getComponentPanel().setLayout( new FlowLayout(0));

        createNewTab("Bond rotations");
        
         DialogComponentNumber c_single = new DialogComponentNumber(
                 new SettingsModelIntegerBounded(
                         DockingNodeModel.CFGKEY_SINGLE,
                         DockingNodeModel.DEFAULT_SINGLE,
                         0, Integer.MAX_VALUE),
                         "Single bond increments", /*step*/ 1, /*componentwidth*/ 2);
         addDialogComponent(c_single);
         c_single.setToolTipText("Single bonds are incremented by 360/n degrees where n is the number of steps (0=no rotation)");
         c_single.getComponentPanel().setLayout( new FlowLayout(2));
         
         DialogComponentNumber c_conjugated = new DialogComponentNumber(
                 new SettingsModelIntegerBounded(
                         DockingNodeModel.CFGKEY_CONJUGATED,
                         DockingNodeModel.DEFAULT_CONJUGATED,
                         0, Integer.MAX_VALUE),
                         "Conjugated bond increments", /*step*/ 1, /*componentwidth*/ 2);
         addDialogComponent(c_conjugated);
         c_conjugated.setToolTipText("Conjugated (csp2-csp2) bonds are incremented by 360/n degrees where n is the number of steps");
         c_conjugated.getComponentPanel().setLayout( new FlowLayout(2));

         DialogComponentNumber c_crowded = new DialogComponentNumber(
                 new SettingsModelIntegerBounded(
                         DockingNodeModel.CFGKEY_CROWDED,
                         DockingNodeModel.DEFAULT_CROWDED,
                         0, Integer.MAX_VALUE),
                         "Crowded bond increments", /*step*/ 1, /*componentwidth*/ 2);
         addDialogComponent(c_crowded);
         c_crowded.setToolTipText("Crowded (csp2-csp2) bonds are incremented by 360/n degrees where n is the number of steps");
         c_crowded.getComponentPanel().setLayout( new FlowLayout(2));
  
         DialogComponentNumber c_alpha = new DialogComponentNumber(
                 new SettingsModelIntegerBounded(
                     DockingNodeModel.CFGKEY_ALPHA,
                     DockingNodeModel.DEFAULT_ALPHA,
                     0, Integer.MAX_VALUE),
                     "Alpha bond increments", /*step*/ 1, /*componentwidth*/ 2);
         addDialogComponent(c_alpha);
         c_alpha.setToolTipText("Alpha (csp2-csp3) bonds are incremented by 360/n degrees where n is the number of steps");
         c_alpha.getComponentPanel().setLayout( new FlowLayout(2));

         DialogComponentNumber c_amide = new DialogComponentNumber(
                 new SettingsModelIntegerBounded(
                         DockingNodeModel.CFGKEY_AMIDE,
                         DockingNodeModel.DEFAULT_AMIDE,
                         0, Integer.MAX_VALUE),
                         "Amide bond increments", /*step*/ 1, /*componentwidth*/ 2);
         
         addDialogComponent(c_amide);
         c_amide.setToolTipText("Amide bonds are incremented by 360/n degrees where n is the number of steps");
         c_amide.getComponentPanel().setLayout( new FlowLayout(2));

         DialogComponentNumber c_ring = new DialogComponentNumber(
                 new SettingsModelIntegerBounded(
                         DockingNodeModel.CFGKEY_RING,
                         DockingNodeModel.DEFAULT_RING,
                         0, Integer.MAX_VALUE),
                         "Ring bond increments", /*step*/ 1, /*componentwidth*/ 2); 
         addDialogComponent(c_ring);
         c_ring.setToolTipText("Ring bonds are incremented by 360/n degrees where n is the number of steps");
         c_ring.getComponentPanel().setLayout( new FlowLayout(2));

        createNewTab("Advanced");
        DialogComponentStringSelection c_generation = new DialogComponentStringSelection(
                new SettingsModelString(
                        DockingNodeModel.CFGKEY_GENERATION,
                        DockingNodeModel.DEFAULT_GENERATION),
                        "Conformer Generation Mode", "Systematic", "Sample", "Random", "None");
        addDialogComponent(c_generation);
        c_generation.setToolTipText("For the most flexible molecules sample or random modes take less time.  Choose None to use the initial/input conformer");
        c_generation.getComponentPanel().setLayout( new FlowLayout(2));

        DialogComponentNumber c_samples = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        DockingNodeModel.CFGKEY_SAMPLES,
                        DockingNodeModel.DEFAULT_SAMPLES,
                        1, Integer.MAX_VALUE),
                        "Number of sample/random conformations", /*step*/ 100, /*componentwidth*/ 4);
        addDialogComponent(c_samples);
        c_samples.setToolTipText("The number of conformations should be sufficiently large to be representative");
        c_samples.getComponentPanel().setLayout( new FlowLayout(2));
       
        DialogComponentNumber c_timelimit = new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                        DockingNodeModel.CFGKEY_TIMELIMIT,
                        DockingNodeModel.DEFAULT_TIMELIMIT, 0., 1440. ),
                        "Molecule time limit (m)", 15., 4 );
        addDialogComponent(c_timelimit );
        c_timelimit.setToolTipText("In order to progress other molecules, the search for a molecule which takes longer than the time limit is abandoned");
        c_timelimit.getComponentPanel().setLayout( new FlowLayout(2));

        DialogComponentNumber c_fraction = new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                        DockingNodeModel.CFGKEY_FRACTION,
                        DockingNodeModel.DEFAULT_FRACTION, 0., 1. ),
                        "Minimum fraction of molecule within pharmacophore volume/area", .1, 3 );
        addDialogComponent(c_fraction );
        c_fraction.setToolTipText("Solutions with few atoms than the minimum within the volume (4-centre) or area (3-centre) are skipped");
        c_fraction.getComponentPanel().setLayout( new FlowLayout(2));

        DialogComponentNumber c_cpkratio = new DialogComponentNumber(
                new SettingsModelDoubleBounded (
                        DockingNodeModel.CFGKEY_CPKRATIO,
                        DockingNodeModel.DEFAULT_CPKRATIO, 0., 5. ),
                        "Contact to VdW radius ratio", .1, 3 );
        
        addDialogComponent(c_cpkratio);
        c_cpkratio.setToolTipText("Conformers with separation distances between distances less than the sum of the contact radii are skipped");
        c_cpkratio.getComponentPanel().setLayout( new FlowLayout(2));
  
        DialogComponentBoolean c_radius = new DialogComponentBoolean(
                new SettingsModelBoolean (
                    DockingNodeModel.CFGKEY_RADIUS,
                    DockingNodeModel.DEFAULT_RADIUS),
                    "Use atom radius (from query file) as tolerance" );
        addDialogComponent(c_radius);
        c_radius.setToolTipText("When separate tolerances for each query centre are required this can be achieved by editing the query file");
        c_radius.getComponentPanel().setLayout( new FlowLayout(0));
 
        DialogComponentBoolean c_conformers = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        DockingNodeModel.CFGKEY_CONFORMERS,
                        DockingNodeModel.DEFAULT_CONFORMERS),
                        "Retain all conformers (not best)" );
        addDialogComponent(c_conformers);
        c_conformers.setToolTipText("Many queries have more than one conformation which match");
        c_conformers.getComponentPanel().setLayout( new FlowLayout(0));
     
        createNewGroup("Systematic limits");

        DialogComponentNumber c_syslimit = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        DockingNodeModel.CFGKEY_SYSLIMIT,
                        DockingNodeModel.DEFAULT_SYSLIMIT,
                        0, Integer.MAX_VALUE),
                        "Number of conformations", /*step*/ 100, /*componentwidth*/ 4);
        addDialogComponent(c_syslimit);
        c_syslimit.setToolTipText("If the estimated number of conformers to be generated exceeds the systematic limit, sampling is used instead");
        c_syslimit.getComponentPanel().setLayout( new FlowLayout(2));
        
        DialogComponentNumber c_bondlimit = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        DockingNodeModel.CFGKEY_BONDLIMIT,
                        DockingNodeModel.DEFAULT_BONDLIMIT,
                        0, Integer.MAX_VALUE),
                        "Number of rotating bonds", /*step*/ 1, /*componentwidth*/ 2);
        addDialogComponent(c_bondlimit);
        c_bondlimit.setToolTipText("If the estimated number of rotatable bonds exceeds the systematic limit, sampling is used instead");
        c_bondlimit.getComponentPanel().setLayout( new FlowLayout(2));

        closeCurrentGroup();
   
        createNewTab("THINK setup");         

        DialogComponentBoolean c_column = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        DockingNodeModel.CFGKEY_SMILES,
                        DockingNodeModel.DEFAULT_SMILES),
                        "Use SMILES in preference to SD column" );
        addDialogComponent(c_column);
        c_column.getComponentPanel().setLayout( new FlowLayout(0));
        c_column.setToolTipText("The molecule data is transferred to THINK using SD or SMILES format depending on this setting");

       DialogComponentNumber c_row = new DialogComponentNumber(
                new SettingsModelIntegerBounded (
                        DockingNodeModel.CFGKEY_ROW,
                        DockingNodeModel.DEFAULT_ROW, 0, Integer.MAX_VALUE),
                        "Site row (0=all)",1, 2 );
        
        addDialogComponent(c_row);
        c_row.getComponentPanel().setLayout( new FlowLayout(0));
        c_row.setToolTipText("The data transferred may be limited to one site rather than all sites using this control");
  
        DialogComponentBoolean c_restart = new DialogComponentBoolean(
                new SettingsModelBoolean (
                        DockingNodeModel.CFGKEY_RESTART,
                        DockingNodeModel.DEFAULT_RESTART),
                        "Restart search (instead of continuing previous search)" );
        addDialogComponent(c_restart);
        c_restart.setToolTipText("Check point files (searchn.dat) are deleted before THINK restarts");
        c_restart.getComponentPanel().setLayout( new FlowLayout(0));

        DialogComponentNumber c_process = new DialogComponentNumber(
                new SettingsModelIntegerBounded(
                        DockingNodeModel.CFGKEY_PROCESS,
                        DockingNodeModel.DEFAULT_PROCESS,
                        1, 36 ),
                        "Maximum number of THINK processes", /*step*/ 1, /*componentwidth*/ 2);
        
        addDialogComponent(c_process);
        c_process.getComponentPanel().setLayout( new FlowLayout(2));
        c_process.setToolTipText("The search is divided between the THINK processes");
 
       DialogComponentString c_working = new DialogComponentString(
                new SettingsModelString (
                        DockingNodeModel.CFGKEY_WORKING,
                        DockingNodeModel.DEFAULT_WORKING),
                        "Working folder" );
        addDialogComponent(c_working);
        c_working.getComponentPanel().setLayout( new FlowLayout(2));
        c_working.setToolTipText("The temporary files used to transfer the data to and from THINK are placed in this folder");


    }
}
